from dataclasses import dataclass
from typing import Optional

from defs import EXPERIMENT_SEEDS, BaseConfigArgs
from lib_llm.models import ModelConfig, get_tokenizer_type
from lib_project.experiment import ExperimentHandle
from utils.memorization import get_memorization_training_config

from .experiment import (
    EXP_ABBREVIATION,
    ExperimentConfig,
    RandomStringConfig,
    RepetitionConflictConfig,
    irc_experiment,
)


@dataclass
class ConfigArgs(BaseConfigArgs):
    alphabet_size: int
    string_length: int
    num_clean_repetitions: int
    num_conflicting_tokens: int
    conflicts_first: bool = True
    num_epochs: int = 50


NUM_CLEAN_REPETITIOANONYMOUS = {
    (
        f"{model_id}_cr-{num_clean_repetitions}_ct-{num_conflicting_tokens}"
        + ("_cf" if conflicts_first else "")
    ): ConfigArgs(
        model_id=model_id,
        alphabet_size=26,
        # Length of the repeated string
        string_length=128,
        num_clean_repetitions=num_clean_repetitions,
        num_conflicting_tokens=num_conflicting_tokens,
    )
    for model_id in [
        "pyt-1b",
    ]
    for num_clean_repetitions in [0, 1, 3, 5, 7]
    for num_conflicting_tokens in [16, 32]
    for conflicts_first in [True, False]
}


CONFIG_ARGS = {
    "test": ConfigArgs(
        model_id="pyt-70m",
        string_length=32,
        alphabet_size=7,
        num_clean_repetitions=1,
        num_conflicting_tokens=8,
        conflicts_first=False,
    ),
} | NUM_CLEAN_REPETITIOANONYMOUS


def create_config(
    eval_type: str,
    seed_id: Optional[int] = None,
) -> ExperimentConfig:
    args = CONFIG_ARGS[eval_type]
    model_config = ModelConfig(
        model_id=args.model_name,
        base_dir=args.model_dir,
        pretrained=True,
    )
    config = ExperimentConfig(
        name=eval_type,
        seed_id=-1,
        seed=-1,
        data=RandomStringConfig(
            seed_id=-1,
            alphabet="latin",
            alphabet_size=args.alphabet_size,
            num_tokens=args.string_length,
            tokenizer_type=get_tokenizer_type(args.model_name),
        ),
        repetition_conflicts=RepetitionConflictConfig(
            seed=-1,
            num_clean_repetitions=args.num_clean_repetitions,
            num_conflicting_tokens=args.num_conflicting_tokens,
            conflicts_first=args.conflicts_first,
        ),
        model=model_config,
        training=get_memorization_training_config(
            seed_id=-1,
            model_id=model_config.model_id_not_none,
            num_epochs=args.num_epochs,
            wandb_project_name=f"llm_mem_{EXP_ABBREVIATION}",
        ),
    )
    if eval_type == "test":
        config.training.args.num_train_epochs = 1
    if seed_id is not None:
        set_seeds(config, seed_id)
    return config


def set_seeds(
    config: ExperimentConfig,
    seed_id: int,
) -> ExperimentConfig:
    seed = EXPERIMENT_SEEDS[seed_id]
    config.seed_id = seed_id
    config.seed
    config.data.seed_id = seed_id
    config.repetition_conflicts.seed = seed
    config.training.seed = seed
    return config


def get_configs() -> list[ExperimentConfig]:
    configs = []
    for eval_type in CONFIG_ARGS.keys():
        config = create_config(eval_type)
        configs.append(config)
    return configs


IMCHandle = ExperimentHandle(
    id=EXP_ABBREVIATION,
    create_configs=get_configs,
    set_seed=set_seeds,
    experiment=irc_experiment,
)
