from dataclasses import dataclass
from typing import Optional

from defs import EXPERIMENT_SEEDS, BaseConfigArgs
from lib_llm.models import ModelConfig, get_tokenizer_type
from lib_project.experiment import ExperimentHandle
from utils.memorization import get_memorization_training_config

from .experiment import (
    EXP_ABBREVIATION,
    ExperimentConfig,
    RandomStringConfig,
    irm_experiment,
)


@dataclass
class ConfigArgs(BaseConfigArgs):
    alphabet_size: int
    string_length: int
    entropy_target: int | None = None
    num_epochs: int = 100


ALPHABET_SIZES = {
    (f"{model_id}_a-{alphabet_size}"): ConfigArgs(
        model_id=model_id,
        alphabet_size=alphabet_size,
        string_length=1024,
    )
    for model_id in [
        "pyt-1b",
    ]
    for alphabet_size in [2, 4, 7, 13, 26]
}
ENTROPIES = {
    (f"{model_id}_h-{entropy_target}"): ConfigArgs(
        model_id=model_id,
        alphabet_size=26,
        entropy_target=entropy_target,
        string_length=1024,
    )
    for model_id in [
        "pyt-1b",
    ]
    for entropy_target in [2, 4, 7, 13]
}


CONFIG_ARGS = (
    {
        "test": ConfigArgs(
            model_id="pyt-70m",
            string_length=32,
            alphabet_size=7,
        ),
    }
    | ALPHABET_SIZES
    | ENTROPIES
)


def create_config(
    eval_type: str,
    seed_id: Optional[int] = None,
) -> ExperimentConfig:
    args = CONFIG_ARGS[eval_type]
    model_config = ModelConfig(
        model_id=args.model_name,
        base_dir=args.model_dir,
        pretrained=True,
    )
    config = ExperimentConfig(
        name=eval_type,
        seed_id=-1,
        seed=-1,
        data=RandomStringConfig(
            seed_id=-1,
            alphabet="latin",
            alphabet_size=args.alphabet_size,
            num_tokens=args.string_length,
            entropy_like=args.entropy_target,
            tokenizer_type=get_tokenizer_type(args.model_name),
        ),
        model=model_config,
        training=get_memorization_training_config(
            seed_id=-1,
            model_id=model_config.model_id_not_none,
            num_epochs=args.num_epochs,
            wandb_project_name=f"llm_mem_{EXP_ABBREVIATION}",
        ),
    )
    if eval_type == "test":
        config.training.args.num_train_epochs = 1
    if seed_id is not None:
        set_seeds(config, seed_id)
    return config


def set_seeds(
    config: ExperimentConfig,
    seed_id: int,
) -> ExperimentConfig:
    seed = EXPERIMENT_SEEDS[seed_id]
    config.seed_id = seed_id
    config.seed
    config.data.seed_id = seed_id
    config.training.seed = seed
    return config


def get_configs() -> list[ExperimentConfig]:
    configs = []
    for eval_type in CONFIG_ARGS.keys():
        config = create_config(eval_type)
        configs.append(config)
    return configs


IMRHandle = ExperimentHandle(
    id=EXP_ABBREVIATION,
    create_configs=get_configs,
    set_seed=set_seeds,
    experiment=irm_experiment,
)
