# coding=utf-8
# Copyright 2023 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Fine-tuning script for Stable Diffusion for text2image with support for LoRA."""

import argparse
import hashlib
import logging
import math
import os
import warnings
import random
import json
from PIL import Image
import shutil
from pathlib import Path
from typing import Optional, Union
import datasets
import numpy as np
import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch.utils.data import Dataset
import transformers
from accelerate import Accelerator
from accelerate.logging import get_logger
from accelerate.utils import ProjectConfiguration, set_seed, DistributedDataParallelKwargs
from datasets import load_dataset
from huggingface_hub import create_repo, upload_folder
from packaging import version
#from peft import LoraConfig
#from peft.utils import get_peft_model_state_dict
from torchvision import transforms
import torchvision.transforms.functional as TF
from tqdm.auto import tqdm
#from transformers import CLIPTextModel, CLIPTokenizer
from transformers import CLIPTokenizer

import diffusers
from diffusers import AutoencoderKL, DDPMScheduler, DiffusionPipeline, StableDiffusionPipeline, UNet2DConditionModel
#from diffusers.loaders import LoraLoaderMixin
from diffusers.optimization import get_scheduler
from diffusers.training_utils import compute_snr
from diffusers.utils import check_min_version, is_wandb_available
from diffusers.utils.import_utils import is_xformers_available

# modify
import sys 
sys.path.append("..") 
from lora_files_patchall.loaders.lora import LoraLoaderMixin
from lora_files_patchall.peft_files.tuners.lora.config import LoraConfig
from lora_files_patchall.peft_files.utils import get_peft_model_state_dict
from lora_files_patchall.text_files.models.clip.modeling_clip import CLIPTextModel
from lora_files_patchall.models.unet_2d_condition import UNet2DConditionModel


# Will error if the minimal version of diffusers is not installed. Remove at your own risks.
check_min_version("0.24.0")

logger = get_logger(__name__, log_level="INFO")


def save_model_card(repo_id: str, images=None, base_model=str, dataset_name=str, repo_folder=None):
    img_str = ""
    for i, image in enumerate(images):
        image.save(os.path.join(repo_folder, f"image_{i}.png"))
        img_str += f"![img_{i}](./image_{i}.png)\n"

    yaml = f"""
---
license: creativeml-openrail-m
base_model: {base_model}
tags:
- stable-diffusion
- stable-diffusion-diffusers
- text-to-image
- diffusers
- lora
inference: true
---
    """
    model_card = f"""
# LoRA text2image fine-tuning - {repo_id}
These are LoRA adaption weights for {base_model}. The weights were fine-tuned on the {dataset_name} dataset. You can find some example images in the following. \n
{img_str}
"""
    with open(os.path.join(repo_folder, "README.md"), "w") as f:
        f.write(yaml + model_card)


def parse_args():
    parser = argparse.ArgumentParser(description="Simple example of a training script.")
    parser.add_argument(
        "--pretrained_model_name_or_path",
        type=str,
        default=None,
        required=True,
        help="Path to pretrained model or model identifier from huggingface.co/models.",
    )
    parser.add_argument(
        "--revision",
        type=str,
        default=None,
        required=False,
        help="Revision of pretrained model identifier from huggingface.co/models.",
    )
    parser.add_argument(
        "--variant",
        type=str,
        default=None,
        help="Variant of the model files of the pretrained model identifier from huggingface.co/models, 'e.g.' fp16",
    )
    parser.add_argument(
        "--dataset_name",
        type=str,
        default=None,
        help=(
            "The name of the Dataset (from the HuggingFace hub) to train on (could be your own, possibly private,"
            " dataset). It can also be a path pointing to a local copy of a dataset in your filesystem,"
            " or to a folder containing files that 🤗 Datasets can understand."
        ),
    )
    parser.add_argument(
        "--dataset_config_name",
        type=str,
        default=None,
        help="The config of the Dataset, leave as None if there's only one config.",
    )
    parser.add_argument(
        "--train_data_dir",
        type=str,
        default=None,
        help=(
            "A folder containing the training data. Folder contents must follow the structure described in"
            " https://huggingface.co/docs/datasets/image_dataset#imagefolder. In particular, a `metadata.jsonl` file"
            " must exist to provide the captions for the images. Ignored if `dataset_name` is specified."
        ),
    )
    parser.add_argument(
        "--image_column", type=str, default="image", help="The column of the dataset containing an image."
    )
    parser.add_argument(
        "--caption_column",
        type=str,
        default="text",
        help="The column of the dataset containing a caption or a list of captions.",
    )
    parser.add_argument(
        "--validation_prompt", type=str, default=None, help="A prompt that is sampled during training for inference."
    )
    parser.add_argument(
        "--num_validation_images",
        type=int,
        default=4,
        help="Number of images that should be generated during validation with `validation_prompt`.",
    )
    parser.add_argument(
        "--validation_epochs",
        type=int,
        default=1,
        help=(
            "Run fine-tuning validation every X epochs. The validation process consists of running the prompt"
            " `args.validation_prompt` multiple times: `args.num_validation_images`."
        ),
    )
    parser.add_argument(
        "--max_train_samples",
        type=int,
        default=None,
        help=(
            "For debugging purposes or quicker training, truncate the number of training examples to this "
            "value if set."
        ),
    )
    parser.add_argument(
        "--output_dir",
        type=str,
        default="sd-model-finetuned-lora",
        help="The output directory where the model predictions and checkpoints will be written.",
    )
    parser.add_argument(
        "--cache_dir",
        type=str,
        default=None,
        help="The directory where the downloaded models and datasets will be stored.",
    )
    parser.add_argument("--seed", type=int, default=None, help="A seed for reproducible training.")
    parser.add_argument(
        "--resolution",
        type=int,
        default=512,
        help=(
            "The resolution for input images, all the images in the train/validation dataset will be resized to this"
            " resolution"
        ),
    )
    parser.add_argument(
        "--center_crop",
        default=False,
        action="store_true",
        help=(
            "Whether to center crop the input images to the resolution. If not set, the images will be randomly"
            " cropped. The images will be resized to the resolution first before cropping."
        ),
    )
    parser.add_argument(
        "--random_flip",
        action="store_true",
        help="whether to randomly flip images horizontally",
    )
    parser.add_argument(
        "--train_batch_size", type=int, default=16, help="Batch size (per device) for the training dataloader."
    )
    parser.add_argument("--num_train_epochs", type=int, default=100)
    parser.add_argument(
        "--max_train_steps",
        type=int,
        default=None,
        help="Total number of training steps to perform.  If provided, overrides num_train_epochs.",
    )
    parser.add_argument(
        "--gradient_accumulation_steps",
        type=int,
        default=1,
        help="Number of updates steps to accumulate before performing a backward/update pass.",
    )
    parser.add_argument(
        "--gradient_checkpointing",
        action="store_true",
        help="Whether or not to use gradient checkpointing to save memory at the expense of slower backward pass.",
    )
    parser.add_argument(
        "--learning_rate",
        type=float,
        default=1e-4,
        help="Initial learning rate (after the potential warmup period) to use.",
    )
    parser.add_argument(
        "--scale_lr",
        action="store_true",
        default=False,
        help="Scale the learning rate by the number of GPUs, gradient accumulation steps, and batch size.",
    )
    parser.add_argument(
        "--lr_scheduler",
        type=str,
        default="constant",
        help=(
            'The scheduler type to use. Choose between ["linear", "cosine", "cosine_with_restarts", "polynomial",'
            ' "constant", "constant_with_warmup"]'
        ),
    )
    parser.add_argument(
        "--lr_warmup_steps", type=int, default=500, help="Number of steps for the warmup in the lr scheduler."
    )
    parser.add_argument(
        "--snr_gamma",
        type=float,
        default=None,
        help="SNR weighting gamma to be used if rebalancing the loss. Recommended value is 5.0. "
        "More details here: https://arxiv.org/abs/2303.09556.",
    )
    parser.add_argument(
        "--use_8bit_adam", action="store_true", help="Whether or not to use 8-bit Adam from bitsandbytes."
    )
    parser.add_argument(
        "--allow_tf32",
        action="store_true",
        help=(
            "Whether or not to allow TF32 on Ampere GPUs. Can be used to speed up training. For more information, see"
            " https://pytorch.org/docs/stable/notes/cuda.html#tensorfloat-32-tf32-on-ampere-devices"
        ),
    )
    parser.add_argument(
        "--dataloader_num_workers",
        type=int,
        default=0,
        help=(
            "Number of subprocesses to use for data loading. 0 means that the data will be loaded in the main process."
        ),
    )
    parser.add_argument("--adam_beta1", type=float, default=0.9, help="The beta1 parameter for the Adam optimizer.")
    parser.add_argument("--adam_beta2", type=float, default=0.999, help="The beta2 parameter for the Adam optimizer.")
    parser.add_argument("--adam_weight_decay", type=float, default=1e-2, help="Weight decay to use.")
    parser.add_argument("--adam_epsilon", type=float, default=1e-08, help="Epsilon value for the Adam optimizer")
    parser.add_argument("--max_grad_norm", default=1.0, type=float, help="Max gradient norm.")
    parser.add_argument("--push_to_hub", action="store_true", help="Whether or not to push the model to the Hub.")
    parser.add_argument("--hub_token", type=str, default=None, help="The token to use to push to the Model Hub.")
    parser.add_argument(
        "--prediction_type",
        type=str,
        default=None,
        help="The prediction_type that shall be used for training. Choose between 'epsilon' or 'v_prediction' or leave `None`. If left to `None` the default prediction type of the scheduler: `noise_scheduler.config.prediciton_type` is chosen.",
    )
    parser.add_argument(
        "--hub_model_id",
        type=str,
        default=None,
        help="The name of the repository to keep in sync with the local `output_dir`.",
    )
    parser.add_argument(
        "--logging_dir",
        type=str,
        default="logs",
        help=(
            "[TensorBoard](https://www.tensorflow.org/tensorboard) log directory. Will default to"
            " *output_dir/runs/**CURRENT_DATETIME_HOSTNAME***."
        ),
    )
    parser.add_argument(
        "--mixed_precision",
        type=str,
        default=None,
        choices=["no", "fp16", "bf16"],
        help=(
            "Whether to use mixed precision. Choose between fp16 and bf16 (bfloat16). Bf16 requires PyTorch >="
            " 1.10.and an Nvidia Ampere GPU.  Default to the value of accelerate config of the current system or the"
            " flag passed with the `accelerate.launch` command. Use this argument to override the accelerate config."
        ),
    )
    parser.add_argument(
        "--report_to",
        type=str,
        default="tensorboard",
        help=(
            'The integration to report the results and logs to. Supported platforms are `"tensorboard"`'
            ' (default), `"wandb"` and `"comet_ml"`. Use `"all"` to report to all integrations.'
        ),
    )
    parser.add_argument("--local_rank", type=int, default=-1, help="For distributed training: local_rank")
    parser.add_argument(
        "--checkpointing_steps",
        type=int,
        default=500,
        help=(
            "Save a checkpoint of the training state every X updates. These checkpoints are only suitable for resuming"
            " training using `--resume_from_checkpoint`."
        ),
    )
    parser.add_argument(
        "--checkpoints_total_limit",
        type=int,
        default=None,
        help=("Max number of checkpoints to store."),
    )
    parser.add_argument(
        "--resume_from_checkpoint",
        type=str,
        default=None,
        help=(
            "Whether training should be resumed from a previous checkpoint. Use a path saved by"
            ' `--checkpointing_steps`, or `"latest"` to automatically select the last available checkpoint.'
        ),
    )
    parser.add_argument(
        "--enable_xformers_memory_efficient_attention", action="store_true", help="Whether or not to use xformers."
    )
    parser.add_argument("--noise_offset", type=float, default=0, help="The scale of noise offset.")
    parser.add_argument(
        "--train_text_encoder",
        action="store_true",
        help="Whether to train the text encoder. If set, the text encoder should be float32 precision.",
    )
    parser.add_argument(
        "--lora_path_img",
        type=str,
        default=None,
        required=True,
        help="Path to img lora weights",
    )
    parser.add_argument(
        "--lora_path_ref",
        type=str,
        default=None,
        required=True,
        help="Path to ref lora weights",
    )
    parser.add_argument(
        "--instance_datadir_img",
        type=str,
        default=None,
        help=("A folder containing the img training data."),
    )
    parser.add_argument(
        "--instance_datadir_ref",
        type=str,
        default=None,
        help=("A folder containing the ref training data"),
    )
    parser.add_argument(
        "--repeats",
        type=int,
        default=1,
        help="How many times to repeat the training data.",
    )
    parser.add_argument(
        "--instance_prompt_img",
        type=str,
        default=None,
        help="The prompt of img",
    )
    parser.add_argument(
        "--instance_prompt_ref",
        type=str,
        default=None,
        required=True,
        help="The prompt with identifier specifying the ref",
    )
    parser.add_argument(
        "--init_merger_img",
        type=float,
        default=1.0,
        help="initial value of merger coefficient vectors",
    )
    parser.add_argument(
        "--init_merger_ref",
        type=float,
        default=1.0,
        help="initial value of merger coefficient vectors",
    )
    parser.add_argument(
        "--interval_num",
        type=int,
        default=5,
        help="How many intervals among all timesteps.",
    )
    parser.add_argument("--num_of_assets", type=int, default=1)
    parser.add_argument(
        "--class_data_dir",
        type=str,
        default=None,
        required=False,
        help="A folder containing the training data of class images.",
    )
    parser.add_argument(
        "--class_prompt",
        type=str,
        default="a photo at the beach",
        help="The prompt to specify images in the same class as provided instance images.",
    )
    parser.add_argument(
        "--with_prior_preservation",
        action="store_true",
        help="Flag to add prior preservation loss.",
    )
    parser.add_argument(
        "--prior_loss_weight",
        type=float,
        default=1.0,
        help="The weight of prior preservation loss.",
    )
    parser.add_argument(
        "--num_class_images",
        type=int,
        default=100,
        help=(
            "Minimal class images for prior preservation loss. If there are not enough images already present in"
            " class_data_dir, additional images will be sampled with class_prompt."
        ),
    )
    parser.add_argument(
        "--prior_generation_precision",
        type=str,
        default=None,
        choices=["no", "fp32", "fp16", "bf16"],
        help=(
            "Choose prior generation precision between fp32, fp16 and bf16 (bfloat16). Bf16 requires PyTorch >="
            " 1.10.and an Nvidia Ampere GPU.  Default to  fp16 if a GPU is available else fp32."
        ),
    )
    parser.add_argument(
        "--sample_batch_size",
        type=int,
        default=4,
        help="Batch size (per device) for sampling images.",
    )
    parser.add_argument(
        "--img_loss_weight",
        type=float,
        default=1.0,
        help="The weight of img preservation loss.",
    )
    parser.add_argument(
        "--ref_loss_weight",
        type=float,
        default=1.0,
        help="The weight of ref preservation loss.",
    )
    parser.add_argument("--sample_index", type=int, default=0)
    parser.add_argument(
        "--label_path",
        type=str,
        default=None,
        help=("A folder containing the label"),
    )
    
    args = parser.parse_args()
    env_local_rank = int(os.environ.get("LOCAL_RANK", -1))
    if env_local_rank != -1 and env_local_rank != args.local_rank:
        args.local_rank = env_local_rank

    # Sanity checks
    if args.dataset_name is None and args.instance_datadir_img is None:
        raise ValueError("Need either a dataset name or a training folder.")

    if args.with_prior_preservation:
        if args.class_data_dir is None:
            raise ValueError("You must specify a data directory for class images.")
        if args.class_prompt is None:
            raise ValueError("You must specify prompt for class images.")
    else:
        # logger is not available yet
        if args.class_data_dir is not None:
            warnings.warn(
                "You need not use --class_data_dir without --with_prior_preservation."
            )
        if args.class_prompt is not None:
            warnings.warn(
                "You need not use --class_prompt without --with_prior_preservation."
            )

    return args

class DreamBoothDataset(Dataset):
    """
    A dataset to prepare the instance and class images with the prompts for fine-tuning the model.
    It pre-processes the images and the tokenizes prompts.
    """

    def __init__(
        self,
        instance_dataroot_img,
        instance_prompt_img,
        instance_dataroot_ref,
        instance_prompt_ref,
        class_data_root=None,
        class_prompt=None,
        size=512,
        repeats=1,
        center_crop=False,
        num_of_assets=1,
        flip_p=0.5,
    ):
        self.size = size
        self.center_crop = center_crop
        self.flip_p = flip_p

        self.instance_prompt_img = instance_prompt_img
        self.instance_prompt_ref = instance_prompt_ref

        self.image_transforms = transforms.Compose(
            [
                transforms.Resize(self.size, interpolation=transforms.InterpolationMode.BILINEAR),
                transforms.ToTensor(),
                transforms.Normalize([0.5], [0.5]),
            ]
        )
        self.mask_transforms = transforms.Compose(
            [
                transforms.ToTensor(),
            ]
        )

        self.instance_dataroot_img = Path(instance_dataroot_img)
        if not self.instance_dataroot_img.exists():
            raise ValueError(
                f"Instance {self.instance_dataroot_img} images root doesn't exists."
            )
        with open(instance_dataroot_img+'/train/metadata.jsonl', 'r', encoding="utf-8") as f:
            for line in f:
                img_data = json.loads(line)
        instance_imgpath_img = os.path.join(instance_dataroot_img+'/train', img_data['file_name'])
        self.instance_image_img = self.image_transforms(Image.open(instance_imgpath_img))

        self.instance_dataroot_ref = Path(instance_dataroot_ref)
        if not self.instance_dataroot_ref.exists():
            raise ValueError(
                f"Instance {self.instance_dataroot_ref} images root doesn't exists."
            )
        instance_imgpath_ref = os.path.join(instance_dataroot_ref, "img.jpg")
        self.instance_image_ref = self.image_transforms(Image.open(instance_imgpath_ref))
        self.instance_masks_ref = []
        for i in range(num_of_assets):
            instance_mask_path = os.path.join(instance_dataroot_ref, f"mask{i}.png")
            curr_mask = Image.open(instance_mask_path)
            curr_mask = self.mask_transforms(curr_mask)[0, None, None, ...]
            self.instance_masks_ref.append(curr_mask)
        self.instance_masks_ref = torch.cat(self.instance_masks_ref)

        self.num_instance_images = 1
        self._length = self.num_instance_images

        if class_data_root is not None:
            self.class_data_root = Path(class_data_root)
            self.class_data_root.mkdir(parents=True, exist_ok=True)
            self.class_images_path = list(self.class_data_root.iterdir())
            self.num_class_images = len(self.class_images_path)
            self._length = max(self.num_class_images, self._length)
            self.class_prompt = class_prompt
        else:
            self.class_data_root = None

    def __len__(self):
        return self._length

    def __getitem__(self, index):
        example = {}

        example["instance_image_img"] = self.instance_image_img
        example["instance_prompt_img"] = self.instance_prompt_img
        example["instance_image_ref"] = self.instance_image_ref
        example["instance_prompt_ref"] = self.instance_prompt_ref
        example["instance_mask_ref"] = self.instance_masks_ref
        '''
        if random.random() > self.flip_p:
            example["instance_image_ref"] = TF.hflip(example["instance_image_ref"])
            example["instance_mask_ref"] = TF.hflip(example["instance_mask_ref"])
        '''
        if self.class_data_root:
            class_image = Image.open(
                self.class_images_path[index % self.num_class_images]
            )
            if not class_image.mode == "RGB":
                class_image = class_image.convert("RGB")
            example["class_image"] = self.image_transforms(class_image)
            example["class_prompt"] = self.class_prompt

        return example


def collate_fn(examples, with_prior_preservation=False):
    pixel_values_img = [example["instance_image_img"] for example in examples]
    prompts_img = [example["instance_prompt_img"] for example in examples]
    pixel_values_ref = [example["instance_image_ref"] for example in examples]
    prompts_ref = [example["instance_prompt_ref"] for example in examples]
    masks_ref = [example["instance_mask_ref"] for example in examples]
    
    pixel_values_img = torch.stack(pixel_values_img)
    pixel_values_img = pixel_values_img.to(memory_format=torch.contiguous_format).float()
    pixel_values_ref = torch.stack(pixel_values_ref)
    pixel_values_ref = pixel_values_ref.to(memory_format=torch.contiguous_format).float()

    masks_ref = torch.stack(masks_ref)

    if with_prior_preservation:
        pixel_values_cls = [example["class_image"] for example in examples]
        prompts_cls = [example["class_prompt"] for example in examples]
        pixel_values_cls = torch.stack(pixel_values_cls)
        pixel_values_cls = pixel_values_cls.to(memory_format=torch.contiguous_format).float()
        batch = {
            "pixel_values_img": pixel_values_img,
            "prompts_img": prompts_img,
            "pixel_values_ref": pixel_values_ref,
            "prompts_ref": prompts_ref,
            "masks_ref": masks_ref,
            "pixel_values_cls": pixel_values_cls,
            "prompts_cls": prompts_cls,
        }
    else:
        batch = {
            "pixel_values_img": pixel_values_img,
            "prompts_img": prompts_img,
            "pixel_values_ref": pixel_values_ref,
            "prompts_ref": prompts_ref,
            "masks_ref": masks_ref,
        }
    return batch


class PromptDataset(Dataset):
    "A simple dataset to prepare the prompts to generate class images on multiple GPUs."

    def __init__(self, prompt, num_samples):
        self.prompt = prompt
        self.num_samples = num_samples

    def __len__(self):
        return self.num_samples

    def __getitem__(self, index):
        example = {}
        example["prompt"] = self.prompt
        example["index"] = index
        return example

class Merger(torch.nn.Module):
    def __init__(
        self,
        interval_num: int,
        init_merger_img: Optional[float] = 1.0,
        init_merger_ref: Optional[float] = 1.0,
    ):
        super().__init__()

        self.merger_img = torch.nn.Parameter(torch.ones((interval_num,), dtype=torch.float32) * init_merger_img)
        self.merger_ref = torch.nn.Parameter(torch.ones((interval_num,), dtype=torch.float32) * init_merger_ref)


def main():
    args = parse_args()
    # add by somene
    lines = open(args.label_path, 'r').readlines()
    line = lines[args.sample_index]
    splits = line.split('\t')
    args.output_dir = args.output_dir + splits[0] + '-' + splits[1]
    args.lora_path_img = args.lora_path_img + splits[0]
    args.instance_datadir_img = args.instance_datadir_img + splits[0]
    with open(args.instance_datadir_img+'/train/metadata.jsonl', 'r', encoding="utf-8") as f:
        for line in f:
            data = json.loads(line)
    args.instance_prompt_img = data['text']
    args.lora_path_ref = args.lora_path_ref + splits[1]
    args.instance_datadir_ref = args.instance_datadir_ref + splits[1]
    # end
    logging_dir = Path(args.output_dir, args.logging_dir)

    accelerator_project_config = ProjectConfiguration(project_dir=args.output_dir, logging_dir=logging_dir)
    
    accelerator = Accelerator(
        gradient_accumulation_steps=args.gradient_accumulation_steps,
        mixed_precision=args.mixed_precision,
        log_with=args.report_to,
        project_config=accelerator_project_config,
    )
    if args.report_to == "wandb":
        if not is_wandb_available():
            raise ImportError("Make sure to install wandb if you want to use it for logging during training.")
        import wandb

    # Currently, it's not possible to do gradient accumulation when training two models with accelerate.accumulate
    # This will be enabled soon in accelerate. For now, we don't allow gradient accumulation when training two models.
    # TODO (sayakpaul): Remove this check when gradient accumulation with two models is enabled in accelerate.
    if args.train_text_encoder and args.gradient_accumulation_steps > 1 and accelerator.num_processes > 1:
        raise ValueError(
            "Gradient accumulation is not supported when training the text encoder in distributed training. "
            "Please set gradient_accumulation_steps to 1. This feature will be supported in the future."
        )

    # Make one log on every process with the configuration for debugging.
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        level=logging.INFO,
    )
    logger.info(accelerator.state, main_process_only=False)
    if accelerator.is_local_main_process:
        datasets.utils.logging.set_verbosity_warning()
        transformers.utils.logging.set_verbosity_warning()
        diffusers.utils.logging.set_verbosity_info()
    else:
        datasets.utils.logging.set_verbosity_error()
        transformers.utils.logging.set_verbosity_error()
        diffusers.utils.logging.set_verbosity_error()

    # If passed along, set the training seed now.
    if args.seed is not None:
        set_seed(args.seed)

    # Generate class images if prior preservation is enabled.
    if args.with_prior_preservation:
        class_images_dir = Path(args.class_data_dir)
        if not class_images_dir.exists():
            class_images_dir.mkdir(parents=True)
        cur_class_images = len(list(class_images_dir.iterdir()))

        if cur_class_images < args.num_class_images:
            torch_dtype = (
                torch.float16
                if accelerator.device.type == "cuda"
                else torch.float32
            )
            if args.prior_generation_precision == "fp32":
                torch_dtype = torch.float32
            elif args.prior_generation_precision == "fp16":
                torch_dtype = torch.float16
            elif args.prior_generation_precision == "bf16":
                torch_dtype = torch.bfloat16
            pipeline = DiffusionPipeline.from_pretrained(
                args.pretrained_model_name_or_path,
                torch_dtype=torch_dtype,
                safety_checker=None,
                revision=args.revision,
            )
            pipeline.set_progress_bar_config(disable=True)

            num_new_images = args.num_class_images - cur_class_images
            logger.info(f"Number of class images to sample: {num_new_images}.")

            sample_dataset = PromptDataset(args.class_prompt, num_new_images)
            sample_dataloader = torch.utils.data.DataLoader(
                sample_dataset, batch_size=args.sample_batch_size
            )

            sample_dataloader = accelerator.prepare(sample_dataloader)
            pipeline.to(accelerator.device)

            for example in tqdm(
                sample_dataloader,
                desc="Generating class images",
                disable=not accelerator.is_local_main_process,
            ):
                images = pipeline(example["prompt"]).images

                for i, image in enumerate(images):
                    hash_image = hashlib.sha1(image.tobytes()).hexdigest()
                    image_filename = (
                        class_images_dir
                        / f"{example['index'][i] + cur_class_images}-{hash_image}.jpg"
                    )
                    image.save(image_filename)

            del pipeline
            if torch.cuda.is_available():
                torch.cuda.empty_cache()

    # Handle the repository creation
    if accelerator.is_main_process:
        if args.output_dir is not None:
            os.makedirs(args.output_dir, exist_ok=True)

        if args.push_to_hub:
            repo_id = create_repo(
                repo_id=args.hub_model_id or Path(args.output_dir).name, exist_ok=True, token=args.hub_token
            ).repo_id
    # Load scheduler, tokenizer and models.
    noise_scheduler = DDPMScheduler.from_pretrained(args.pretrained_model_name_or_path, subfolder="scheduler")
    tokenizer = CLIPTokenizer.from_pretrained(
        args.pretrained_model_name_or_path, subfolder="tokenizer", revision=args.revision
    )
    text_encoder = CLIPTextModel.from_pretrained(
        args.pretrained_model_name_or_path, subfolder="text_encoder", revision=args.revision
    )
    vae = AutoencoderKL.from_pretrained(
        args.pretrained_model_name_or_path, subfolder="vae", revision=args.revision, variant=args.variant
    )
    unet = UNet2DConditionModel.from_pretrained(
        args.pretrained_model_name_or_path, subfolder="unet", revision=args.revision, variant=args.variant
    )

    # load lora weights
    lora_state_dict_img, network_alphas_img = LoraLoaderMixin.lora_state_dict(args.lora_path_img)
    LoraLoaderMixin.load_lora_into_unet(lora_state_dict_img, network_alphas=network_alphas_img, unet=unet, adapter_name="lora_img")
    LoraLoaderMixin.load_lora_into_text_encoder(lora_state_dict_img, network_alphas=network_alphas_img, text_encoder=text_encoder, adapter_name="lora_img")
    lora_state_dict_ref, network_alphas_ref = LoraLoaderMixin.lora_state_dict(args.lora_path_ref)
    LoraLoaderMixin.load_lora_into_unet(lora_state_dict_ref, network_alphas=network_alphas_ref, unet=unet, adapter_name="lora_ref")
    LoraLoaderMixin.load_lora_into_text_encoder(lora_state_dict_ref, network_alphas=network_alphas_ref, text_encoder=text_encoder, adapter_name="lora_ref")

    # modify tokenizer
    tokenizer.add_tokens(['<asset0>'])
    text_encoder.resize_token_embeddings(len(tokenizer))
    text_encoder.get_input_embeddings().load_state_dict(torch.load(args.lora_path_ref+'/embeddings.pt'))

    # trainable lora weights
    merger = Merger(args.interval_num, args.init_merger_img, args.init_merger_ref)

    # freeze parameters of models to save more memory
    unet.requires_grad_(False)
    vae.requires_grad_(False)
    text_encoder.requires_grad_(False)
    merger.requires_grad_(True)

    # For mixed precision training we cast all non-trainable weigths (vae, non-lora text_encoder and non-lora unet) to half-precision
    # as these weights are only used for inference, keeping weights in full precision is not required.
    weight_dtype = torch.float32
    if accelerator.mixed_precision == "fp16":
        weight_dtype = torch.float16
    elif accelerator.mixed_precision == "bf16":
        weight_dtype = torch.bfloat16

    # Move unet, vae and text_encoder to device and cast to weight_dtype
    vae.to(accelerator.device, dtype=weight_dtype)
    unet.to(accelerator.device, dtype=weight_dtype)
    text_encoder.to(accelerator.device, dtype=weight_dtype)
    merger.to(accelerator.device, dtype=weight_dtype)
    
    if args.enable_xformers_memory_efficient_attention:
        if is_xformers_available():
            import xformers

            xformers_version = version.parse(xformers.__version__)
            if xformers_version == version.parse("0.0.16"):
                logger.warn(
                    "xFormers 0.0.16 cannot be used for training in some GPUs. If you observe problems during training, please update xFormers to at least 0.0.17. See https://huggingface.co/docs/diffusers/main/en/optimization/xformers for more details."
                )
            unet.enable_xformers_memory_efficient_attention()
        else:
            raise ValueError("xformers is not available. Make sure it is installed correctly")

    if args.gradient_checkpointing:
        unet.enable_gradient_checkpointing()
        if args.train_text_encoder:
            text_encoder.gradient_checkpointing_enable()

    # Enable TF32 for faster training on Ampere GPUs,
    # cf https://pytorch.org/docs/stable/notes/cuda.html#tensorfloat-32-tf32-on-ampere-devices
    if args.allow_tf32:
        torch.backends.cuda.matmul.allow_tf32 = True

    if args.scale_lr:
        args.learning_rate = (
            args.learning_rate * args.gradient_accumulation_steps * args.train_batch_size * accelerator.num_processes
        )

    # Initialize the optimizer
    if args.use_8bit_adam:
        try:
            import bitsandbytes as bnb
        except ImportError:
            raise ImportError(
                "Please install bitsandbytes to use 8-bit Adam. You can do so by running `pip install bitsandbytes`"
            )

        optimizer_cls = bnb.optim.AdamW8bit
    else:
        optimizer_cls = torch.optim.AdamW

    # Optimizer creation
    #params_to_optimize = merger.parameters()
    for key, p in unet.named_parameters():
        if "patch_img" in key or "patch_ref" in key:
            p.requires_grad = True
    for key, p in text_encoder.named_parameters():
        if "patch_img" in key or "patch_ref" in key:
            p.requires_grad = True
    params_to_optimize = list(merger.parameters()) + list(filter(lambda p: p.requires_grad, unet.parameters())) + list(filter(lambda p: p.requires_grad, text_encoder.parameters()))
    '''
    params_to_optimize = ([
        {"params": list(merger.parameters()), "lr": args.learning_rate},
        {"params": list(filter(lambda p: p.requires_grad, unet.parameters())) + list(filter(lambda p: p.requires_grad, text_encoder.parameters())), "lr": 2e-2},
    ])
    '''
    optimizer = optimizer_cls(
        params_to_optimize,
        lr=args.learning_rate,
        betas=(args.adam_beta1, args.adam_beta2),
        weight_decay=args.adam_weight_decay,
        eps=args.adam_epsilon,
    )

    # Dataset and DataLoaders creation:
    train_dataset = DreamBoothDataset(
        instance_dataroot_img=args.instance_datadir_img,
        instance_prompt_img=args.instance_prompt_img,
        instance_dataroot_ref=args.instance_datadir_ref,
        instance_prompt_ref=args.instance_prompt_ref,
        class_data_root=args.class_data_dir
        if args.with_prior_preservation
        else None,
        class_prompt=args.class_prompt,
        size=args.resolution,
        repeats=args.repeats,
        center_crop=args.center_crop,
        num_of_assets=args.num_of_assets,
    )

    train_dataloader = torch.utils.data.DataLoader(
        train_dataset,
        batch_size=args.train_batch_size,
        shuffle=True,
        collate_fn=lambda examples: collate_fn(
            examples, args.with_prior_preservation
        ),
        num_workers=args.dataloader_num_workers,
    )

    # Scheduler and math around the number of training steps.
    overrode_max_train_steps = False
    num_update_steps_per_epoch = math.ceil(len(train_dataloader) / args.gradient_accumulation_steps)
    if args.max_train_steps is None:
        args.max_train_steps = args.num_train_epochs * num_update_steps_per_epoch
        overrode_max_train_steps = True

    lr_scheduler = get_scheduler(
        args.lr_scheduler,
        optimizer=optimizer,
        num_warmup_steps=args.lr_warmup_steps * accelerator.num_processes,
        num_training_steps=args.max_train_steps * accelerator.num_processes,
    )

    # Prepare everything with our `accelerator`.
    unet, text_encoder, merger, optimizer, train_dataloader, lr_scheduler = accelerator.prepare(unet, text_encoder, merger, optimizer, train_dataloader, lr_scheduler)

    # We need to recalculate our total training steps as the size of the training dataloader may have changed.
    num_update_steps_per_epoch = math.ceil(len(train_dataloader) / args.gradient_accumulation_steps)
    if overrode_max_train_steps:
        args.max_train_steps = args.num_train_epochs * num_update_steps_per_epoch
    # Afterwards we recalculate our number of training epochs
    args.num_train_epochs = math.ceil(args.max_train_steps / num_update_steps_per_epoch)

    # We need to initialize the trackers we use, and also store our configuration.
    # The trackers initializes automatically on the main process.
    if accelerator.is_main_process:
        accelerator.init_trackers("ref-tbie", config=vars(args))

    # Train!
    total_batch_size = args.train_batch_size * accelerator.num_processes * args.gradient_accumulation_steps

    logger.info("***** Running training *****")
    logger.info(f"  Num examples = {len(train_dataset)}")
    logger.info(f"  Num Epochs = {args.num_train_epochs}")
    logger.info(f"  Instantaneous batch size per device = {args.train_batch_size}")
    logger.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_batch_size}")
    logger.info(f"  Gradient Accumulation steps = {args.gradient_accumulation_steps}")
    logger.info(f"  Total optimization steps = {args.max_train_steps}")
    global_step = 0
    first_epoch = 0

    # Potentially load in the weights and states from a previous save
    if args.resume_from_checkpoint:
        if args.resume_from_checkpoint != "latest":
            path = os.path.basename(args.resume_from_checkpoint)
        else:
            # Get the most recent checkpoint
            dirs = os.listdir(args.output_dir)
            dirs = [d for d in dirs if d.startswith("checkpoint")]
            dirs = sorted(dirs, key=lambda x: int(x.split("-")[1]))
            path = dirs[-1] if len(dirs) > 0 else None

        if path is None:
            accelerator.print(
                f"Checkpoint '{args.resume_from_checkpoint}' does not exist. Starting a new training run."
            )
            args.resume_from_checkpoint = None
            initial_global_step = 0
        else:
            accelerator.print(f"Resuming from checkpoint {path}")
            accelerator.load_state(os.path.join(args.output_dir, path))
            global_step = int(path.split("-")[1])

            initial_global_step = global_step
            first_epoch = global_step // num_update_steps_per_epoch
    else:
        initial_global_step = 0

    progress_bar = tqdm(
        range(0, args.max_train_steps),
        initial=initial_global_step,
        desc="Steps",
        # Only show the progress bar once on each machine.
        disable=not accelerator.is_local_main_process,
    )

    for epoch in range(first_epoch, args.num_train_epochs):
        train_loss = 0.0
        train_loss_img = 0.0
        train_loss_ref = 0.0
        for step, batch in enumerate(train_dataloader):
            with accelerator.accumulate(merger):
                model_inputs = []
                input_ids_list = []
                for i in range(2):
                    pixel_values_key = "pixel_values_img" if i == 0 else "pixel_values_ref"
                    prompts_key = "prompts_img" if i == 0 else "prompts_ref"
                    pixel_values = batch[pixel_values_key]
                    prompts = batch[prompts_key]
                    model_input = vae.encode(pixel_values.to(dtype=weight_dtype)).latent_dist.sample()
                    model_input = model_input * vae.config.scaling_factor
                    model_inputs.append(model_input)

                    input_ids = tokenizer(prompts, max_length=tokenizer.model_max_length, padding="max_length", truncation=True, return_tensors="pt").input_ids
                    input_ids_list.append(input_ids.to(accelerator.device))

                # Sample noise that we'll add to the latents
                noise = torch.randn_like(model_inputs[0])
                if args.noise_offset:
                    # https://www.crosslabs.org//blog/diffusion-with-offset-noise
                    noise += args.noise_offset * torch.randn(
                        (model_inputs[0].shape[0], model_inputs[0].shape[1], 1, 1), device=model_inputs[0].device
                    )

                bsz = model_inputs[0].shape[0]
                # Sample a random timestep for each image
                timesteps = torch.randint(0, noise_scheduler.config.num_train_timesteps, (bsz,), device=model_inputs[0].device)
                timesteps = timesteps.long()

                merger_num = int(timesteps[0] / math.ceil(1.0 * noise_scheduler.config.num_train_timesteps / args.interval_num))

                # Add noise to the model input according to the noise magnitude at each timestep
                # (this is the forward diffusion process)
                for i in range(2):
                    model_inputs[i] = noise_scheduler.add_noise(model_inputs[i], noise, timesteps)

                max_masks = torch.max(batch["masks_ref"], axis=1).values
                downsampled_mask = F.interpolate(input=max_masks, size=(64, 64))

                unet.set_adapters(["lora_img"])
                LoraLoaderMixin.set_adapters_for_text_encoder(LoraLoaderMixin, adapter_names=["lora_img"], text_encoder=text_encoder)
                # 1. img weights + img
                with torch.no_grad():
                    encoder_hidden_state = text_encoder(input_ids_list[0], current_t=timesteps[0])[0]
                    model_pred_img = unet(model_inputs[0], timesteps, encoder_hidden_state, current_t=timesteps[0]).sample

                unet.set_adapters(["lora_ref"])
                LoraLoaderMixin.set_adapters_for_text_encoder(LoraLoaderMixin, adapter_names=["lora_ref"], text_encoder=text_encoder)
                # 2. ref weights + ref
                with torch.no_grad():
                    encoder_hidden_state = text_encoder(input_ids_list[1], current_t=timesteps[0])[0]
                    model_pred_ref = unet(model_inputs[1], timesteps, encoder_hidden_state, current_t=timesteps[0]).sample * downsampled_mask

                unet.set_adapters(["lora_img", "lora_ref"], [merger.merger_img[merger_num], merger.merger_ref[merger_num]])
                LoraLoaderMixin.set_adapters_for_text_encoder(LoraLoaderMixin, adapter_names=["lora_img", "lora_ref"], text_encoder=text_encoder, text_encoder_weights=[merger.merger_img[merger_num], merger.merger_ref[merger_num]])
                # 3. merged weights + img
                encoder_hidden_state = text_encoder(input_ids_list[0], current_t=timesteps[0])[0]
                model_pred_mimg = unet(model_inputs[0], timesteps, encoder_hidden_state, current_t=timesteps[0]).sample
                # 4. merged weights + ref
                encoder_hidden_state = text_encoder(input_ids_list[1], current_t=timesteps[0])[0]
                model_pred_mref = unet(model_inputs[1], timesteps, encoder_hidden_state, current_t=timesteps[0]).sample * downsampled_mask

                loss_img = F.mse_loss(model_pred_mimg.float(), model_pred_img.float(), reduction="mean")
                loss_ref = F.mse_loss(model_pred_mref.float(), model_pred_ref.float(), reduction="mean")
                loss = args.img_loss_weight * loss_img + args.ref_loss_weight * loss_ref

                if args.with_prior_preservation:
                    input_ids = tokenizer(batch["prompts_cls"], max_length=tokenizer.model_max_length, padding="max_length", truncation=True, return_tensors="pt").input_ids
                    encoder_hidden_state = text_encoder(input_ids.to(accelerator.device))[0]
                    model_input = vae.encode(batch["pixel_values_cls"].to(dtype=weight_dtype)).latent_dist.sample()
                    model_input = model_input * vae.config.scaling_factor
                    model_input = noise_scheduler.add_noise(model_input, noise, timesteps)
                    model_pred_mcls = unet(model_input, timesteps, encoder_hidden_state).sample
                    loss = loss + args.prior_loss_weight * F.mse_loss(model_pred_mcls.float(), noise.float(), reduction="mean")

                # Gather the losses across all processes for logging (if we use distributed training).
                avg_loss = accelerator.gather(loss.repeat(args.train_batch_size)).mean()
                train_loss += avg_loss.item() / args.gradient_accumulation_steps
                avg_loss = accelerator.gather(loss_img.repeat(args.train_batch_size)).mean()
                train_loss_img += avg_loss.item() / args.gradient_accumulation_steps
                avg_loss = accelerator.gather(loss_ref.repeat(args.train_batch_size)).mean()
                train_loss_ref += avg_loss.item() / args.gradient_accumulation_steps

                # Backpropagate
                accelerator.backward(loss)
                if accelerator.sync_gradients:
                    #params_to_clip = list(merger.parameters()) + list(filter(lambda p: p.requires_grad, unet.parameters())) + list(filter(lambda p: p.requires_grad, text_encoder.parameters()))
                    accelerator.clip_grad_norm_(params_to_optimize, args.max_grad_norm)
                optimizer.step()
                lr_scheduler.step()
                optimizer.zero_grad()

            # Checks if the accelerator has performed an optimization step behind the scenes
            if accelerator.sync_gradients:
                progress_bar.update(1)
                global_step += 1
                accelerator.log({"train_loss": train_loss}, step=global_step)
                accelerator.log({"train_loss_img": train_loss_img}, step=global_step)
                accelerator.log({"train_loss_ref": train_loss_ref}, step=global_step)
                train_loss = 0.0
                train_loss_img = 0.0
                train_loss_ref = 0.0

                if global_step % args.checkpointing_steps == 0:
                    if accelerator.is_main_process:
                        # _before_ saving state, check if this save would set us over the `checkpoints_total_limit`
                        if args.checkpoints_total_limit is not None:
                            checkpoints = os.listdir(args.output_dir)
                            checkpoints = [d for d in checkpoints if d.startswith("checkpoint")]
                            checkpoints = sorted(checkpoints, key=lambda x: int(x.split("-")[1]))

                            # before we save the new checkpoint, we need to have at _most_ `checkpoints_total_limit - 1` checkpoints
                            if len(checkpoints) >= args.checkpoints_total_limit:
                                num_to_remove = len(checkpoints) - args.checkpoints_total_limit + 1
                                removing_checkpoints = checkpoints[0:num_to_remove]

                                logger.info(
                                    f"{len(checkpoints)} checkpoints already exist, removing {len(removing_checkpoints)} checkpoints"
                                )
                                logger.info(f"removing checkpoints: {', '.join(removing_checkpoints)}")

                                for removing_checkpoint in removing_checkpoints:
                                    removing_checkpoint = os.path.join(args.output_dir, removing_checkpoint)
                                    shutil.rmtree(removing_checkpoint)

                        save_path = os.path.join(args.output_dir, f"checkpoint-{global_step}")
                        accelerator.save_state(save_path)
                        logger.info(f"Saved state to {save_path}")

            logs = {"step_loss": loss.detach().item(), "lr": lr_scheduler.get_last_lr()[0]}
            progress_bar.set_postfix(**logs)

            if global_step >= args.max_train_steps:
                break

    # Save the lora layers
    accelerator.wait_for_everyone()
    if accelerator.is_main_process:

        merger = accelerator.unwrap_model(merger)
        torch.save(merger.state_dict(), args.output_dir+'/merger.pt')

        unet = accelerator.unwrap_model(unet)
        unet_save_dict = {}
        for key in unet.state_dict():
            if "patch_img" in key or "patch_ref" in key:
                unet_save_dict.update({key: unet.state_dict()[key]})
        torch.save(unet_save_dict, args.output_dir+'/weight_unet.pt')

        text_encoder = accelerator.unwrap_model(text_encoder)
        text_save_dict = {}
        for key in text_encoder.state_dict():
            if "patch_img" in key or "patch_ref" in key:
                text_save_dict.update({key: text_encoder.state_dict()[key]})
        torch.save(text_save_dict, args.output_dir+'/weight_text.pt')

    accelerator.end_training()


if __name__ == "__main__":
    main()