import json
import logging
import os
import platform
import random
import re
from typing import List, Optional
# import boto3
# from botocore.config import Config
# import traceback
import time


import anthropic
from google import genai
from google.genai import types
import openai
import openai.api_requestor
import requests.exceptions
import torch
from anthropic import AI_PROMPT, HUMAN_PROMPT, Anthropic
from langchain.schema import AIMessage, BaseMessage, HumanMessage
from retry import retry

LOGGER = logging.getLogger(__name__)

openai.api_requestor.TIMEOUT_SECS = 20


# response = client_gemini.models.generate_content(
#     model="gemini-2.0-flash", contents="Explain how AI works"
# )
# print(response.text)


# session = boto3.Session()


# config = Config(

#     read_timeout=120, # corresponds to inference time limit set for Bedrock 

#     connect_timeout=120,

#     retries={

#         "max_attempts": 5,

#     },

# )


# bedrock = session.client(

#     service_name='bedrock-runtime',

#     region_name="us-east-2",

#     endpoint_url="https://bedrock-runtime.us-east-2.amazonaws.com",

#     config=config

# )





def prompt_to_openai_for_guess_correct(game_type, guesser_response, entity_name):
    if game_type == "celebrities":
        openai_prompt = f"""
        You are evaluating a guess made by a player in a 20-questions game. The player is attempting to guess the name of a celebrity.

        Here is the guesser's response: "{guesser_response}"

        Your task is to check if the guesser's response includes the **correct name of the celebrity** they are trying to guess. The name of the celebrity in English is: "{entity_name}". 

        The guess can be in any language or transliterated. It does not need to match the full name exactly (e.g., middle names can be neglected), but it must directly reference the celebrity's name. The guess must **explicitly include the celebrity's name** and cannot just describe their characteristics, actions, or attributes.

        Respond strictly in the following format:
        Response: "Yes"/"No"
        """
    
    elif game_type == "things":
        openai_prompt = f"""
        You are evaluating a guess made by a player in a 20-questions game. The player is attempting to guess the name of an object or thing.

        Here is the guesser's response: "{guesser_response}"

        Your task is to check if the guesser's response includes the **exact name of the object or thing** they are trying to guess. The name of the object or thing in English is: "{entity_name}". 

        The guess can be in any language or transliterated. Ensure the guess explicitly includes the correct name of the object or thing and not just descriptions, attributes, or vague references.

        Respond strictly in the following format:
        Response: "Yes"/"No"
        """

    return openai_prompt

def prompt_to_openai_for_giveup_check(game_type, guesser_response):
    if game_type == "celebrities":
        openai_prompt = f"""
        You are analyzing a player's response in a 20-questions guessing game. The player is attempting to guess the name of a **celebrity**.

        **Player's Response:** "{guesser_response}"

        Task:
        Determine whether the player's response **explicitly or implicitly** indicates that they have given up (i.e., they admit they are unable to guess the celebrity).  

        Important Notes:
        - The response **can be in any language** 

        A response should be considered as "giving up" if it:
        - Clearly states **inability to make a guess** and doesn't proceed to guess or ask question in the same response (e.g., "I don’t know," "I give up," "I can't figure it out," "No idea").  
        - Asks the other party to **reveal the answer** (e.g., "Tell me the answer").  
        - Uses words or phrases that strongly imply defeat or frustration and doesn't proceed to guess or ask question in the same response (e.g., "I surrender," "I have no guess," "I quit").  
        - Doesn't make a guess and asks a question/responds with something not related to the game (e.g., "What's the weather like?"). 

        A response should NOT be considered as "giving up" if it:
        - Makes an actual guess and asks if it is correct (e.g., "Is it Brad Pitt?").  
        - Asks a clarifying question to narrow down the options (e.g., "Is the person alive?").
        - Expresses uncertainty but still attempts to reason (e.g., "Not sure, but is it a musician?").
        - Clearly states **inability to make a guess** but proceeds to guess or ask question in the same response ("I can't figure it out, but is it a singer?").
        - Uses words or phrases that strongly imply defeat or frustration but proceeds to guess or ask question in the same response (e.g. "I have no guess, but is it Lionel Messi?").

        Answer Yes if the player's response indicates that they have given up, and No if it does not.

        Strictly respond in the following format:
        Response: "Yes"/"No"
        """

    elif game_type == "things":
        openai_prompt = f"""
        You are analyzing a player's response in a 20-questions guessing game. The player is attempting to guess the name of an **object or thing**.

        **Player's Response:** "{guesser_response}"

        Task:
        Determine whether the player's response **explicitly or implicitly** indicates that they have given up (i.e., they admit they are unable to guess the object or thing).

        Important Note:
        - The response **can be in any language**
        
        A response should be considered as "giving up" if it:
        - Clearly states **inability to make a guess** and doesn't proceed to guess or ask question in the same response (e.g., "I don’t know," "I give up," "I can't figure it out," "No idea").
        - Asks the other party to **reveal the answer** (e.g., "Tell me the answer").
        - Uses words or phrases that strongly imply defeat or frustration and doesn't proceed to guess or ask question in the same response (e.g., "I surrender," "I have no guess," "I quit").
        - Doesn't make a guess and asks a question/responds with something not related to the game (e.g., "What's the weather like?").

       A response should NOT be considered as "giving up" if it:
        - Makes an actual guess and asks if it is correct (e.g., "Is it a chair?").
        - Asks a clarifying question to narrow down the options (e.g., "Is it a living thing?").
        - Expresses uncertainty but still attempts to reason (e.g., "Not sure, but is it a living thing?").
        - Clearly states **inability to make a guess** but proceeds to guess or ask question in the same response ("I can't figure it out, but is it a living thing?").
        - Uses words or phrases that strongly imply defeat or frustration but proceeds to guess or ask question in the same response (e.g. "I have no guess, but is it a chair?").

        Answer Yes if the player's response indicates that they have given up, and No if it does not.

        Strictly respond in the following format:
        Response: "Yes"/"No"
        """

    return openai_prompt

class Q20Game:
    def __init__(
        self,
        item: str,
        answerer_model: str = "gpt-4o-mini-2024-07-18",
        guesser_model: str = "gpt-3.5-turbo-0613",
        num_turns: int = 20,
        temperature: float = 0.8,
        guesser_tokenizer=None,
        openai_api: bool = True,
        openai_api_key: Optional[str] = None,
        guesser_kargs={},
        language = "english",
        game_type = "things"
    ) -> None:
        self.item = item
        self.answerer_model = answerer_model
        self.guesser_model = guesser_model
        self.num_turns = num_turns
        self.temperature = temperature
        self.openai_api = openai_api
        self.guesser_tokenizer = guesser_tokenizer
        self.guesser_kargs = guesser_kargs
        self.language = language
        self.game_type = game_type
        

        if isinstance(guesser_model, str) and "gemini" in guesser_model:
            self.client_gemini = genai.Client(api_key=os.environ["GENAI_API_KEY"])

        # Read the prompts.json file
        with open("prompts.json") as f:
            prompts = json.load(f)
        
        self.prompts = prompts[self.game_type][self.language]
        
        self.vicuna_prompt = self.prompts['initial_system_prompt']
        self.first_user_utterance = self.prompts['guesser_prompt']
        self.guesser_win = False
        if openai_api_key is not None:
            openai.api_key = openai_api_key

        if isinstance(answerer_model, str) and not answerer_model.startswith("gpt"):
            self.user_api_base = "http://0.0.0.0:8000/v1"
        else:
            self.user_api_base = "https://api.openai.com/v1"

        if isinstance(guesser_model, str) and not guesser_model.startswith("gpt"):
            self.guesser_api_base = "http://0.0.0.0:8000/v1"
        else:
            self.guesser_api_base = "https://api.openai.com/v1"

        self.anthropic_api = Anthropic()

        self.guesser_messages = []

    #Function for the API call to openai
    def api_call_openai(self, model_name, prompt, temp, max_tokens):

        # Create messages list using prompt
        messages = [
            {
                "role": "user",
                "content": f"{prompt}",
            }
        ]

        try:
            openai.api_base = "https://api.openai.com/v1"
            response = openai.ChatCompletion.create(
                model=model_name,
                messages=messages,
                max_tokens=max_tokens,
                n=1,
                stop=None,
                temperature=temp,
            )
            return response.choices[0].message.to_dict()["content"].strip().strip("**").strip("*").strip()
            
        except Exception as e:
            return None
    
    # Function for API call to gemini
    def api_call_gemini(self, model_name, prompt, temp, max_tokens):

        try:
            response = self.client_gemini.models.generate_content(
                model= model_name,
                contents=[prompt],
                config=types.GenerateContentConfig(
                    max_output_tokens=max_tokens,
                    temperature=temp
                )
            )

            # print("Response Text: ", response.text)
            return response.text
        except Exception as e:
            print(e)
            return None
    


    # def api_call_llama(self, prompt, temperature, max_tokens):
        
    #     api_template = {
    #         "modelId": "meta.llama3-3-70b-instruct-v1:0",
    #         "contentType": "application/json",
    #         "accept": "application/json",
    #         "body": ""
    #     }

    #     body = {
    #         "prompt": f"""
    #         <|begin_of_text|>
    #         <|start_header_id|>user<|end_header_id|>
    #         {prompt}
    #         <|eot_id|>
    #         <|start_header_id|>assistant<|end_header_id|>
    #         """,
    #         "temperature": temperature,
    #         "max_gen_len": max_tokens
    #     }

    #     api_template["body"] = json.dumps(body)

    #     success = False
    #     for i in range(5):
    #         try:
    #             response = bedrock.invoke_model(**api_template)
    #             success = True
    #             break
    #         except:
                
    #             traceback.print_exc()
    #             time.sleep(1)

    #     if success:
    #         response_body = json.loads(response.get('body').read())
    #         return response_body["generation"].strip()
    #     else:
    #         print("***exception!!!!!")
    #         return None


    # def api_call_mistral(self, prompt, max_tokens, temperature):
        
    #     api_template = {
    #         "modelId": "mistral.mistral-large-2402-v1:0",
    #         "contentType": "application/json",
    #         "accept": "application/json",
    #         "body": ""
    #     }

    #     body = {
    #         "prompt": "<s>[INST] " + prompt + " [/INST]",
    #         "temperature": temperature,
    #         "max_tokens": max_tokens
    #     }
        

    #     api_template["body"] = json.dumps(body)

    #     success = False
    #     for i in range(5):
    #         try:
    #             response = bedrock.invoke_model(**api_template)
    #             success = True
    #             break
    #         except:
                
    #             traceback.print_exc()
    #             time.sleep(1)

    #     if success:
    #         response_body = json.loads(response.get('body').read())
    #         generated_text = response_body.get('outputs', [{}])[0].get('text', '').strip()
    #         return generated_text
    #     else:
    #         print("***exception!!!!!")
    #         return None
        

    def confusion_matrix(self, path):
        self.reset()
        with open(path) as f:
            raw_messages = json.load(f)
            self.item = path.split("/")[-1].split("_")[0]
            roles = ["assistant", "user"]
            for i, message in enumerate(raw_messages):
                self.guesser_messages.append(
                    {"role": roles[i % 2], "content": message["content"]}
                )

        self.guesser_messages = self.guesser_messages[:-2]
        self.guesser_messages[-1]["content"] = (
            self.guesser_messages[-1]["content"] + " You must guess now, what's it?"
        )
        guesser_msg = self.guesser(self.guesser_messages)
        self.guesser_messages.append(guesser_msg)
        guesser_question = guesser_msg["content"].strip()
        self.guesser_messages[-1]["content"] = (
            self.guesser_messages[-1]["content"] + " Is it right?"
        )
        usr_msg = self.answerer(guesser_question)
        self.guesser_messages.append(
            {"role": "user", "content": f"{usr_msg['content'].strip()}"}
        )

        if "bingo" in self.guesser_messages[-1]["content"].lower():
            self.guesser_win = True
            return True

        return False

    @retry(
        (
            openai.error.Timeout,
            requests.exceptions.ReadTimeout,
            openai.error.ServiceUnavailableError,
            openai.error.RateLimitError,
            openai.error.APIError,
            requests.exceptions.HTTPError,
            openai.error.APIConnectionError,
        ),
        tries=5,
        delay=0.5,
        backoff=0.5,
        max_delay=2,
        logger=LOGGER,
    )
    def guesser(self, messages):


        if isinstance(self.guesser_model, str) and self.guesser_model.startswith(
            "claude"
        ):
            
            ## convert to claude type
            prompt = ""
            for item in messages:
                if item["role"].upper() == "USER":
                    prompt += f"{HUMAN_PROMPT} {item['content']}"
                elif item["role"].upper() == "ASSISTANT":
                    prompt += f"{AI_PROMPT} {item['content']}"
    
            prompt += f"{AI_PROMPT}"

            completion = self.anthropic_api.completions.create(
                model=self.guesser_model,
                max_tokens_to_sample=256,
                prompt=prompt,
                temperature=self.temperature,
            )

            content = completion.complete.lstrip()
            content = content.replace("\n", ". ")

            return {
                "role": "assistant",
                "content": content,
            }


        elif not isinstance(self.guesser_model, str):
            # """Wraps hf's `generate` adding some specific method's defaults"""
            prompt = self.dialog_history() + " ASSISTANT:"
            # Ensure tokenizer has a pad token
            if self.guesser_tokenizer.pad_token is None:
                self.guesser_tokenizer.pad_token = self.guesser_tokenizer.eos_token

            # Encode with attention mask
            encoded_input = self.guesser_tokenizer(
                prompt, return_tensors="pt", padding=True
            )

            input_ids = encoded_input["input_ids"].to(self.guesser_model.device)
            attention_mask = encoded_input["attention_mask"].to(self.guesser_model.device)

            with torch.no_grad():
                gen = self.guesser_model.generate(
                    input_ids=input_ids,
                    attention_mask=attention_mask,
                    **self.guesser_kargs,
                )
                gen_str = (
                    self.guesser_tokenizer.decode(gen[0][input_ids[0].shape[0] :])
                    .split("</s>")[0]
                    .split("USER")[0]
                    .split("<|eot_id|>")[0]
                    .lstrip()
                    .strip() 
                )

            # if "llama" in self.guesser_model:
            #     # print(len((prompt, self.temperature, 64)))
            #     # print(self.temperature)
            #     gen_str = self.api_call_llama(prompt, self.temperature, 100)
            
            # elif "mistral" in self.guesser_model:
            #     gen_str = self.api_call_mistral(prompt, 100, self.temperature)

            # # Check if there is a </s> in the gen_str, t there is then split it and take the previous part
                if "</s>" in gen_str:
                    gen_str = gen_str.split("</s>")[0].strip()

                # Check if there are any /n, if yes then replace it with ". "
                gen_str = gen_str.replace("\n", ". ")

                # Check if there is "ASSISTANT", if yes then split it
                if "ASSISTANT" in gen_str:
                    gen_str = gen_str.split("ASSISTANT")[0].strip()

                return {
                    "role": "assistant",
                    "content": gen_str,
                }
            
        elif (isinstance(self.guesser_model, str)) and ("gemini" in self.guesser_model):
                                                    
            prompt = ""
            for item in messages:
                if item["role"].upper() == "USER":
                    prompt += f"USER: {item['content']}\n"
                elif item["role"].upper() == "ASSISTANT":
                    prompt += f"ASSISTANT: {item['content']}\n"

            prompt+= "ASSISTANT:"

            times_runned = 0
            while(times_runned<5):
                response = self.api_call_gemini(self.guesser_model, prompt, self.temperature, 100)

                if response is None:
                    print("Response is None! Retrying")
                    times_runned += 1
                    if (times_runned == 5):
                        response = "N/A"
                        print("Response from Gemini is None! Breaking the loop")
                        break
                    continue
                break
            
            # Split the response by first USER or User and then take the first part
            if "USER" in response:
                response = response.split("USER")[0].strip()
                # Remove any \n
                response = response.replace("\n", ". ")
            
            if "User" in response:
                response = response.split("User")[0].strip()
                # Remove any \n
                response = response.replace("\n", ". ")

            response = response.replace("\n", ". ")
            
            return {
                "role": "assistant",
                "content": response.strip(),
            }


        else:
            openai.api_base = "https://api.openai.com/v1"
            response = openai.ChatCompletion.create(
                model=self.guesser_model,
                messages=messages,
                max_tokens=100,
                n=1,
                stop=None,
                temperature=self.temperature,
            )

            content = response.choices[0].message.to_dict()["content"].strip()
            content = content.replace("\n", ". ")

            return {
                "role": "assistant",
                "content": content,
            }

    def dialog_history(self):
        history = self.vicuna_prompt + " "
        for item in self.guesser_messages:
            if item["role"].upper() == "USER":
                history += "USER: " + item["content"]
            elif item["role"].upper() == "ASSISTANT":
                history += " " + "ASSISTANT: " + item["content"] + "</s>"
        return history

    def game_play(self, path, user_mode=False):
        # Check if the game is already played by looking if the directory exists
        if os.path.exists(os.path.join(path, f"{self.item}.txt")):
            return
    
        self.reset()
        # print(f"Item: {self.item}")
        for t in range(self.num_turns):
            # System asking a question
            if (not user_mode) or user_mode is None:
                guesser_msg = self.guesser(self.guesser_messages)
                # guesser_msg["content"] = re.sub(r'the entity you are thinking of', 'it', guesser_msg["content"])
                # guesser_msg["content"] = re.sub(r"the entity you're thinking of", 'it', guesser_msg["content"])
                # guesser_msg["content"] = re.sub(r" you're thinking of", '', guesser_msg["content"])
                # guesser_msg["content"] = re.sub(r" you are thinking of", '', guesser_msg["content"])
            else:
                user_q = input(
                    f"Type in your questions for turn {t+1}. (e.g. Is it a living thing?)\n"
                )
                guesser_msg = {"role": "assistant", "content": user_q}
            self.guesser_messages.append(guesser_msg)
            guesser_question = guesser_msg["content"].strip()

            if t == self.num_turns - 1:
                self.guesser_messages[-1]["content"] = (
                    self.guesser_messages[-1]["content"] + self.prompts['guesser_asking_if_guess_is_correct']
                )

            usr_msg = self.answerer(guesser_question)


            self.guesser_messages.append(
                {"role": "user", "content": f"{usr_msg['content'].strip()}"}
            )

            if "bingo" in usr_msg["content"].lower():
                self.guesser_win = True
                return True
            
            # Check if given up
            if "you have given up" in usr_msg["content"].lower():
                self.guesser_win = False
                return False
            
            if t == self.num_turns - 2:
                self.guesser_messages[-1]["content"] = (
                    self.guesser_messages[-1]["content"]
                    + self.prompts['telling_guesser_it_has_to_guess_now']
                )

        return False

    def save_session(self, path):
        # Print the conversation
        if not os.path.exists(path):
            os.makedirs(path)
        output_file = os.path.join(path, f"{self.item}.txt")
        # Check if the game is already played by looking if the directory exists
        if os.path.exists(output_file):
            print(f"Game already played for {self.item}")
            return

        with open(output_file, "w") as out_f:
            out_f.write(f"item: {self.item}\n")
            for t, message in enumerate(self.guesser_messages):
                out_f.write(
                    f"Turn {(t+1)//2}, {message['role'].capitalize()}: {message['content'].lstrip()}\n"
                )

    def reward(self):
        if self.guesser_win:
            n_turns = (len(self.guesser_messages) + 1) // 2
            return 1 - max(n_turns - 5, 0) * 0.02
        return 0

    def num_success(self):
        return 1 if self.guesser_win else 0

    def num_yes(self):
        n_yes = sum(
            ["yes" in msg["content"].lower() for msg in self.guesser_messages[2::2]]
        )
        return n_yes

    @retry(
        (
            openai.error.Timeout,
            requests.exceptions.ReadTimeout,
            openai.error.ServiceUnavailableError,
            openai.error.RateLimitError,
            openai.error.APIError,
            openai.error.APIConnectionError,
            anthropic.InternalServerError,
        ),
        tries=5,
        delay=0.5,
        backoff=0.5,
        max_delay=2,
        logger=LOGGER,
    )

    def answerer(self, question):

        if isinstance(self.guesser_model, str) and "gpt" in self.guesser_model:
            openai.api_base = "https://api.openai.com/v1"

        user_messages = [
            {
                "role": "user",
                "content": self.prompts['answerer_system_prompt'].format(item = self.item),
            },
            {
                "role": "user",
                "content": self.prompts['answerer_prompt'].format(question = question, item = self.item),
            },
        ]


        times_runned = 0

        while(times_runned<5):

            response_from_openai_guess_check = self.api_call_openai("gpt-4o-mini-2024-07-18",  prompt_to_openai_for_guess_correct(self.game_type, question, self.item), 0.2, 30)
            if response_from_openai_guess_check is None:
                print("Response is None! Retrying")
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_guess_check = "N/A"
                    print("Response from OpenAI is None! Breaking the loop")
                    break
                continue
                
            if response_from_openai_guess_check.startswith("Response:") or response_from_openai_guess_check.startswith("***Response:***") or response_from_openai_guess_check.startswith("**Response:**"):
                response_from_openai_guess_check = response_from_openai_guess_check.split(":")[1].strip().lower().strip('"').strip()
                # Check if it is omly yes or no
                if response_from_openai_guess_check == "yes" or response_from_openai_guess_check == "no":
                    break
                else:
                    times_runned += 1
                    if (times_runned == 5):
                        response_from_openai_guess_check = "N/A"
                        print("Response from OpenAI is not containing only yes or no! Breaking the loop")
                        break
                    continue

            if not response_from_openai_guess_check.startswith("Response:") or not response_from_openai_guess_check.startswith("***Response:***") or not response_from_openai_guess_check.startswith("**Response:**"):
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_guess_check = "N/A"
                    print("Response from OpenAI is not in starting with response! Breaking the loop")
                    break
                continue
        
        # Giveup check
        times_runned = 0

        while(times_runned<5):

            response_from_openai_check_giveup = self.api_call_openai("gpt-4o-mini-2024-07-18",  prompt_to_openai_for_giveup_check(self.game_type, question), 0.2, 30)
            if response_from_openai_check_giveup is None:
                print("Response is None! Retrying")
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_check_giveup = "N/A"
                    print("Response from OpenAI is None! Breaking the loop")
                    break
                continue
                
            if response_from_openai_check_giveup.startswith("Response:") or response_from_openai_check_giveup.startswith("***Response:***") or response_from_openai_check_giveup.startswith("**Response:**"):
                response_from_openai_check_giveup = response_from_openai_check_giveup.split(":")[1].strip().lower().strip('"').strip()
                # Check if it is omly yes or no
                if response_from_openai_check_giveup == "yes" or response_from_openai_check_giveup == "no":
                    break
                else:
                    times_runned += 1
                    if (times_runned == 5):
                        response_from_openai_check_giveup = "N/A"
                        print("Response from OpenAI is not containing only yes or no! Breaking the loop")
                        break
                    continue

            if not response_from_openai_check_giveup.startswith("Response:") or not response_from_openai_check_giveup.startswith("***Response:***") or not response_from_openai_check_giveup.startswith("**Response:**"):
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_check_giveup = "N/A"
                    print("Response from OpenAI is not in starting with response! Breaking the loop")
                    break
                continue

        # Check if the response_from_openai_check_giveup is yes
        if response_from_openai_check_giveup == "yes":
            return {"role": "user", "content": "You have given up!"}
        
        if any(
            [
                re.search(rf"(?:^|\W){i.strip().lower()}(?:$|\W)", question.lower())
                for i in self.item.lower().split("|")
            ]
        ) or response_from_openai_guess_check == "yes":
            return {"role": "user", "content": "Bingo!"}


        if isinstance(self.guesser_model, str) and "gpt" in self.guesser_model:
            response = openai.ChatCompletion.create(
                model=self.guesser_model,
                messages=user_messages,
                max_tokens=30,
                n=1,
                stop=None,
                temperature=0.2,
            )

        
            return response.choices[0].message.to_dict()
        
        elif isinstance(self.guesser_model, str) and "gemini" in self.guesser_model:

            prompt = ""
            for item in user_messages:
                if item["role"].upper() == "USER":
                    prompt += f"USER: {item['content']}\n"
                elif item["role"].upper() == "ASSISTANT":
                    prompt += f"ASSISTANT: {item['content']}\n"

            prompt+= "ASSISTANT:"

            times_runned = 0
            while(times_runned<5):

                response = self.api_call_gemini(self.guesser_model, prompt, 0.2, 30)
                if response is None:
                    print("Response is None! Retrying")
                    times_runned += 1
                    if (times_runned == 5):
                        response = "N/A"
                        print("Response from Gemini is None! Breaking the loop")
                        break
                    continue
                break

            return {"role": "user", "content": response.strip()}
        
        elif not isinstance(self.guesser_model, str):

            prompt = ""
            for item in user_messages:
                if item["role"].upper() == "USER":
                    prompt += f"USER: {item['content']}\n"
                elif item["role"].upper() == "ASSISTANT":
                    prompt += f"ASSISTANT: {item['content']}\n"

            prompt+= "ASSISTANT:"

            # Ensure tokenizer has a pad token
            if self.guesser_tokenizer.pad_token is None:
                self.guesser_tokenizer.pad_token = self.guesser_tokenizer.eos_token

            # Encode with attention mask
            encoded_input = self.guesser_tokenizer(
                prompt, return_tensors="pt", padding=True
            )

            input_ids = encoded_input["input_ids"].to(self.guesser_model.device)
            attention_mask = encoded_input["attention_mask"].to(self.guesser_model.device)

            with torch.no_grad():
                gen = self.guesser_model.generate(
                    input_ids=input_ids,
                    attention_mask=attention_mask,
                    **self.guesser_kargs,
                )
                gen_str = (
                    self.guesser_tokenizer.decode(gen[0][input_ids[0].shape[0] :])
                    .split("</s>")[0]
                    .split("USER")[0]
                    .split("<|eot_id|>")[0]
                    .lstrip()
                    .strip() 
                )

                # Check if there is a </s> in the gen_str, t there is then split it and take the previous part
                if "</s>" in gen_str:
                    gen_str = gen_str.split("</s>")[0].strip()


                # Replace \n with ". "
                gen_str = gen_str.replace("\n", ". ")

                # Check if there is any "ASSISTANT" in the gen_str, if yes then split it
                if "ASSISTANT" in gen_str:
                    gen_str = gen_str.split("ASSISTANT")[0].strip()


            # if "llama" in self.guesser_model:
            #     gen_str = self.api_call_llama(prompt, 0.2, 30)
            
            # elif "mistral" in self.guesser_model:
            #     gen_str = self.api_call_mistral(prompt, 30, 0.2)

                return {
                    "role": "user",
                    "content": gen_str,
                }


    def reset(self):
        # Initialize the conversation
        self.guesser_messages = [
            {
                "role": "user",
                "content": self.first_user_utterance,
            }
        ]


class Q20GameCelebrity(Q20Game):
    def __init__(self, item: str, background: str, language, game_type, **kwargs) -> None:
        super().__init__(item, **kwargs, language=language, game_type=game_type)
        self.background = background
        self.first_user_utterance = self.prompts['guesser_prompt']

    @retry(
        (
            openai.error.Timeout,
            requests.exceptions.ReadTimeout,
            openai.error.ServiceUnavailableError,
            openai.error.RateLimitError,
            openai.error.APIError,
            openai.error.APIConnectionError,
        ),
        tries=5,
        delay=0.5,
        backoff=0.5,
        max_delay=2,
        logger=LOGGER,
    )
    def answerer(self, question):
        
        if isinstance(self.guesser_model, str) and "gpt" in self.guesser_model:
            openai.api_base = "https://api.openai.com/v1"

        if len(self.background) < 10:
            user_messages = [
                {
                    "role": "user",
                    "content": self.prompts['answerer_system_prompt_without_background_of_celebrity_given'].format(item = self.item),
                },
                {
                    "role": "user",
                    "content": self.prompts['answerer_prompt_without_background_of_celebrity_given'].format(item = self.item, question = question),
                },
            ]
        else:
            user_messages = [
                {
                    "role": "user",
                    "content": self.prompts['answerer_system_prompt_with_background_of_celebrity_given'].format(item = self.item, background = self.background),
                },
                {
                    "role": "user",
                    "content": self.prompts['answerer_prompt_with_background_of_celebrity_given'].format(item = self.item, question = question),
                },
            ]

        times_runned = 0

        while(times_runned<5):

            response_from_openai_guess_check = self.api_call_openai("gpt-4o-mini-2024-07-18",  prompt_to_openai_for_guess_correct(self.game_type, question, self.item), 0.2, 30)
            if response_from_openai_guess_check is None:
                print("Response is None! Retrying")
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_guess_check = "N/A"
                    print("Response from OpenAI is None! Breaking the loop")
                    break
                continue
            
            if response_from_openai_guess_check.startswith("Response:") or response_from_openai_guess_check.startswith("***Response:***") or response_from_openai_guess_check.startswith("**Response:**"):
                response_from_openai_guess_check = response_from_openai_guess_check.split(":")[1].strip().lower().strip('"').strip()
                # Check if it is omly yes or no
                if response_from_openai_guess_check == "yes" or response_from_openai_guess_check == "no":
                    break
                else:
                    times_runned += 1
                    print(f"Response from OpenAI is not containing only yes or no! {response_from_openai_guess_check}")
                    if (times_runned == 5):
                        response_from_openai_guess_check = "N/A"
                        print("Response from OpenAI is not containing only yes or no! Breaking the loop")
                        break
                    continue
            
            if not response_from_openai_guess_check.startswith("Response:") or not response_from_openai_guess_check.startswith("***Response:***") or not response_from_openai_guess_check.startswith("**Response:**"):
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_guess_check = "N/A"
                    print("Response from OpenAI is not in starting with response! Breaking the loop")
                    break
                continue

        # Giveup check
        times_runned = 0

        while(times_runned<5):
            
            response_from_openai_check_giveup = self.api_call_openai("gpt-4o-mini-2024-07-18",  prompt_to_openai_for_giveup_check(self.game_type, question), 0.2, 30)
            if response_from_openai_check_giveup is None:
                print("Response is None! Retrying")
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_check_giveup = "N/A"
                    print("Response from OpenAI is None! Breaking the loop")
                    break
                continue
                
            if response_from_openai_check_giveup.startswith("Response:") or response_from_openai_check_giveup.startswith("***Response:***") or response_from_openai_check_giveup.startswith("**Response:**"):
                response_from_openai_check_giveup = response_from_openai_check_giveup.split(":")[1].strip().lower().strip('"').strip()
                # Check if it is omly yes or no
                if response_from_openai_check_giveup == "yes" or response_from_openai_check_giveup == "no":
                    break
                else:
                    times_runned += 1
                    if (times_runned == 5):
                        response_from_openai_check_giveup = "N/A"
                        print("Response from OpenAI is not containing only yes or no! Breaking the loop")
                        break
                    continue

            if not response_from_openai_check_giveup.startswith("Response:") or not response_from_openai_check_giveup.startswith("***Response:***") or not response_from_openai_check_giveup.startswith("**Response:**"):
                times_runned += 1
                if (times_runned == 5):
                    response_from_openai_check_giveup = "N/A"
                    print("Response from OpenAI is not in starting with response! Breaking the loop")
                    break
                continue


        # Check if the response_from_openai_check_giveup is yes
        if response_from_openai_check_giveup == "yes":
            return {"role": "user", "content": "You have given up!"}
        
        if any(
            [
                re.search(rf"(?:^|\W){i.strip().lower()}(?:$|\W)", question.lower())
                for i in self.item.lower().split("|")
            ]
        ) or response_from_openai_guess_check == "yes":
            return {"role": "user", "content": "Bingo!"}
        
        if isinstance(self.guesser_model, str) and "gpt" in self.guesser_model:
            response = openai.ChatCompletion.create(
                model=self.guesser_model,
                messages=user_messages,
                max_tokens=30,
                n=1,
                stop=None,
                temperature=0.2,
            )

            return response.choices[0].message.to_dict()
        
        elif isinstance(self.guesser_model, str) and "gemini" in self.guesser_model:

            prompt = ""
            for item in user_messages:
                if item["role"].upper() == "USER":
                    prompt += f"USER: {item['content']}\n"
                elif item["role"].upper() == "ASSISTANT":
                    prompt += f"ASSISTANT: {item['content']}\n"

            prompt+= "ASSISTANT:"

            times_runned = 0
            while(times_runned<5):

                response = self.api_call_gemini(self.guesser_model, prompt, 0.2, 30)
                if response is None:
                    print("Response is None! Retrying")
                    times_runned += 1
                    if (times_runned == 5):
                        response = "N/A"
                        print("Response from Gemini is None! Breaking the loop")
                        break
                    continue
                break

            return {"role": "user", "content": response.strip()}
        
        elif not isinstance(self.guesser_model, str):

            prompt = ""
            for item in user_messages:
                if item["role"].upper() == "USER":
                    prompt += f"USER: {item['content']}\n"
                elif item["role"].upper() == "ASSISTANT":
                    prompt += f"ASSISTANT: {item['content']}\n"

            prompt+= "ASSISTANT:"

            # Ensure tokenizer has a pad token
            if self.guesser_tokenizer.pad_token is None:
                self.guesser_tokenizer.pad_token = self.guesser_tokenizer.eos_token

            # Encode with attention mask
            encoded_input = self.guesser_tokenizer(
                prompt, return_tensors="pt", padding=True
            )

            input_ids = encoded_input["input_ids"].to(self.guesser_model.device)
            attention_mask = encoded_input["attention_mask"].to(self.guesser_model.device)

            with torch.no_grad():
                gen = self.guesser_model.generate(
                    input_ids=input_ids,
                    attention_mask=attention_mask,
                    **self.guesser_kargs,
                )
                gen_str = (
                    self.guesser_tokenizer.decode(gen[0][input_ids[0].shape[0] :])
                    .split("</s>")[0]
                    .split("USER")[0]
                    .split("<|eot_id|>")[0]
                    .lstrip()
                    .strip() 
                )


                # Check if there is a </s> in the gen_str, t there is then split it and take the previous part
                if "</s>" in gen_str:
                    gen_str = gen_str.split("</s>")[0].strip()


                # Replace \n with ". "
                gen_str = gen_str.replace("\n", ". ")

                # Check if there is any "ASSISTANT" in the gen_str, if yes then split it
                if "ASSISTANT" in gen_str:
                    gen_str = gen_str.split("ASSISTANT")[0].strip()


            # if "llama" in self.guesser_model:
            #     gen_str = self.api_call_llama(prompt, 0.2, 30)

            # elif "mistral" in self.guesser_model:
            #     gen_str = self.api_call_mistral(prompt, 30, 0.2)

                return {
                    "role": "user",
                    "content": gen_str,
                }


    def reset(self):
        # Initialize the conversation
        self.guesser_messages = [
            {
                "role": "user",
                "content": self.first_user_utterance,
            }
        ]