from typing import Dict, List
import re
import librosa


def clear_think(raw_content: str) -> str:
    assert isinstance(raw_content, str)
    cleaned_content = re.sub(r'<think>.*?</think>', '', raw_content, count= 1, flags=re.DOTALL)
    return cleaned_content

def criteria2str(criteria: list[str]) -> str:
    if criteria is None:
        criteria = []
    assert isinstance(criteria, list)

    candidate_criteria_text = ""
    c = []
    for idx, candidate_criterion in enumerate(criteria, start= 1):
        if isinstance(candidate_criterion, dict):
            c_text = candidate_criterion['content']
        elif isinstance(candidate_criterion, str):
            c_text = candidate_criterion
        c.append(f"<criteria {idx}>{c_text}</criteria {idx}>")
    candidate_criteria_text = "\n".join(c)
    return candidate_criteria_text

def single_criteria2str(criteria) -> str:
    return f"<criteria 1>{criteria['content']}</criteria 1>"

def extract_criteria_from_text(text):

    pattern = r"<criteria\s*(\d+)>(.*?)</criteria\s*\1>"
    matches = re.findall(pattern, text, flags=re.DOTALL)

    criteria_list = []
    for num, content in matches:
        criteria_list.append(content.strip())

    return criteria_list

def extract_criteria_n_from_text(text):

    _criteria_list = text.split("\n")
    criteria_list = []
    for c in _criteria_list:
        if c == "":
            continue
        criteria_list.append(c)

    return criteria_list

def extract_judge_pair(text: str):

    results = {}
    match_a = re.search(r"<Judge A>(.*?)</Judge A>", text, re.DOTALL)
    if match_a:
        block_a = match_a.group(1)
        judge_a_list = extract_criteria_from_text(block_a.strip())
        results["judge_a_list"] = judge_a_list

    match_b = re.search(r"<Judge B>(.*?)</Judge B>", text, re.DOTALL)
    if match_b:
        block_b = match_b.group(1)
        judge_b_list = extract_criteria_from_text(block_b.strip())
        results["judge_b_list"] = judge_b_list
    
    return results

def extract_judge_only(text: str):

    results = {}
    match_a = re.search(r"<Judge A>(.*?)</Judge A>", text, re.DOTALL)
    if match_a:
        block_a = match_a.group(1)
        results["judge_a"] = block_a
    
    match_b = re.search(r"<Judge B>(.*?)</Judge B>", text, re.DOTALL)
    if match_b:
        block_b = match_b.group(1)
        results["judge_b"] = block_b
    
    return results

def extract_omni_answer(response, answer):
    
    assert type(answer) == int
    assert answer in [0, 1, 2]

    pred = None
    pattern = r'\[{1,2}([abcABC])\]{1,2}'
    matches = re.findall(pattern, response)
    if len(matches) > 0:
        pred = matches[-1]
    else:
        pred = None
    if isinstance(pred, str):
        pred = pred.upper()
    
    correct = False
    if pred == 'A' and answer == 0:
        correct = True
    elif pred == 'B' and answer == 1:
        correct = True
    elif pred == 'C' and answer == 2:
        correct = True
    
    return correct, pred

def parse_direct_judge_output(output: str) -> Dict:

    results = {"criteria": [], "verdict": None}

    think_block = re.search(r"<think>(.*?)</think>", output, re.S)
    if think_block:
        think_content = think_block.group(1)

        criteria_matches = re.findall(r"<criteria\s*\d+>(.*?)</criteria\s*\d+>", think_content, re.S)
        for crit in criteria_matches:
            crit_before_A = crit.split("<Judge A>")[0].strip()
            # 提取 Judge A
            judge_A_match = re.search(r"<Judge A>(.*?)</Judge A>", crit, re.S)
            judge_A = judge_A_match.group(1).strip() if judge_A_match else None

            # 提取 Judge B
            judge_B_match = re.search(r"<Judge B>(.*?)</Judge B>", crit, re.S)
            judge_B = judge_B_match.group(1).strip() if judge_B_match else None
            

            criterion_dict={
                "criterion": crit_before_A,
                "judge_A": judge_A,
                "judge_B": judge_B,
            }

            if criterion_dict["criterion"] is None or criterion_dict['judge_A'] is None or criterion_dict['judge_B'] is None:
                raise Exception(f"Parse Criterion failed. Text:[{crit}]. will skip it.")
            
            results["criteria"].append(criterion_dict)

    else:
        raise Exception(f"can't parse think part from output:[{output}]")

    pred = None
    pattern = r'\[{1,2}([abcABC])\]{1,2}'
    matches = re.findall(pattern, output)
    if len(matches) > 0:
        pred = matches[-1]
    else:
        pred = None
    if isinstance(pred, str):
        pred = pred.upper()
    results["verdict"] = pred

    return results

def is_longer_than_100s(path: str) -> bool:

    duration = librosa.get_duration(path=path)
    return duration > 100
