"""
This module contains miscellaneous utilities for the different architectures.
"""

import torch
import torch.nn as nn


class LayerNormalization(nn.Module):
    """
    Implementation of the Layer Normalization layer.
    """

    def __init__(self, features: int, eps: float = 1e-6):
        """
        ___init__ method for the LayerNormalization class.

        Normalize the activations of the previous layer for each feature.

        Args:
                features (int): Number of features to normalize.
                eps (float): Epsilon value to avoid division by
                                         zero and for numerical stability.
        """
        super().__init__()
        self.eps = eps

        # Learnable parameters
        self.alpha = nn.Parameter(torch.ones(features))
        self.bias = nn.Parameter(torch.zeros(features))

    def forward(self, x):
        mean = x.mean(dim=-1, keepdim=True)
        std = x.std(dim=-1, keepdim=True)

        normed = (x - mean) / (std + self.eps)
        out = self.alpha * normed + self.bias
        return out
