import os
import subprocess

from setuptools import find_packages, setup
from torch.utils.cpp_extension import BuildExtension, CUDAExtension

def get_git_commit_number():
    if not os.path.exists('.git'):
        return '0000000'

    cmd_out = subprocess.run(['git', 'rev-parse', 'HEAD'], stdout=subprocess.PIPE)
    git_commit_number = cmd_out.stdout.decode('utf-8')[:7]
    return git_commit_number


def make_cuda_ext(name, module, sources):
    cuda_ext = CUDAExtension(
        name='%s.%s' % (module, name),
        sources=[os.path.join(*module.split('.'), src) for src in sources]
    )
    return cuda_ext

def write_version_to_file(version, target_file):
    with open(target_file, 'w') as f:
        print('__version__ = "%s"' % version, file=f)

if __name__ == '__main__':
    version = '0.0.0+%s' % get_git_commit_number()
    write_version_to_file(version, 'edl/version.py')
    make_cuda = False
    if make_cuda:

        setup(
            name='edl',
            version=version,
            description='OpenEDL is a codebase for evidential deep learning.',
            license='Apache License 2.0',
            packages=find_packages(),
            cmdclass={
                'build_ext': BuildExtension,
            },
            ext_modules=[
                make_cuda_ext(
                    name='dissonance_cuda',
                    module='edl.ops.dissonance',
                    sources=[
                        'src/edl.cpp',
                        'src/edl_api.cpp',
                        'src/edl_kernel.cu',
                    ]
                ),
            ],
        )
    else:
        setup(
            name='edl',
            version=version,
            description='EDL is a codebase for evidential deep learning.',
            author='***',
            license='Apache License 2.0',
            packages=find_packages(),
            cmdclass={
                'build_ext': BuildExtension,
            }
        )
