import json
import pandas as pd
import os
import argparse
from tabulate import tabulate

def tsv_to_markdown(path):
    # Read the TSV file into a pandas DataFrame
    df = pd.read_csv(path, sep='\t')
    
    # Generate a Markdown formatted table
    markdown_table = tabulate(df.head(3), headers='keys', tablefmt='pipe', showindex=False)
    
    return markdown_table

def convert_data_to_jsonl_and_tsv(data_path, input_prefix_path, output_path, mode):
    """
    Convert data from a specified file to JSONL format and save the 'table_for_pd' as TSV files.

    Parameters:
    - data_path: String path to the JSON file containing the data to be converted.
    - input_prefix_path: String path prefix where TSV files will be stored.
    - output_path: String path for the output JSONL file.
    """
    # Load data from the specified JSON file
    with open(data_path, 'r') as data_file:
        data = json.load(data_file)
    
    os.makedirs(os.path.join(input_prefix_path, mode), exist_ok=True)
    
    with open(output_path, 'w') as jsonl_file:
        for key, value in data.items():
            # Write each dictionary entry as a JSON object in a new line
            
            # Save the 'table_for_pd' as a TSV file
            table_df = pd.DataFrame(value['table_for_pd'])
            value['question_id'] = key
            tsv_path = os.path.join(input_prefix_path, f"{mode}/{key}.tsv")
            table_df.to_csv(tsv_path, sep='\t', index=False)
            print(f"Saved TSV for key {key} at {tsv_path}")
            value['data_path'] = tsv_path
            
            data_overview = tsv_to_markdown(tsv_path)
            value['data_overview'] = data_overview
            
            jsonl_file.write(json.dumps(value, ensure_ascii=False) + '\n')

def main():
    parser = argparse.ArgumentParser(description="Convert data to JSONL and save 'table_for_pd' as TSV files.")
    parser.add_argument('--data_path', type=str, required=True, help="Path to the JSON file containing the data.")
    parser.add_argument('--input_prefix_path', type=str, required=True, help="Path prefix for input and TSV storage.")
    parser.add_argument('--mode', type=str, required=True, help="Path prefix for input and TSV storage.")
    parser.add_argument('--output_path', type=str, required=True, help="Output path for the JSONL file.")
    args = parser.parse_args()
    
    convert_data_to_jsonl_and_tsv(args.data_path, args.input_prefix_path, args.output_path, args.mode)

if __name__ == "__main__":
    main()
