#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os  
import sys
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))  
import argparse  
import json  
from openai import AzureOpenAI  
from tqdm import tqdm  
import pdb  
import re  
import torch
  
# Update the import statements to use absolute imports  
from concurrent.futures import ThreadPoolExecutor, as_completed  
from model_api_call import get_chat_response_azure  
from functools import partial  
import sqlite3
import pandas as pd
from transformers import AutoTokenizer
from vllm import LLM, SamplingParams
import gc
from tabulate import tabulate
from code_utils import extract_code_plan, extract_code


def process_batch_data(data, batch_size):
    # Assuming process_batch_data is a custom function to split data into batches
    return [data[i:i + batch_size] for i in range(0, len(data), batch_size)]


def wrap_up_wikitq_ins_prompt(data, prompt_template, prompt_sug):
    question = data.get('question', '')
    data_path = data.get('data_path', '')
    data_overview = data.get('data_overview', '')
    
    prompt = prompt_template.replace('[[question]]', question)
    prompt = prompt.replace('[[data_path]]', data_path)
    
    prompt = prompt.replace('[[data_overview]]', data_overview)
    prompt = prompt.replace('[[successful_plan_suggestions]]', prompt_sug)
    data['rag_final_cot_prompt'] = prompt
    # pdb.set_trace()
    return data

def wrap_up_wikitq_mg_prompt(data, prompt_template):
    question = data.get('question', '')
    data_path = data.get('data_path', '')
    data_overview = data.get('data_overview', '')
    success_sug = data.get('rag_mg', '')
    if not success_sug: success_sug = 'You did well job, follow your thinking.'
    
    prompt = prompt_template.replace('[[question]]', question)
    prompt = prompt.replace('[[data_path]]', data_path)
    
    prompt = prompt.replace('[[data_overview]]', data_overview)
    prompt = prompt.replace('[[successful_plan_suggestions]]', success_sug)
    data['rag_final_cot_prompt'] = prompt
    # pdb.set_trace()
    return data


def wrap_up_rag_few_shot_prompt(data):
    few_shot_prompt = data.get('rag_few_shot_prompt')
    question = data.get('question', '')
    data_title = ''
    data_overview= data.get('data_overview', '')
    if not data_title: data_title = ''
    
    prompt = few_shot_prompt.replace('[[question]]', question)
    
    prompt = prompt.replace('[[data_overview]]', data_overview)
    prompt = prompt.replace('[[data_title]]', data_title)
    data['rag_final_cot_prompt'] = prompt
    # pdb.set_trace()
    return data

def load_and_infer_from_jsonl_dataset(prompt_path, result_path, model_name, prompt_template, temperature=0.0, max_tokens=60, top_p=1, frequency_penalty=0, presence_penalty=0, stop=None, max_retries=10, batch_size=20, min_batch_size=4, dataset_name="wikitq", max_model_len=16000):
    """
    Load prompts from a .jsonl file, get responses using Azure OpenAI in parallel, and save the results in another .jsonl file with progress monitoring.
    This function keeps all original items from the input JSON lines and adds the response, along with indexing each processed line for better tracking.
    """
    prompt_sketch, prompt_sug = prompt_template
    with open(prompt_path, 'r') as f:
        dataset = [json.loads(line) for line in f]
        if dataset_name == "wikitq":
            dataset = [wrap_up_wikitq_ins_prompt(data, prompt_sketch, prompt_sug) for data in dataset]
        # for few shot:
            # dataset = [wrap_up_rag_few_shot_prompt(data) for data in dataset]
        if not dataset[0].get('require_opt', ''): 
            dataset_wrong = dataset
            dataset_right = []
        else:
            dataset_wrong = [data for data in dataset if data['require_opt'] == 'true']  
            dataset_right = [data for data in dataset if data['require_opt'] == 'false']  
        print('there are {} cases needed to be optimized!'.format(len(dataset_wrong)))
        # assert len(dataset) == len(dataset_wrong) + len(dataset_right)
        batch_data = process_batch_data(dataset_wrong, batch_size=batch_size)
        
    llm = LLM(model=model_name, max_model_len=max_model_len, gpu_memory_utilization=0.9, trust_remote_code=True, tensor_parallel_size=1)  # Use the model's max input len
    stop_tokens = None  # should be a list
    tokenizer = AutoTokenizer.from_pretrained(model_name, trust_remote_code=True)
    
    sampling_params = SamplingParams(
        temperature=temperature,
        top_p=top_p,
        max_tokens=max_tokens,
        stop=None,
        stop_token_ids=[tokenizer.eos_token_id]
    )
    
    with open(result_path, 'w', encoding='utf-8') as f:
        for idx, batch_sample in enumerate(tqdm(batch_data)):
            print(f'Infering {idx}th batch...', flush=True)
            with torch.no_grad():
                batch_sample_prompt_list = [batch['rag_final_cot_prompt'] for batch in batch_sample]
                completions = llm.generate(batch_sample_prompt_list, sampling_params)
                for j, completion in enumerate(completions):
                    data_w = batch_data[idx][j]
                    # code = extract_code_plan("```code_plan\nStep 1:" + completion.outputs[0].text)  # or the correct field name
                    # code = extract_sqlite(completion.outputs[0].text)
                    code = extract_code("```python\n# Step 1: Import neccessary libraries\nimport pandas as pd\n" + completion.outputs[0].text) 
                    data_w['rag_final_cot'] = code
                    f.write(json.dumps(data_w, ensure_ascii=False) + '\n')
            # pdb.set_trace()
            torch.cuda.empty_cache()
            gc.collect()
        for data_r in dataset_right:  
            f.write(json.dumps(data_r, ensure_ascii=False) + '\n') 

  
def inference():  
    parser = argparse.ArgumentParser(description='Call OpenAI API with specified parameters and configurations.')  
    parser.add_argument('--deployment_name', type=str, required=True, help='Model name to use for the API call.')  
    parser.add_argument('--temperature', type=float, default=0.0, help='Temperature for the response. Default is 0.0.')  
    parser.add_argument('--max_tokens', type=int, default=60, help='Maximum number of tokens to generate. Default is 60.')  
    parser.add_argument('--top_p', type=float, default=1, help='Top P value. Default is 1.')  
    parser.add_argument('--frequency_penalty', type=float, default=0, help='Frequency penalty. Default is 0.')  
    parser.add_argument('--presence_penalty', type=float, default=0, help='Presence penalty. Default is 0.')  
    parser.add_argument('--stop', nargs='*', help='Stop sequence(s). Multiple values are allowed.')  
    parser.add_argument('--prompt_path', type=str, required=True, help='Path to the input .jsonl file containing prompts.')  
    parser.add_argument('--result_path', type=str, required=True, help='Path where the output .jsonl file with results will be saved.')  
    parser.add_argument('--batch_size', type=int, required=False, help='if your API could be run in parallel')
    parser.add_argument('--dataset', type=str, required=True, help='dataset name')
    parser.add_argument('--max_model_len', type=int, required=False, help='max model length')
    parser.add_argument('--which_gpu', type=str, required=False, default='0', help='No.gpu')
      
    args = parser.parse_args()  
    os.environ["CUDA_VISIBLE_DEVICES"] = args.which_gpu
      
    args = parser.parse_args()  
    
    if args.dataset == "wikitq":
        from prompts.wikitq_prompt_combine.prompt.wikitq_all import rag_phi_al_cot, pos_suggestions_v1, pos_suggestions_v2, pos_suggestions_v3, pos_suggestions_v4, pos_suggestions_v5, rag_phi_final_cot, rag_llama_al_v4, rag_llama_final_cot, pos_suggestions_v7, neg_suggestions_v2, final_code_phi
        from prompts.wikitq_prompt_combine.prompt.wikitq_backup import rag_phi_final_cot
        
    prompt_template = rag_phi_al_cot, pos_suggestions_v7

        
    
    load_and_infer_from_jsonl_dataset(args.prompt_path, args.result_path, model_name=args.deployment_name, prompt_template=prompt_template, temperature=args.temperature, max_tokens=args.max_tokens,  
                                top_p=args.top_p, stop=args.stop, batch_size=args.batch_size, dataset_name=args.dataset, max_model_len=args.max_model_len)
 
  
if __name__ == "__main__":  
    inference()  
