import matplotlib.pyplot as plt
import pandas as pd
import numpy as np


plt.style.use("paper.mplstyle")

PT = 1.0 / 72.27  # 72.27 points to an inch.
GOLDEN_RATIO = (1 + 5**0.5) / 2
IMAGE_WIDTH = 239.39438 * PT

df = pd.read_csv("data/trueskill_results.csv")
print(df)

fig, axs = plt.subplots(
    1, 2, figsize=(IMAGE_WIDTH, IMAGE_WIDTH / GOLDEN_RATIO), width_ratios=[1, 3.5]
)


measures = ["mcp", "rapl:cores", "rapl:pkg"]
x = np.arange(len(measures))
width = 0.25
colors = ["#8A378C", "#CF063A", "#dd640c"]

lenend_y_coord = [0.95, 0.82, 0.69]
methods_names = ["NumPy discrete", "NumPy continuous", "cLUT"]

for i, (method, color) in enumerate(
    zip(["baseline_discrete", "baseline_continuous", "clut_lut_c_wrapper"], colors)
):
    means = [
        df.query("method == @method")[f"{measure}_mean"].values[0]
        for measure in measures
    ]
    stds = [
        df.query("method == @method")[f"{measure}_std"].values[0]
        for measure in measures
    ]

    axs[1].bar(
        x + i * width,
        means,
        width,
        yerr=stds,
        capsize=3,
        edgecolor="white",
        label=method,
        color=color,
        error_kw=dict(ecolor="gray", lw=1.5),
    )

    axs[1].text(
        0.95,
        lenend_y_coord[i],
        methods_names[i],
        color=color,
        transform=axs[1].transAxes,
        ha="right",
        va="top",
        fontsize=8,
    )

axs[1].set_xticks(x + width / 2)
axs[1].set_xticklabels(measures, rotation=30)
axs[1].set_ylabel("Energy [J]")

sampling_x = np.arange(1)
for i, (method, color) in enumerate(
    zip(["baseline_discrete", "baseline_continuous", "clut_lut_c_wrapper"], colors)
):
    mean = df.query("method == @method")["sampling_runtimes_mean"].values[0]
    std = df.query("method == @method")["sampling_runtimes_std"].values[0]

    axs[0].bar(
        sampling_x + i * width,
        [mean],
        width,
        yerr=[std],
        capsize=2,
        edgecolor="white",
        label=method,
        color=color,
        error_kw=dict(ecolor="gray", lw=0.5, capthick=1),
    )

axs[0].set_xticks(sampling_x + width / 2)
axs[0].set_xticklabels(["wall-time"], rotation=30)
axs[0].set_ylabel("Time [s]")


plt.tight_layout(pad=0.1)
plt.tight_layout()
plt.savefig(
    "figures/pdf_trueskill_speed_energy.pdf", format="pdf", dpi=600, bbox_inches="tight"
)
plt.show()
