#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <time.h>
#include <math.h>

#include "numpy_sampling.h"

double* get_cdf(const double *p, size_t len) {
    
    double *cdf = malloc(len * sizeof(double));

    cdf[0] = p[0];
    for (size_t i = 1; i < len; ++i) {
        cdf[i] = cdf[i - 1] + p[i];
    }
    return cdf;
}

double* normalize(const double *cdf, size_t len) {
    
    double *normalized = malloc(len * sizeof(double));

    double total = cdf[len - 1];
    for (size_t i = 0; i < len; ++i) {
        normalized[i] = cdf[i] / total;
    }
    return normalized;
}

double sample_uniform() {
    double r = (double) rand() / RAND_MAX;
    return r;
}

int searchsorted(const double *cdf, size_t len, double value) {
    size_t left = 0;
    size_t right = len;
    while (left < right) {
        size_t mid = (left + right) / 2;
        if (value < cdf[mid]) {
            right = mid;
        } else {
            left = mid + 1;
        }
    }
    return (int)left;
}


double* build_cdf(const double *p, size_t len) {
    double *raw_cdf = get_cdf(p, len);
    double *norm_cdf = normalize(raw_cdf, len);
    free(raw_cdf);
    return norm_cdf;
}

double* generate_standard_normal_pdf(float start, float end, float step, size_t *out_len) {
    size_t len = (size_t)((end - start) / step) + 1;
    double *pdf = malloc(len * sizeof(double));

    for (size_t i = 0; i < len; ++i) {
        double x = start + i * step;
        pdf[i] = exp(-0.5 * x * x) / sqrt(2 * M_PI);
    }

    *out_len = len;
    return pdf;
}


double* generate_random_pdf(int l) {
    double *pdf = malloc(l * sizeof(double));

    double sum = 0.0;
    for (int i = 0; i < l; i++) {
        pdf[i] = (double) rand() / RAND_MAX;
        sum += pdf[i];
    }

    for (int i = 0; i < l; i++) {
        pdf[i] /= sum;
    }

    return pdf;
}


numpy_context_t *build_numpy(uint64_t *counts, size_t counts_size)
{
    /* TODO: make use of counts and counts_size instead of thise floats */
    (void)counts;
    (void)counts_size;
    float range_start = -3.0;
    float range_end = 3.0;
    float range_step = 0.01;

    numpy_context_t *ctx = malloc(sizeof(numpy_context_t));

    double *p = generate_standard_normal_pdf(
		range_start, range_end, 
		range_step, &ctx->len
	);
	ctx->cdf = build_cdf(p, ctx->len);

    free(p);
    return ctx;
}

float numpy_sampling(numpy_context_t *ctx)
{
    double uniform_sample = sample_uniform();
    int idx = searchsorted(
        ctx->cdf, ctx->len, 
        uniform_sample
    );
    float x = ctx->start + idx * ctx->step;
    return x;
}

void free_numpy(numpy_context_t *ctx)
{
    free(ctx->cdf);
    free(ctx);
}

// int main() {
//     srand((unsigned int) time(NULL));


//     size_t num_samples = 100000;

    



//     FILE *out = fopen("result.csv", "w");
//     for (size_t i = 0; i < num_samples; ++i) {
//         double x = start + indices[i] * step;
//         fprintf(out, "%.5f\n", x);
//     }

//     // free(cdf);
//     // free(uniform_samples);
//     // free(indices);

//     return 0;
// }
