/* Based on https://web.eece.maine.edu/~vweaver/projects/rapl/rapl-read.c */
/* Vince Weaver -- vincent.weaver @ maine.edu -- 11 September 2015        */
/*                                                                        */

#include "rapl.h"

#if defined(__x86_64__) && defined(__linux__)

#include <stdio.h>
#include <stddef.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <stdint.h>

#include <linux/perf_event.h>
#include <sys/syscall.h>

#define NUM_RAPL_DOMAINS 5
#define MAX_CPUS	1024
#define MAX_PACKAGES	16

#ifndef RAPL_DEBUG_PRINT
#define RAPL_DEBUG_PRINT 0
#endif

static long perf_event_open(struct perf_event_attr *hw_event, pid_t pid,
				int cpu, int group_fd, unsigned long flags)
{
	return syscall(__NR_perf_event_open, hw_event, pid, cpu, group_fd, flags);
}

char rapl_domain_names[NUM_RAPL_DOMAINS][30]= {
	"energy-cores",
	"energy-gpu",
	"energy-pkg",
	"energy-ram",
	"energy-psys",
};

static int total_cores=0,total_packages=0;
static int package_map[MAX_PACKAGES];

static int detect_packages(void) {

	char filename[BUFSIZ];
	FILE *fff;
	int package;
	int i;

	for(i=0;i<MAX_PACKAGES;i++) package_map[i]=-1;

	#if RAPL_DEBUG_PRINT
	printf("\t");
	#endif
	for(i=0;i<MAX_CPUS;i++) {
		sprintf(filename,"/sys/devices/system/cpu/cpu%d/topology/physical_package_id",i);
		fff=fopen(filename,"r");
		if (fff==NULL) break;
		fscanf(fff,"%d",&package);
		#if RAPL_DEBUG_PRINT
		printf("%d (%d)",i,package);
		if (i%8==7) printf("\n\t"); else printf(", ");
		#endif
		fclose(fff);

		if (package_map[package]==-1) {
			total_packages++;
			package_map[package]=i;
		}

	}

	#if RAPL_DEBUG_PRINT
	printf("\n");
	#endif

	total_cores=i;

	#if RAPL_DEBUG_PRINT
	printf("\tDetected %d cores in %d packages\n\n",
		total_cores,total_packages);
	#endif

	return 0;
}


static int check_paranoid(void)
{
	int paranoid_value;
	FILE *fff;

	fff=fopen("/proc/sys/kernel/perf_event_paranoid","r");
	if (fff==NULL) {
		fprintf(stderr,"Error! could not open /proc/sys/kernel/perf_event_paranoid %s\n",
			strerror(errno));

		/* We can't return a negative value as that implies no paranoia */
		return 500;
	}

	fscanf(fff,"%d",&paranoid_value);
	fclose(fff);

	return paranoid_value;

}

static struct _rapl_context {
	int fd[NUM_RAPL_DOMAINS][MAX_PACKAGES];
    double scale[NUM_RAPL_DOMAINS];
    char units[NUM_RAPL_DOMAINS][BUFSIZ];
	int config[NUM_RAPL_DOMAINS];
	int type;
} rapl_context;


int rapl_setup()
{
	FILE *fff;
	char filename[BUFSIZ];
	int i;

    detect_packages();

	#if RAPL_DEBUG_PRINT
	printf("\nTrying perf_event interface to gather results\n\n");
	#endif

	fff=fopen("/sys/bus/event_source/devices/power/type","r");
	if (fff==NULL) {
		return -1;
	}
	fscanf(fff,"%d",&rapl_context.type);
	fclose(fff);

	for(i=0;i<NUM_RAPL_DOMAINS;i++) {

		sprintf(filename,"/sys/bus/event_source/devices/power/events/%s",
			rapl_domain_names[i]);

		fff=fopen(filename,"r");

		if (fff!=NULL) {
			fscanf(fff,"event=%x",&rapl_context.config[i]);
			#if RAPL_DEBUG_PRINT
			printf("\tEvent=%s Config=%d ",rapl_domain_names[i],rapl_context.config[i]);
			#endif
			fclose(fff);
		} else {
			continue;
		}

		sprintf(filename,"/sys/bus/event_source/devices/power/events/%s.scale",
			rapl_domain_names[i]);
		fff=fopen(filename,"r");

		if (fff!=NULL) {
			fscanf(fff,"%lf",&rapl_context.scale[i]);
			#if RAPL_DEBUG_PRINT
			printf("scale=%g ",rapl_context.scale[i]);
			#endif
			fclose(fff);
		}

		sprintf(filename,"/sys/bus/event_source/devices/power/events/%s.unit",
			rapl_domain_names[i]);
		fff=fopen(filename,"r");

		if (fff!=NULL) {
			fscanf(fff,"%s",rapl_context.units[i]);
			#if RAPL_DEBUG_PRINT
			printf("units=%s ",rapl_context.units[i]);
			#endif
			fclose(fff);
		}

		#if RAPL_DEBUG_PRINT
		printf("\n");
		#endif
	}

	return 0;
}

size_t rapl_start(void)
{
	struct perf_event_attr attr;
	int i,j;
	int paranoid_value;
	size_t open_handles = 0;

	for(j=0;j<total_packages;j++) {

		for(i=0;i<NUM_RAPL_DOMAINS;i++) {

			rapl_context.fd[i][j]=-1;

			memset(&attr,0x0,sizeof(attr));
			attr.type=rapl_context.type;
			attr.config=rapl_context.config[i];
			if (rapl_context.config[i]==0) continue;

			rapl_context.fd[i][j]=perf_event_open(&attr,-1, package_map[j],-1,0);
			if (rapl_context.fd[i][j]<0) {
				if (errno==EACCES) {
					paranoid_value=check_paranoid();
					if (paranoid_value>0) {
						printf("\t/proc/sys/kernel/perf_event_paranoid is %d\n",paranoid_value);
						printf("\tThe value must be 0 or lower to read system-wide RAPL values\n");
					}

					printf("\tPermission denied; run as root or adjust paranoid value\n\n");
					return -1;
				}
				else {
					printf("\terror opening core %d config %d: %s\n\n",
						package_map[j], rapl_context.config[i], strerror(errno));
					return -1;
				}
			} else {
				open_handles++;
			}
		}
	}
    return open_handles;
}

void rapl_print_domains(FILE *fd, const char *prefix)
{
	int i, j;
	for(j=0;j<total_packages;j++) {
		for(i=0;i<NUM_RAPL_DOMAINS;i++) {
			if (rapl_context.fd[i][j]!=-1) {
				fprintf(fd, "%s-%s-%d ", prefix, rapl_domain_names[i], j);
			}
		}
	}
}

size_t rapl_end_read(rapl_dst_t *dst, size_t max_count)
{
	size_t insert_pos = 0;
	
	int i,j;
    uint64_t value;

	for(j=0;j<total_packages;j++) {
		for(i=0;i<NUM_RAPL_DOMAINS;i++) {
			if (rapl_context.fd[i][j]!=-1) {
				if (insert_pos >= max_count)
					return insert_pos;

				read(rapl_context.fd[i][j], &value, sizeof(value));
				close(rapl_context.fd[i][j]);
				dst[insert_pos++] = 
				#if REPORT_RAPL_TICKS
					value;
				#else
					(double)value*rapl_context.scale[i];
				#endif
			}
		}
	}

	return insert_pos;
}

int rapl_end_print(void)
{
    int i,j;
    uint64_t value;
	

	for(j=0;j<total_packages;j++) {
		printf("Package %d:\n",j);

		for(i=0;i<NUM_RAPL_DOMAINS;i++) {

			if (rapl_context.fd[i][j]!=-1) {
				read(rapl_context.fd[i][j], &value, sizeof(value));
				close(rapl_context.fd[i][j]);

				printf("\t%s: %" RAPL_DST_FMT " %s\n",
					rapl_domain_names[i],
					#if REPORT_RAPL_TICKS
					value,
					#else
					(double)value * rapl_context.scale[i],
					#endif
					rapl_context.units[i]);

			}

		}

	}
	printf("\n");

	return 0;
}

#else // Nothing on Windows

int rapl_setup(void)
{
	return 0;
}

void rapl_print_domains(FILE *fd, const char *prefix)
{
	(void)fd;
	(void)prefix;
}

size_t rapl_start(void)
{
	return 0;
}

size_t rapl_end_read(rapl_dst_t *dst, size_t max_count)
{
	(void)dst;
	(void)max_count;
	return 0;
}

int rapl_end_print(void)
{
	return 0;
}

#endif
