import os
import numpy as np
import torch
import matplotlib.pyplot as plt
from sklearn.metrics import accuracy_score, f1_score

def set_seed(seed):
    """
    Set random seed for reproducibility.
    
    Args:
        seed (int): Random seed.
    """
    np.random.seed(seed)
    torch.manual_seed(seed)
    if torch.cuda.is_available():
        torch.cuda.manual_seed(seed)

def save_results(results, filename):
    """
    Save experimental results to a file.
    
    Args:
        results (dict): Dictionary containing the results.
        filename (str): Name of the file to save the results.
    """
    with open(filename, 'w') as f:
        for key, value in results.items():
            f.write(f'{key}: {value}\n')

def evaluate(labels, predictions):
    """
    Evaluate the model's performance using accuracy and F1 score.
    
    Args:
        labels (list): Ground truth labels.
        predictions (list): Predicted labels.
        
    Returns:
        accuracy (float): Accuracy score.
        f1 (float): F1 score.
    """
    accuracy = accuracy_score(labels, predictions)
    f1 = f1_score(labels, predictions, average='weighted')
    return accuracy, f1

def visualize_embeddings(embeddings, labels, title):
    """
    Visualize the learned graph embeddings using t-SNE.
    
    Args:
        embeddings (numpy.ndarray): Graph embeddings.
        labels (list): Corresponding labels for each embedding.
        title (str): Title of the plot.
    """
    from sklearn.manifold import TSNE
    
    tsne = TSNE(n_components=2, random_state=42)
    embeddings_2d = tsne.fit_transform(embeddings)
    
    plt.figure(figsize=(8, 6))
    for i in range(max(labels) + 1):
        mask = labels == i
        plt.scatter(embeddings_2d[mask, 0], embeddings_2d[mask, 1], label=f'Class {i}')
    plt.legend()
    plt.title(title)
    plt.show()