"""
@author: Shuai Wang
@e-mail: ws199807@outlook.com
This module provides the dynamic version ScenarioManager implementation.
It must not be modified and is for reference only!
"""

from __future__ import print_function
import sys
import time
import carla
import numpy as np

import py_trees
import rospy
from enum import Enum

from srunner.autoagents.agent_wrapper import AgentWrapper
from srunner.scenariomanager.carla_data_provider import CarlaDataProvider
from srunner.scenariomanager.result_writer import ResultOutputProvider
from srunner.scenariomanager.timer import GameTime
from srunner.scenariomanager.watchdog import Watchdog
from srunner.AdditionTools.scenario_utils import calculate_distance_transforms
from srunner.AdditionTools.scenario_utils import calculate_distance_locations
from carla_ros_scenario_runner_types.msg import CarlaScenarioStatus
from srunner.scenariomanager.scenarioatomics.atomic_criteria import Status, CollisionTest

# # distance threshold for trigerring small scenarios in route
# distance_threshold = 15


class StepStatus(Enum):
    """
    States of an application
    """
    STOPPED = 0
    RUNNING = 1


class ScenarioManagerDynamic(object):

    """
    Dynamic version scenario manager class. This class holds all functionality
    required to initialize, trigger, update and stop a scenario.

    The user must not modify this class.

    To use the ScenarioManager:
    1. Create an object via manager = ScenarioManager()
    2. Load a scenario via manager.load_scenario()
    3. Trigger the execution of the scenario manager.run_scenario()
       This function is designed to explicitly control init, trigger, update and stop
       of the scenario
    4. Trigger a result evaluation with manager.analyze_scenario()
    5. If needed, cleanup with manager.stop_scenario()
    """

    def __init__(self, debug_mode=False, sync_mode=False, timeout=2.0):
        """
        Setups up the parameters, which will be filled at load_scenario()

        """
        self.scenario = None
        self.scenario_tree = None
        self.scenario_class = None
        self.ego_vehicles = None
        self.other_actors = None

        self._debug_mode = debug_mode
        self._agent = None
        self._sync_mode = sync_mode
        self._watchdog = None
        self._timeout = timeout

        self._running = False
        self._timestamp_last_run = 0.0
        self.scenario_duration_system = 0.0
        self.scenario_duration_game = 0.0
        self.start_system_time = None
        self.end_system_time = None

        self.scenario_list = None
        self.triggered_scenario = None

        self.running_record = []

        self.scenario_step_status_publisher = rospy.Publisher("/scenario/step", CarlaScenarioStatus,
                                                         latch=True, queue_size=1)

    def scenario_step_status_updated(self, status):
        # rospy.loginfo("Scenario step status updated to {}".format(status))
        val = CarlaScenarioStatus.STOPPED
        if status == StepStatus.RUNNING:
            val = CarlaScenarioStatus.RUNNING
        status = CarlaScenarioStatus()
        status.status = val
        self.scenario_step_status_publisher.publish(status)

    def _reset(self):
        """
        Reset all parameters
        """
        self._running = False
        self._timestamp_last_run = 0.0
        self.scenario_duration_system = 0.0
        self.scenario_duration_game = 0.0
        self.start_system_time = None
        self.end_system_time = None
        self.running_record = []
        GameTime.restart()

    def cleanup(self):
        """
        This function triggers a proper termination of a scenario
        """

        if self._watchdog is not None:
            self._watchdog.stop()
            self._watchdog = None

        # if self.scenario is not None:
        #     self.scenario.terminate()

        if self.scenario_class is not None:
            self.scenario_class.__del__()

        if self._agent is not None:
            self._agent.cleanup()
            self._agent = None

        CarlaDataProvider.cleanup()

    def load_scenario(self, scenario, agent=None):
        """
        Load a new scenario
        """
        self._reset()
        self._agent = AgentWrapper(agent) if agent else None
        if self._agent is not None:
            self._sync_mode = True
        self.scenario_class = scenario
        self.scenario = scenario.scenario
        self.scenario_tree = self.scenario.scenario_tree
        self.ego_vehicles = scenario.ego_vehicles
        self.other_actors = scenario.other_actors

        # all spawned scenarios on route
        self.scenario_list = scenario.list_scenarios
        # triggered scenario set
        self.triggered_scenario = set()

        if self._agent is not None:
            self._agent.setup_sensors(self.ego_vehicles[0], self._debug_mode)

        self._running = True
        self._init_scenarios()
        self._running = False

    def run_scenario(self):
        """
        Trigger the start of the scenario and wait for it to finish/fail
        """
        print("ScenarioManager: Running scenario {}".format(self.scenario_tree.name))
        self.start_system_time = time.time()
        start_game_time = GameTime.get_time()

        self._watchdog = Watchdog(float(self._timeout))
        self._watchdog.start()
        self._running = True

        # ego_vehicle_route = CarlaDataProvider.get_ego_vehicle_route()
        # ego_vehicle_route = [item[0] for item in ego_vehicle_route]
        # for i in ego_vehicle_route:
        #     print(i)

        """First spawn all actors on the map"""
        # self._init_scenarios()

        self.scenario_class.criteria['adv_collision'] = CollisionTest(actor=self.scenario_class.ego_vehicles[0],
                                                                      other_actor=self.scenario_list[0].other_actors[0],
                                                                      terminate_on_failure=True)

        # actors = list(CarlaDataProvider.get_actors())
        # print('ego', actors[0][1].get_speed_limit())
        # if len(actors) > 1:
        #     print('other', actors[1][1].get_speed_limit())
        # if len(actors) > 2:
        #     print('other 2', actors[2][1].get_speed_limit())

        while self._running:
            timestamp = None
            world = CarlaDataProvider.get_world()
            if world:
                snapshot = world.get_snapshot()
                if snapshot:
                    timestamp = snapshot.timestamp
            # print("scenario defination: ", self.scenario_class.sampled_scenarios_definitions)
            # print("ego transform: ", CarlaDataProvider.get_transform(self.ego_vehicles[0]))
            if timestamp:
                self._get_update(timestamp)
                # print('scenario manager tick')

        self.cleanup()

        self.end_system_time = time.time()
        end_game_time = GameTime.get_time()

        self.scenario_duration_system = self.end_system_time - \
            self.start_system_time
        self.scenario_duration_game = end_game_time - start_game_time

        if self.scenario_tree.status == py_trees.common.Status.FAILURE:
            print("ScenarioManager: Terminated due to failure")

        return self.running_record

    def _init_scenarios(self):
        # spawn background actors
        self.scenario_class.initialize_actors()
        # spawn actors for each scenario
        if self._running:
            for i in range(len(self.scenario_list)):
                self.scenario_list[i].initialize_actors()
                self.scenario_class.other_actors += self.scenario_list[i].other_actors

    def get_running_status(self):
        """
        returns:
           bool:  False if watchdog exception occured, True otherwise
        """
        return self._watchdog.get_status() if self._watchdog is not None else False

    def stop_scenario(self):
        """
        This function is used by the overall signal handler to terminate the scenario execution
        """
        self._running = False

    def analyze_scenario(self, stdout, filename, junit, json):
        """
        This function is intended to be called from outside and provide
        the final statistics about the scenario (human-readable, in form of a junit
        report, etc.)
        """

        failure = False
        timeout = False
        result = "SUCCESS"

        if self.scenario.test_criteria is None:
            print("Nothing to analyze, this scenario has no criteria")
            return True

        for criterion in self.scenario.get_criteria():
            if (not criterion.optional and
                    criterion.test_status != "SUCCESS" and
                    criterion.test_status != "ACCEPTABLE"):
                failure = True
                result = "FAILURE"
            elif criterion.test_status == "ACCEPTABLE":
                result = "ACCEPTABLE"

        if self.scenario.timeout_node.timeout and not failure:
            timeout = True
            result = "TIMEOUT"

        output = ResultOutputProvider(self, result, stdout, filename, junit, json)
        output.write()

        return failure or timeout

    def update_running_status(self):
        record, stop = self.scenario_class.get_running_status(self.running_record)

        # record['adv_actions'] = self.scenario_list[0].actions
        # record['adv_obs'] = self.scenario_list[0].current_obs
        # record['adv_action'] = self.scenario_list[0].action
        # record['adv_reward'] = self.scenario_list[0].current_reward
        # record['adv_done'] = stop
        # record['adv_cost'] = - int(record['adv_collision'] == Status.FAILURE)

        self.running_record.append(record)
        if stop:
            self._running = False

    def _get_update(self, timestamp):
        """
        This function used to trigger, update and stop scenario

        """
        """
        testing erea
        """

        """
        """
        if self._timestamp_last_run < timestamp.elapsed_seconds and self._running:
            self._timestamp_last_run = timestamp.elapsed_seconds

            self._watchdog.update()

            if self._debug_mode:
                print("\n--------- Tick ---------\n")

            # Update game time and actor information
            GameTime.on_carla_tick(timestamp)
            CarlaDataProvider.on_carla_tick()

            # Tick scenario
            # note: for the background
            self.scenario_tree.tick_once()

            for spawned_scenario in self.scenario_list:
                ego_location = CarlaDataProvider.get_location(self.ego_vehicles[0])
                cur_distance = None
                reference_location = None
                if spawned_scenario.reference_actor:
                    reference_location = CarlaDataProvider.get_location(spawned_scenario.reference_actor)
                if reference_location:
                    cur_distance = calculate_distance_locations(ego_location, reference_location)
                # print('timestamp.elapsed_seconds', timestamp.elapsed_seconds)
                # print('ego_location', ego_location)
                # print('reference_location', reference_location)
                # print('cur_distance', cur_distance)

                if cur_distance and cur_distance < spawned_scenario.trigger_distance_threshold:
                    if spawned_scenario not in self.triggered_scenario:
                        print("Trigger scenario: " + spawned_scenario.name)
                        self.triggered_scenario.add(spawned_scenario)

            # self.scenario_step_status_updated(StepStatus.STOPPED)
            for running_scenario in self.triggered_scenario.copy():
                """
                update behavior
                """
                # print("Running scenario: " + running_scenario.name)
                running_scenario.update_behavior()
                # pass
                # comment: stop the scenario in certain condition, update triggered_scenario into not triggered
                # if running_scenario.check_stop_condition():
                #     print("Scenario" + running_scenario.name + "should stop")
                #     running_scenario.remove_all_actors()
                #     self.triggered_scenario.remove(running_scenario)
                #     self._running = False
            # self.scenario_step_status_updated(StepStatus.RUNNING)

            self.update_running_status()

            # actors = list(CarlaDataProvider.get_actors())
            # print('ego', actors[0][1].get_transform())
            # # v = actors[0][1].get_velocity()
            # # print('ego', np.sqrt(v.x**2 + v.y**2))
            # if len(actors) > 1:
            #     print('other', actors[1][1].get_transform())
            #     # v = actors[1][1].get_velocity()
            #     # print('other', np.sqrt(v.x**2 + v.y**2))
            # if len(actors) > 2:
            #     print('other 2', actors[2][1].get_transform())
            #     # v = actors[2][1].get_velocity()
            #     # print('other 2', np.sqrt(v.x**2 + v.y**2))

        if self._sync_mode and self._running and self._watchdog.get_status():
            # tick other parts in scenarios
            CarlaDataProvider.get_world().tick()
