import json
import re
import os
import torch
import random
import signal
import warnings
import tiktoken
import numpy as np
from math_verify import parse, verify
from math_verify.errors import TimeoutException
from math_verify.metric import math_metric
from math_verify.parser import ExprExtractionConfig, LatexExtractionConfig

def write_json(obj, file_name):
    with open(file_name, "w") as f:
        json.dump(obj, f)

def read_json(file_name):
    with open(file_name, "r") as f:
        return json.load(f)
    
def extract_answer(text: str) -> str:
    match = re.findall(r"<answer>(.*?)</answer>", text, re.DOTALL)
    if match:
        answer = match[-1].strip()
    else:
        answer = None
    return answer

def get_number_choice(text):
    if not text:
        return "N/A"
    match = re.findall(r"answer is \((\d)\)", text)
    if match:
        return match[-1]
    else:
        match = re.findall(r"\((\d)\)", text)
        return match[-1] if match else "N/A"
    return "N/A"

def get_alphabet_choice(text, num_choice=4):
    choices = '|'.join([chr(65 + i) for i in range(num_choice)])
    if text:
        # First try to match with parentheses
        match = re.findall(f'([{choices}])\)', text)
        if not match:
            # If no match with parentheses, try without
            match = re.findall(f'([{choices}])', text)
    else:
        return "N/A"
    return match[-1] if match else "N/A"
    
def get_true_false(text):
    if not text:
        return "N/A"
    match = re.findall(r"(true|false)", text, re.IGNORECASE)
    return match[-1].lower() if match else "N/A"

def get_yes_no(text):
    if not text:
        return "N/A"
    match = re.findall(r"(yes|no)", text, re.IGNORECASE)
    return match[-1].lower() if match else "N/A"

def get_keywords(output):
    keywords = output.split("Keywords:")[-1].split(",")
    keywords = [i.strip().lower().replace(".", "") for i in keywords]
    return keywords

def get_token_count(string, encoding_name="gpt-3.5-turbo"):
    encoding = tiktoken.encoding_for_model(encoding_name)
    num_tokens = len(encoding.encode(string))
    return num_tokens

def is_math_equiv(solution_str, ground_truth) -> float:
    retval = 0.0
    try:
        string_in_last_boxed = last_boxed_only_string(solution_str)
        if string_in_last_boxed is not None:
            answer = remove_boxed(string_in_last_boxed)
            if is_equiv(answer, ground_truth):
                retval = 1.0
    except Exception as e:
        print(e)

    return retval


# string normalization from https://github.com/EleutherAI/lm-evaluation-harness/blob/master/lm_eval/tasks/hendrycks_math.py
def is_equiv(str1, str2, verbose=False):
    if str1 is None and str2 is None:
        print("WARNING: Both None")
        return True
    if str1 is None or str2 is None:
        return False

    try:
        ss1 = strip_string(str1)
        ss2 = strip_string(str2)
        if verbose:
            print(ss1, ss2)
        return ss1 == ss2
    except Exception:
        return str1 == str2


def remove_boxed(s):
    if "\\boxed " in s:
        left = "\\boxed "
        if s[: len(left)] != left:
            return "N/A"
        return s[len(left) :]

    left = "\\boxed{"

    if s[: len(left)] != left or s[-1] != "}":
        return "N/A"

    return s[len(left) : -1]


def last_boxed_only_string(string):
    idx = string.rfind("\\boxed")
    if "\\boxed " in string:
        return "\\boxed " + string.split("\\boxed ")[-1].split("$")[0]
    if idx < 0:
        idx = string.rfind("\\fbox")
        if idx < 0:
            return None

    i = idx
    right_brace_idx = None
    num_left_braces_open = 0
    while i < len(string):
        if string[i] == "{":
            num_left_braces_open += 1
        if string[i] == "}":
            num_left_braces_open -= 1
            if num_left_braces_open == 0:
                right_brace_idx = i
                break
        i += 1

    retval = None if right_brace_idx is None else string[idx : right_brace_idx + 1]

    return retval


def fix_fracs(string):
    substrs = string.split("\\frac")
    new_str = substrs[0]
    if len(substrs) > 1:
        substrs = substrs[1:]
        for substr in substrs:
            new_str += "\\frac"
            if substr[0] == "{":
                new_str += substr
            else:
                try:
                    assert len(substr) >= 2
                except:  # noqa: E722
                    return string
                a = substr[0]
                b = substr[1]
                if b != "{":
                    if len(substr) > 2:
                        post_substr = substr[2:]
                        new_str += "{" + a + "}{" + b + "}" + post_substr
                    else:
                        new_str += "{" + a + "}{" + b + "}"
                else:
                    if len(substr) > 2:
                        post_substr = substr[2:]
                        new_str += "{" + a + "}" + b + post_substr
                    else:
                        new_str += "{" + a + "}" + b
    string = new_str
    return string


def fix_a_slash_b(string):
    if len(string.split("/")) != 2:
        return string
    a = string.split("/")[0]
    b = string.split("/")[1]
    try:
        a = int(a)
        b = int(b)
        assert string == "{}/{}".format(a, b)
        new_string = "\\frac{" + str(a) + "}{" + str(b) + "}"
        return new_string
    except:  # noqa: E722
        return string


def remove_right_units(string):
    # "\\text{ " only ever occurs (at least in the val set) when describing units
    if "\\text{ " in string:
        splits = string.split("\\text{ ")
        assert len(splits) == 2
        return splits[0]
    else:
        return string


def fix_sqrt(string):
    if "\\sqrt" not in string:
        return string
    splits = string.split("\\sqrt")
    new_string = splits[0]
    for split in splits[1:]:
        if split[0] != "{":
            a = split[0]
            new_substr = "\\sqrt{" + a + "}" + split[1:]
        else:
            new_substr = "\\sqrt" + split
        new_string += new_substr
    return new_string


def strip_string(string):
    # linebreaks
    string = string.replace("\n", "")

    # remove inverse spaces
    string = string.replace("\\!", "")

    # replace \\ with \
    string = string.replace("\\\\", "\\")

    # replace tfrac and dfrac with frac
    string = string.replace("tfrac", "frac")
    string = string.replace("dfrac", "frac")

    # remove \left and \right
    string = string.replace("\\left", "")
    string = string.replace("\\right", "")

    # Remove circ (degrees)
    string = string.replace("^{\\circ}", "")
    string = string.replace("^\\circ", "")

    # remove dollar signs
    string = string.replace("\\$", "")

    # remove units (on the right)
    string = remove_right_units(string)

    # remove percentage
    string = string.replace("\\%", "")
    string = string.replace("\%", "")  # noqa: W605

    # " 0." equivalent to " ." and "{0." equivalent to "{." Alternatively, add "0" if "." is the start of the string
    string = string.replace(" .", " 0.")
    string = string.replace("{.", "{0.")
    # if empty, return empty string
    if len(string) == 0:
        return string
    if string[0] == ".":
        string = "0" + string

    # to consider: get rid of e.g. "k = " or "q = " at beginning
    if len(string.split("=")) == 2 and len(string.split("=")[0]) <= 2:
        string = string.split("=")[1]

    # fix sqrt3 --> sqrt{3}
    string = fix_sqrt(string)

    # remove spaces
    string = string.replace(" ", "")

    # \frac1b or \frac12 --> \frac{1}{b} and \frac{1}{2}, etc. Even works with \frac1{72} (but not \frac{72}1).
    # Also does a/b --> \\frac{a}{b}
    string = fix_fracs(string)

    # manually change 0.5 --> \frac{1}{2}
    if string == "0.5":
        string = "\\frac{1}{2}"

    # NOTE: X/Y changed to \frac{X}{Y} in dataset, but in simple cases fix in case the model output is X/Y
    string = fix_a_slash_b(string)

    return string


def is_math_verify_equiv(ref, pred, timeout=10):
    """
    Tests the mathematical equivalence of ref and pred with a 10-second timeout 
    and suppresses warnings. This function can also handle answer choices, 
    for example, A vs. (A).
    """

    # Handler to raise an exception when the timeout is reached
    def timeout_handler(signum, frame):
        raise TimeoutError("The mathematical equivalence check timed out.")

    # Set the signal handler and a 10-second alarm
    signal.signal(signal.SIGALRM, timeout_handler)
    signal.alarm(timeout)

    try:
        # Suppress warnings within this block
        with warnings.catch_warnings():
            warnings.simplefilter("ignore")

            ref = str(ref)
            pred = str(pred)

            if len(pred) > 200:
                return False

            # The core logic of the function is placed here
            if any([verify(parse(ref), parse(pred)),
                    verify(parse(f"\\boxed{{{ref}}}"), parse(pred)),
                    verify(parse(f"${ref}$"), parse(f"${pred}$")),
                    verify(parse(ref), parse(pred.replace("\\(", "").replace("\\)", "")))]):
                signal.alarm(0)  # Disable the alarm if the check completes in time
                return True

    except (TimeoutError, Exception):
        # The function will return False if it times out or if any other error occurs
        return False
    finally:
        # Always disable the alarm when the function exits
        signal.alarm(0)

    return False
    
def has_consensus(predictions):
    ref = predictions[0]
    for exp in predictions[1:]:
        if not ref == exp:
            return False
    return True
    
def verl_math_equiv(model_output: str, ground_truth: str) -> bool:
    verify_func = math_metric(
        gold_extraction_target=(LatexExtractionConfig(),),
        pred_extraction_target=(ExprExtractionConfig(), LatexExtractionConfig()),
    )
    ret_score = 0.0
    # Wrap the ground truth in \boxed{} format for verification
    ground_truth_boxed = "\\boxed{" + str(ground_truth) + "}"
    try:
        ret_score, _ = verify_func([ground_truth_boxed], [model_output])
    except Exception:
        pass
    except TimeoutException:
        return 0.0
    return ret_score