"""
@author: Haixu Wu
"""
import torch.nn.functional as F
import torch.nn as nn
import torch
import numpy as np
import math
from ..layers.quad_prod import ProductLayer


################################################################
# Multiscale modules 2D
################################################################
class DoubleConv(nn.Module):
    """(convolution => [BN] => ReLU) * 2"""

    def __init__(self, in_channels, out_channels, mid_channels=None):
        super().__init__()
        if not mid_channels:
            mid_channels = out_channels
        self.double_conv = nn.Sequential(
            nn.Conv2d(in_channels, mid_channels, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(mid_channels),
            nn.ReLU(inplace=True),
            nn.Conv2d(mid_channels, out_channels, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(out_channels),
            nn.ReLU(inplace=True)
        )

    def forward(self, x):
        return self.double_conv(x)


class Down(nn.Module):
    """Downscaling with maxpool then double conv"""

    def __init__(self, in_channels, out_channels):
        super().__init__()
        self.maxpool_conv = nn.Sequential(
            nn.MaxPool2d(2),
            DoubleConv(in_channels, out_channels)
        )

    def forward(self, x):
        return self.maxpool_conv(x)


class Up(nn.Module):
    """Upscaling then double conv"""

    def __init__(self, in_channels, out_channels, bilinear=True):
        super().__init__()

        # if bilinear, use the normal convolutions to reduce the number of channels
        if bilinear:
            self.up = nn.Upsample(scale_factor=2, mode='bilinear', align_corners=True)
            self.conv = DoubleConv(in_channels, out_channels, in_channels // 2)
        else:
            self.up = nn.ConvTranspose2d(in_channels, in_channels // 2, kernel_size=2, stride=2)
            self.conv = DoubleConv(in_channels, out_channels)

    def forward(self, x1, x2):
        x1 = self.up(x1)
        # input is CHW
        diffY = x2.size()[2] - x1.size()[2]
        diffX = x2.size()[3] - x1.size()[3]

        x1 = F.pad(x1, [diffX // 2, diffX - diffX // 2,
                        diffY // 2, diffY - diffY // 2])
        # if you have padding issues, see
        # https://github.com/HaiyongJiang/U-Net-Pytorch-Unstructured-Buggy/commit/0e854509c2cea854e247a9c615f175f76fbb2e3a
        # https://github.com/xiaopeng-liao/Pytorch-UNet/commit/8ebac70e633bac59fc22bb5195e513d5832fb3bd
        x = torch.cat([x2, x1], dim=1)
        return self.conv(x)


class OutConv(nn.Module):
    def __init__(self, in_channels, out_channels):
        super(OutConv, self).__init__()
        self.conv = nn.Conv2d(in_channels, out_channels, kernel_size=1)

    def forward(self, x):
        return self.conv(x)


################################################################
# Patchify and Neural Spectral Block
################################################################
# class NeuralSpectralBlock2d(nn.Module):
#     def __init__(self, width, num_basis, patch_size=[3, 3], num_token=4):
#         super(NeuralSpectralBlock2d, self).__init__()
#         self.patch_size = patch_size
#         self.width = width
#         self.num_basis = num_basis

#         # basis
#         self.modes_list = (1.0 / float(num_basis)) * torch.tensor([i for i in range(num_basis)],
#                                                                   dtype=torch.float).cuda()
#         self.weights = nn.Parameter(
#             (1 / (width)) * torch.rand(width, self.num_basis * 2, dtype=torch.float))
#         # latent
#         self.head = 8
#         self.num_token = num_token
#         self.latent = nn.Parameter(
#             (1 / (width)) * torch.rand(self.head, self.num_token, width // self.head, dtype=torch.float))
#         self.encoder_attn = nn.Conv2d(self.width, self.width * 2, kernel_size=1, stride=1)
#         self.decoder_attn = nn.Conv2d(self.width, self.width, kernel_size=1, stride=1)
#         self.softmax = nn.Softmax(dim=-1)

#     def self_attn(self, q, k, v):
#         # q,k,v: B H L C/H
#         attn = self.softmax(torch.einsum("bhlc,bhsc->bhls", q, k))
#         return torch.einsum("bhls,bhsc->bhlc", attn, v)

#     def latent_encoder_attn(self, x):
#         # x: B C H W
#         B, C, H, W = x.shape
#         L = H * W
#         latent_token = self.latent[None, :, :, :].repeat(B, 1, 1, 1)
#         x_tmp = self.encoder_attn(x).view(B, C * 2, -1).permute(0, 2, 1).contiguous() \
#             .view(B, L, self.head, C // self.head, 2).permute(4, 0, 2, 1, 3).contiguous()
#         latent_token = self.self_attn(latent_token, x_tmp[0], x_tmp[1]) + latent_token
#         latent_token = latent_token.permute(0, 1, 3, 2).contiguous().view(B, C, self.num_token)
#         return latent_token

#     def latent_decoder_attn(self, x, latent_token):
#         # x: B C L
#         x_init = x
#         B, C, H, W = x.shape
#         L = H * W
#         latent_token = latent_token.view(B, self.head, C // self.head, self.num_token).permute(0, 1, 3, 2).contiguous()
#         x_tmp = self.decoder_attn(x).view(B, C, -1).permute(0, 2, 1).contiguous() \
#             .view(B, L, self.head, C // self.head).permute(0, 2, 1, 3).contiguous()
#         x = self.self_attn(x_tmp, latent_token, latent_token)
#         x = x.permute(0, 1, 3, 2).contiguous().view(B, C, H, W) + x_init  # B H L C/H
#         return x

#     def get_basis(self, x):
#         # x: B C N
#         x_sin = torch.sin(self.modes_list[None, None, None, :] * x[:, :, :, None] * math.pi)
#         x_cos = torch.cos(self.modes_list[None, None, None, :] * x[:, :, :, None] * math.pi)
#         return torch.cat([x_sin, x_cos], dim=-1)

#     def compl_mul2d(self, input, weights):
#         return torch.einsum("bilm,im->bil", input, weights)

#     def forward(self, x):
#         B, C, H, W = x.shape
#         # patchify
#         x = x.view(x.shape[0], x.shape[1],
#                    x.shape[2] // self.patch_size[0], self.patch_size[0], x.shape[3] // self.patch_size[1],
#                    self.patch_size[1]).contiguous() \
#             .permute(0, 2, 4, 1, 3, 5).contiguous() \
#             .view(x.shape[0] * (x.shape[2] // self.patch_size[0]) * (x.shape[3] // self.patch_size[1]), x.shape[1],
#                   self.patch_size[0],
#                   self.patch_size[1])
#         # Neural Spectral
#         # (1) encoder
#         latent_token = self.latent_encoder_attn(x)
#         # (2) transition
#         latent_token_modes = self.get_basis(latent_token)
#         latent_token = self.compl_mul2d(latent_token_modes, self.weights) + latent_token
#         # (3) decoder
#         x = self.latent_decoder_attn(x, latent_token)
#         # de-patchify
#         x = x.view(B, (H // self.patch_size[0]), (W // self.patch_size[1]), C, self.patch_size[0],
#                    self.patch_size[1]).permute(0, 3, 1, 4, 2, 5).contiguous() \
#             .view(B, C, H, W).contiguous()
#         return x  
class NeuralSpectralBlock2d(nn.Module):
    def __init__(self, width, num_basis, patch_size=[3, 3], num_token=4, channel_mixing='prod-layer', num_prods=2):
        super(NeuralSpectralBlock2d, self).__init__()
        self.patch_size = patch_size
        self.width = width
        self.num_basis = num_basis

        # basis
        self.modes_list = (1.0 / float(num_basis)) * torch.tensor([i for i in range(num_basis)],
                                                                  dtype=torch.float).cuda()
        self.weights = nn.Parameter(
            (1 / (width)) * torch.rand(width, self.num_basis * 2, dtype=torch.float))
        # latent
        self.head = 8
        self.num_token = num_token
        self.latent = nn.Parameter(
            (1 / (width)) * torch.rand(self.head, self.num_token, width // self.head, dtype=torch.float))
        # if prod_layer:
        #     self.encoder_attn = ProductLayer(in_dim=self.width, out_dim=self.width*2, num_prods=num_prods)
        # else:
        #     self.encoder_attn = nn.Conv2d(self.width, self.width * 2, kernel_size=1, stride=1)
        # self.decoder_attn = nn.Conv2d(self.width, self.width, kernel_size=1, stride=1)
        self.encoder_attn = nn.Conv2d(self.width, self.width * 2, kernel_size=1, stride=1)
        prod_layer = (channel_mixing=='prod-layer')
        if prod_layer:
            self.decoder_attn = ProductLayer(in_dim=self.width, out_dim=self.width, num_prods=num_prods)
        else:
            self.decoder_attn = nn.Conv2d(self.width, self.width, kernel_size=1, stride=1)
        self.softmax = nn.Softmax(dim=-1)

    def self_attn(self, q, k, v):
        # q,k,v: B H L C/H
        attn = self.softmax(torch.einsum("bhlc,bhsc->bhls", q, k))
        return torch.einsum("bhls,bhsc->bhlc", attn, v)

    def latent_encoder_attn(self, x):
        # x: B C H W
        B, C, H, W = x.shape
        L = H * W
        latent_token = self.latent[None, :, :, :].repeat(B, 1, 1, 1)
        x_tmp = self.encoder_attn(x).view(B, C * 2, -1).permute(0, 2, 1).contiguous() \
            .view(B, L, self.head, C // self.head, 2).permute(4, 0, 2, 1, 3).contiguous()
        latent_token = self.self_attn(latent_token, x_tmp[0], x_tmp[1]) + latent_token
        latent_token = latent_token.permute(0, 1, 3, 2).contiguous().view(B, C, self.num_token)
        return latent_token

    def latent_decoder_attn(self, x, latent_token):
        # x: B C L
        x_init = x
        B, C, H, W = x.shape
        L = H * W
        latent_token = latent_token.view(B, self.head, C // self.head, self.num_token).permute(0, 1, 3, 2).contiguous()
        x_tmp = self.decoder_attn(x).view(B, C, -1).permute(0, 2, 1).contiguous() \
            .view(B, L, self.head, C // self.head).permute(0, 2, 1, 3).contiguous()
        x = self.self_attn(x_tmp, latent_token, latent_token)
        x = x.permute(0, 1, 3, 2).contiguous().view(B, C, H, W) + x_init  # B H L C/H
        return x

    def get_basis(self, x):
        # x: B C N
        x_sin = torch.sin(self.modes_list[None, None, None, :] * x[:, :, :, None] * math.pi)
        x_cos = torch.cos(self.modes_list[None, None, None, :] * x[:, :, :, None] * math.pi)
        return torch.cat([x_sin, x_cos], dim=-1)

    def compl_mul2d(self, input, weights):
        return torch.einsum("bilm,im->bil", input, weights)

    def forward(self, x):
        B, C, H, W = x.shape
        # patchify
        x = x.view(x.shape[0], x.shape[1],
                   x.shape[2] // self.patch_size[0], self.patch_size[0], x.shape[3] // self.patch_size[1],
                   self.patch_size[1]).contiguous() \
            .permute(0, 2, 4, 1, 3, 5).contiguous() \
            .view(x.shape[0] * (x.shape[2] // self.patch_size[0]) * (x.shape[3] // self.patch_size[1]), x.shape[1],
                  self.patch_size[0],
                  self.patch_size[1])
        # Neural Spectral
        # (1) encoder
        latent_token = self.latent_encoder_attn(x)
        # (2) transition
        latent_token_modes = self.get_basis(latent_token)
        latent_token = self.compl_mul2d(latent_token_modes, self.weights) + latent_token
        # (3) decoder
        x = self.latent_decoder_attn(x, latent_token)
        # de-patchify
        x = x.view(B, (H // self.patch_size[0]), (W // self.patch_size[1]), C, self.patch_size[0],
                   self.patch_size[1]).permute(0, 3, 1, 4, 2, 5).contiguous() \
            .view(B, C, H, W).contiguous()
        return x


class LSM_2D(nn.Module):
    def __init__(self, in_dim, out_dim, d_model, num_token, num_basis, patch_size, padding, bilinear=True, channel_mixing='', num_prod=2):
        super(LSM_2D, self).__init__()
        in_channels = in_dim
        out_channels = out_dim
        self.in_channels = in_dim
        self.out_channels = out_dim
        width = d_model
        # multiscale modules
        self.inc = DoubleConv(width, width)
        self.down1 = Down(width, width * 2)
        self.down2 = Down(width * 2, width * 4)
        self.down3 = Down(width * 4, width * 8)
        factor = 2 if bilinear else 1
        self.down4 = Down(width * 8, width * 16 // factor)
        self.up1 = Up(width * 16, width * 8 // factor, bilinear)
        self.up2 = Up(width * 8, width * 4 // factor, bilinear)
        self.up3 = Up(width * 4, width * 2 // factor, bilinear)
        self.up4 = Up(width * 2, width, bilinear)
        self.outc = OutConv(width, width)
        # Patchified Neural Spectral Blocks
        self.process1 = NeuralSpectralBlock2d(width, num_basis, patch_size, num_token, channel_mixing=channel_mixing, num_prods=num_prod)
        self.process2 = NeuralSpectralBlock2d(width * 2, num_basis, patch_size, num_token, channel_mixing=channel_mixing, num_prods=num_prod)
        self.process3 = NeuralSpectralBlock2d(width * 4, num_basis, patch_size, num_token, channel_mixing=channel_mixing, num_prods=num_prod)
        self.process4 = NeuralSpectralBlock2d(width * 8, num_basis, patch_size, num_token, channel_mixing=channel_mixing, num_prods=num_prod)
        self.process5 = NeuralSpectralBlock2d(width * 16 // factor, num_basis, patch_size, num_token, channel_mixing=channel_mixing, num_prods=num_prod)
        # projectors
        self.padding = padding
        self.fc0 = nn.Linear(in_channels, width)
        self.fc1 = nn.Linear(width, 128)
        self.fc2 = nn.Linear(128, out_channels)

    def forward(self, x, **kwargs):
        x = x.permute(0, 2, 3, 1)
        x = self.fc0(x)
        x = x.permute(0, 3, 1, 2)
        if not all(item == 0 for item in self.padding):
            x = F.pad(x, [0, self.padding[0], 0, self.padding[1]])
        x1 = self.inc(x)
        x2 = self.down1(x1)
        x3 = self.down2(x2)
        x4 = self.down3(x3)
        x5 = self.down4(x4)
        x = self.up1(self.process5(x5), self.process4(x4))
        x = self.up2(x, self.process3(x3))
        x = self.up3(x, self.process2(x2))
        x = self.up4(x, self.process1(x1))
        x = self.outc(x)

        if not all(item == 0 for item in self.padding):
            x = x[..., :-self.padding[1], :-self.padding[0]]
        x = x.permute(0, 2, 3, 1)
        x = self.fc1(x)
        x = F.gelu(x)
        x = self.fc2(x)
        x = x.permute(0, 3, 1, 2)
        return x

    # def get_grid(self, shape, device):
    #     batchsize, size_x, size_y = shape[0], shape[1], shape[2]
    #     gridx = torch.tensor(np.linspace(0, 1, size_x), dtype=torch.float)
    #     gridx = gridx.reshape(1, size_x, 1, 1).repeat([batchsize, 1, size_y, 1])
    #     gridy = torch.tensor(np.linspace(0, 1, size_y), dtype=torch.float)
    #     gridy = gridy.reshape(1, 1, size_y, 1).repeat([batchsize, size_x, 1, 1])
    #     return torch.cat((gridx, gridy), dim=-1).to(device)
