import os
import pandas as pd
from cache import CacheManager
from rag1_scoring import load_rag1_excel, score_student_answer
from rag2_retrieval import load_faiss_index
from config import ANSWER_KEY_EXCEL, STUDENT_ANSWERS_EXCEL, FAISS_INDEX_PATH

# -----------------------------
# Ensure results folder exists
# -----------------------------
os.makedirs("results", exist_ok=True)

# -----------------------------
# Load RAG1 answer key (faculty-provided)
# -----------------------------
rag1_data = load_rag1_excel(ANSWER_KEY_EXCEL)

# -----------------------------
# Initialize cache manager
# -----------------------------
cache_manager = CacheManager()

# -----------------------------
# Load RAG2 FAISS index (textbook / other material)
# -----------------------------
vectorstore = load_faiss_index(FAISS_INDEX_PATH)

# -----------------------------
# Load student answers
# -----------------------------
df_answers = pd.read_excel(STUDENT_ANSWERS_EXCEL)
# Expected columns: Student_ID, Question, Answer

# -----------------------------
# Evaluate answers
# -----------------------------
results = []
for idx, row in df_answers.iterrows():
    student_id = row['StudentID']
    question_text = row['Question']
    student_answer = row['Answer']
    
    # Score student answer using optimized pipeline
    score, reasoning = score_student_answer(
        question_text,
        student_answer,
        rag1_data,
        cache_manager,
        vectorstore=vectorstore  # used for similarity search / RAG2 fallback
    )
    
    # Store result
    results.append({
        "Student_ID": student_id,
        "Question": question_text,
        "Score": score,
        "Reasoning": reasoning
    })

# -----------------------------
# Save results to Excel
# -----------------------------
df_results = pd.DataFrame(results)
df_results.to_excel("results/final_scores.xlsx", index=False)
print("Grading complete. Results saved to 'results/final_scores.xlsx'.")

# -----------------------------
# Save cache after grading (NEW ADDITION)
# -----------------------------
try:
    # Import the save_cache function
    from save_cache import save_cache
    
    # Save the populated cache
    save_cache(cache_manager)
    print("Cache saved successfully after grading!")
    
except ImportError:
    print("Warning: save_cache module not found. Cache not saved.")
except Exception as e:
    print(f"Warning: Could not save cache - {e}")
    # Continue anyway - grading results are already saved
