
from langchain_google_genai import ChatGoogleGenerativeAI
from config import GOOGLE_GEMINI_API_KEY, LLM_MODEL_NAME

def get_gemini_response(prompt: str, model: str = LLM_MODEL_NAME) -> str:
    """
    Robust wrapper for calling Gemini via langchain_google_genai.
    Tries a few common response shapes and prints debug info on failures.
    """
    try:
        llm = ChatGoogleGenerativeAI(model=model, google_api_key=GOOGLE_GEMINI_API_KEY)
        return llm.invoke(prompt)  # returns a string in your version
    except Exception as e:
        print(f"[get_gemini_response] Error: {e}")
        return f"ERROR: {e}"

    try:
        # Try common call patterns in order of likelihood
        # 1) Many wrappers implement predict / __call__
        if hasattr(llm, "predict"):
            resp = llm.predict(prompt)
            if resp:
                return str(resp)

        # 2) Some wrappers use generate(...) returning an object with .generations or .generations[0][0].text
        if hasattr(llm, "generate"):
            try:
                gen = llm.generate([prompt])  # some expect list
            except TypeError:
                gen = llm.generate(prompt)
            # Inspect object
            if hasattr(gen, "generations"):
                # Try common nested structures
                try:
                    g = gen.generations
                    if isinstance(g, list) and len(g) > 0 and isinstance(g[0], list) and len(g[0]) > 0:
                        # LangChain often uses generations[0][0].text
                        text = getattr(g[0][0], "text", None) or getattr(g[0][0], "message", None)
                        if text:
                            return text
                    # fallback to string conversion
                    return str(gen)
                except Exception:
                    return str(gen)

        # 3) Some libs have an 'invoke' or 'call' that returns .content (less common)
        if hasattr(llm, "invoke"):
            out = llm.invoke(prompt)
            # if it's an object with a .content attribute
            if hasattr(out, "content"):
                return str(out.content)
            # try string conversion
            return str(out)

        # 4) Last resort: try __call__
        try:
            out = llm(prompt)
            return str(out)
        except Exception as e:
            print(f"[get_gemini_response] __call__ failed: {e}")

        return "ERROR: Unable to parse LLM response (no known accessor matched)."

    except Exception as e:
        print(f"[get_gemini_response] Exception while calling LLM: {e}")
        return f"ERROR: LLM call raised exception: {e}"
