import pickle
import json
from cache import CacheManager

# -----------------------------
# Path to save caches
# -----------------------------
COLD_CACHE_PATH = "data/final_cold_cache.json"
HOT_CACHE_PATH = "data/final_hot_cache.json"
PICKLE_PATH = "data/final_cache.pkl"

# -----------------------------
# Assume cache_manager is already available
# If not, you can load from a pickle used in grading
# -----------------------------
# from grader import cache_manager

def save_cache(cache_manager: CacheManager):
    # Save as JSON for readability
    with open(COLD_CACHE_PATH, "w", encoding="utf-8") as f:
        json.dump(cache_manager.cold_cache, f, indent=4)

    with open(HOT_CACHE_PATH, "w", encoding="utf-8") as f:
        json.dump(cache_manager.hot_cache, f, indent=4)

    # Save entire cache object using pickle (for reloading later)
    with open(PICKLE_PATH, "wb") as f:
        pickle.dump(cache_manager, f)

    print(f"Cold cache saved to {COLD_CACHE_PATH}")
    print(f"Hot cache saved to {HOT_CACHE_PATH}")
    print(f"Entire CacheManager saved to {PICKLE_PATH}")


# -----------------------------
# Driver code (main execution)
# -----------------------------
if __name__ == "__main__":
    # Option 1: Load cache_manager from grader if it was run recently
    try:
        # Try to import cache_manager from grader
        import sys
        import os
        
        # Add current directory to Python path
        sys.path.append(os.getcwd())
        
        # Try to load existing cache manager from pickle if available
        if os.path.exists(PICKLE_PATH):
            with open(PICKLE_PATH, "rb") as f:
                cache_manager = pickle.load(f)
            print("Loaded existing cache manager from pickle file.")
        else:
            # Create new cache manager (this will be empty)
            cache_manager = CacheManager()
            print("Created new empty cache manager.")
        
        # Save the cache
        save_cache(cache_manager)
        
    except Exception as e:
        print(f"Error: {e}")
        print("\nTo use this script:")
        print("1. Run grader.py first to populate the cache")
        print("2. Then run this script to save the cache")
        print("   OR")
        print("3. Import and use the save_cache() function directly")