# Digital Stylus Exam System

A Next.js web application for proctored digital exams that combines a live browser camera feed, stylus-based handwriting capture, and AI-powered handwriting recognition. Teachers can monitor students in real-time, configure how recognized text is delivered (in-app display, email via webhook, or other integrations), and interact with a responsive, accessible interface featuring fully functional controls.

## Key Features

- Live Camera Feed (Browser)
  - Student devices publish a camera feed via the browser (getUserMedia) with privacy prompts.
  - Teacher view provides real-time monitoring UI with controls.
  - Components: `components/camera-feed.tsx`, `components/live-monitoring-interface.tsx`

- Stylus Handwriting Capture
  - Pointer-based drawing canvas to collect stylus-written answers.
  - Components: `components/stylus-ai-processor.tsx`, `components/student-exam-interface.tsx`

- AI Handwriting Recognition
  - Server endpoint processes canvas captures to extract text and produce outcomes.
  - API Route: `app/api/ai-grading/route.ts` (uses the Vercel AI SDK by default)
  - Outcomes may include extracted text, grading hints, or next-step suggestions.

- Delivery Options for Recognized Text
  - In-app display for instant feedback to teachers.
  - Webhook-based delivery pattern for email or external systems.
  - Teacher UI lets you select the delivery mode and configure destinations.
  - Components: `components/live-monitoring-interface.tsx`, `components/teacher-dashboard.tsx`

- Fully Functional Controls
  - All primary buttons are wired (start/stop camera, clear canvas, process handwriting, choose delivery, send).
  - Built with shadcn/ui for accessible, consistent interactions.

## Architecture Overview

- Framework: Next.js (App Router)
- UI: Tailwind CSS v4 + shadcn/ui
- AI: Vercel AI SDK via `app/api/ai-grading/route.ts`
- App Structure (selected files):
  - Pages
    - `app/page.tsx` – Landing
    - `app/student/page.tsx` – Student camera/stylus interface
    - `app/teacher/page.tsx` – Teacher monitoring & delivery configuration
    - `app/teacher/grading/page.tsx`, `app/grading/page.tsx`, `app/analytics/page.tsx` – Extended workflows
  - APIs
    - `app/api/ai-grading/route.ts` – Handwriting recognition and outcome generation
    - Other routes: `app/api/exam/*`, `app/api/grading/status/route.ts`
  - Components (selected)
    - `components/camera-feed.tsx`
    - `components/stylus-ai-processor.tsx`
    - `components/student-exam-interface.tsx`
    - `components/live-monitoring-interface.tsx`
    - `components/teacher-dashboard.tsx`
    - `components/enhanced-button-controls.tsx`
    - `components/ui/*` – shadcn/ui primitives

## Getting Started

Prerequisites
- Node.js 18+ and npm (or pnpm/yarn)
- A modern browser with camera permissions

Install and Run
\`\`\`bash
# Install dependencies
npm install

# Start dev server
npm run dev

# Open in your browser
# http://localhost:3000
\`\`\`

Build and Start (production)
\`\`\`bash
npm run build
npm start
\`\`\`

Deploy
- Recommended: Vercel. Push this repo to GitHub and Import to Vercel, or use the “Publish” button from v0.

## Environment Variables

Only set what you need for your chosen delivery and AI provider(s). Typical examples:

- AI SDK (provider-specific key, example)
  - OPENAI_API_KEY=...          # if using OpenAI via @ai-sdk/openai

- Optional Webhook Delivery
  - DELIVERY_WEBHOOK_URL=https://your-service.example.com/hook
  - DELIVERY_WEBHOOK_AUTH=Bearer your-secret-token

Note: If you enable email via webhook, your webhook should implement sending via your preferred provider (e.g., Resend, SendGrid). The app will POST recognized content to your configured webhook URL.

## Usage

1) Student Capture
- Navigate to /student
- Allow camera permission and start feed
- Write answers on the stylus canvas
- Submit/process handwriting to send to the AI endpoint

2) Teacher Monitoring
- Navigate to /teacher
- View the live camera feed(s) and AI results
- Choose delivery mode:
  - Display: show recognized text in the UI
  - Email/Webhook: send a POST to your DELIVERY_WEBHOOK_URL with recognized content
- Use the provided buttons to control monitoring and delivery

3) AI Processing
- The student or teacher triggers processing
- `app/api/ai-grading/route.ts` reads the image/content and produces recognized text and/or an outcome payload
- Results appear in the teacher UI or are delivered to your configured destination

## Accessibility & UX

- Buttons, forms, and dialogs use accessible components (shadcn/ui)
- Semantic HTML and ARIA attributes where appropriate
- Responsive layout with mobile-first patterns (flex/grid)
- Clear focus states and keyboard navigation support

## Customization

- Styling: Tailwind tokens in `app/globals.css`
- UI: Replace or extend components in `components/*`
- AI: Adjust model/provider in `app/api/ai-grading/route.ts`
- Delivery: Customize webhook payloads or add a dedicated email provider route

## Security & Privacy

- Camera feeds require explicit user permission
- Avoid storing raw video unless necessary
- Review any outbound delivery URLs and secrets
- If you process PII, ensure compliance with your organization’s policies

## Project Scripts

Common NPM scripts (if present in package.json):
- `dev` – run local dev server
- `build` – build for production
- `start` – start production server
- `lint` – lint codebase

## Contributing

- Fork the repo and create feature branches
- Keep changes scoped and include tests where applicable
- Open a PR with a clear description and screenshots for UI work

## License

MIT. See LICENSE (add one if missing).
