import { type NextRequest, NextResponse } from "next/server"

interface GradingRequest {
  question: string
  studentAnswer: string
  correctAnswer: string
  subject: "History" | "Geography" | "Political Science"
  totalMarks: number
}

interface GradingResponse {
  totalMarks: number
  awardedMarks: number
  pointBreakdown: {
    point: string
    awarded: boolean
    reason: string
  }[]
  overallFeedback: string
  suggestions: string[]
  similarityScore: number
}

const subjectPrompts = {
  History: `You are an expert Class 12 History teacher evaluating student answers. Focus on:
- Historical accuracy and factual knowledge
- Understanding of cause-and-effect relationships
- Analysis of historical events and their significance
- Use of appropriate historical terminology
- Critical thinking and evaluation skills`,

  Geography: `You are an expert Class 12 Geography teacher evaluating student answers. Focus on:
- Understanding of geographical concepts and processes
- Knowledge of spatial relationships and patterns
- Environmental awareness and sustainability concepts
- Use of appropriate geographical terminology
- Application of geographical skills and case studies`,

  "Political Science": `You are an expert Class 12 Political Science teacher evaluating student answers. Focus on:
- Understanding of political concepts and theories
- Knowledge of Indian political system and governance
- Analysis of democratic processes and institutions
- Critical evaluation of political issues
- Use of appropriate political terminology`,
}

function tokenize(text: string): string[] {
  return text
    .toLowerCase()
    .replace(/[^a-z0-9\s]/g, " ")
    .split(/\s+/)
    .filter(Boolean)
}

function heuristicGrade({ question, studentAnswer, correctAnswer, subject, totalMarks }: GradingRequest): GradingResponse {
  const studentTokens = new Set(tokenize(studentAnswer))
  const correctTokens = new Set(tokenize(correctAnswer))

  let overlap = 0
  for (const t of correctTokens) if (studentTokens.has(t)) overlap++
  const similarity = correctTokens.size ? overlap / correctTokens.size : 0
  const similarityScore = Math.round(similarity * 100)

  const concept = similarityScore >= 35
  const factual = similarityScore >= 55
  const analysis = studentAnswer.split(/[.,;!?]/).length > 2 && similarityScore >= 45
  const examples = /(for example|e\.g\.|such as|like)/i.test(studentAnswer)
  const structure = studentAnswer.trim().length > 0 && /\n|\./.test(studentAnswer)

  const points = [
    { point: "Understanding of concept", awarded: concept, reason: concept ? "Core idea present" : "Core concept missing" },
    { point: "Factual accuracy", awarded: factual, reason: factual ? "Key facts align" : "Important facts missing" },
    { point: "Analysis and evaluation", awarded: analysis, reason: analysis ? "Some reasoning shown" : "Limited analysis" },
    { point: "Use of examples", awarded: examples, reason: examples ? "Includes examples" : "Add specific examples" },
    { point: "Structure and clarity", awarded: structure, reason: structure ? "Readable structure" : "Improve organization" },
  ]

  const weight = [0.3, 0.25, 0.2, 0.1, 0.15]
  const awardedRatio = points.reduce((acc, p, i) => acc + (p.awarded ? weight[i] : 0), 0)
  const awardedMarks = Math.round(totalMarks * Math.max(0.1, Math.min(0.95, Math.max(awardedRatio, similarity * 0.9))))

  const keywords = Array.from(correctTokens).filter((k) => !studentTokens.has(k)).slice(0, 5)
  const suggestions = [
    keywords.length ? `Include keywords: ${keywords.join(", ")}` : "Add more subject-specific terms",
    "Provide clearer structure with short paragraphs",
    "Support points with concise examples",
  ]

  const overallFeedback = `Similarity ${similarityScore}%. ${subject} answer shows ${concept ? "basic" : "limited"} understanding${analysis ? ", some analysis" : ""}. Focus on missing keywords and examples.`

  return {
    totalMarks,
    awardedMarks,
    pointBreakdown: points,
    overallFeedback,
    suggestions,
    similarityScore,
  }
}

export async function POST(request: NextRequest) {
  try {
    const body: GradingRequest = await request.json()
    const { question, studentAnswer, correctAnswer, subject, totalMarks } = body

    // If no LLM key is provided, use heuristic grader
    const { getKeys } = await import("@/lib/keys")
    const keys = getKeys()
    if (!keys.llm && !process.env.OPENAI_API_KEY) {
      const response = heuristicGrade(body)
      return NextResponse.json(response)
    }

    const gradingPrompt = `${subjectPrompts[subject]}

INPUT:
QUESTION: ${question}
STUDENT ANSWER: ${studentAnswer}
CORRECT ANSWER: ${correctAnswer}
TOTAL MARKS: ${totalMarks}

Based on the input provided, evaluate the student answer and provide:
1. Point-by-point breakdown (mark each point as ✓ or ✗)
2. Reason for each point's evaluation
3. Total marks to be awarded
4. Overall feedback
5. Suggestions for improvement
6. Similarity score (0-100%)

Provide your response in the following JSON format:
{
  "pointBreakdown": [
    {"point": "Understanding of concept", "awarded": true/false, "reason": "explanation"},
    {"point": "Factual accuracy", "awarded": true/false, "reason": "explanation"},
    {"point": "Analysis and evaluation", "awarded": true/false, "reason": "explanation"},
    {"point": "Use of examples", "awarded": true/false, "reason": "explanation"},
    {"point": "Structure and clarity", "awarded": true/false, "reason": "explanation"}
  ],
  "awardedMarks": number,
  "overallFeedback": "detailed feedback",
  "suggestions": ["suggestion1", "suggestion2", "suggestion3"],
  "similarityScore": number
}
`

    // Dynamically import AI SDKs to avoid throwing on server startup if packages are missing
    let generateText: any = null
    let providerFunc: any = null
    try {
      const ai = await import("ai")
      generateText = ai.generateText
    } catch (e) {
      console.warn("Optional 'ai' SDK not available:", e)
    }

    try {
      const sdk = await import("@ai-sdk/openai")
      providerFunc = (apiKey?: string) => sdk.openai({ apiKey })
    } catch (e) {
      console.warn("Optional '@ai-sdk/openai' not available:", e)
    }

    // choose provider - either user-provided keys or environment
    const keysObj = keys
    const apiKeyToUse = keysObj.llm || process.env.OPENAI_API_KEY || null

    let text = ""
    if (generateText && providerFunc) {
      try {
        const provider = apiKeyToUse ? providerFunc(apiKeyToUse) : (await import("@ai-sdk/openai")).openai
        const result = await generateText({ model: provider("gpt-4"), prompt: gradingPrompt, temperature: 0.3 })
        text = result?.text || ""
      } catch (aiErr) {
        console.error("AI SDK error, falling back to heuristic:", aiErr)
        const response = heuristicGrade(body)
        return NextResponse.json(response)
      }
    } else {
      // fallback: use heuristic if AI SDK not available
      const response = heuristicGrade(body)
      return NextResponse.json(response)
    }

    let aiResponse
    try {
      const jsonMatch = text.match(/\{[\s\S]*\}/)
      if (jsonMatch) {
        aiResponse = JSON.parse(jsonMatch[0])
      } else {
        throw new Error("No JSON found in response")
      }
    } catch (parseError) {
      aiResponse = heuristicGrade(body)
    }

    const response: GradingResponse = {
      totalMarks,
      awardedMarks: aiResponse.awardedMarks,
      pointBreakdown: aiResponse.pointBreakdown,
      overallFeedback: aiResponse.overallFeedback,
      suggestions: aiResponse.suggestions,
      similarityScore: aiResponse.similarityScore,
    }

    return NextResponse.json(response)
  } catch (error) {
    console.error("AI Grading Error:", error)
    try {
      const body: GradingRequest = await request.json()
      return NextResponse.json(heuristicGrade(body))
    } catch {
      return NextResponse.json({ error: "Failed to grade answer" }, { status: 500 })
    }
  }
}
