import { type NextRequest, NextResponse } from "next/server"

const state: Map<string, { startedAt: number }> = new Map()

export async function GET(request: NextRequest) {
  const { searchParams } = new URL(request.url)
  const submissionId = searchParams.get("submissionId")

  if (!submissionId) {
    return NextResponse.json({ error: "Submission ID required" }, { status: 400 })
  }

  if (!state.has(submissionId)) {
    state.set(submissionId, { startedAt: Date.now() })
  }
  const startedAt = state.get(submissionId)!.startedAt
  const elapsed = (Date.now() - startedAt) / 1000

  const steps = ["queued", "processing", "ocr_complete", "ai_grading", "review", "completed"]
  const durations = [2, 5, 5, 6, 2] // seconds per step
  let idx = 0
  let acc = 0
  for (let i = 0; i < durations.length; i++) {
    acc += durations[i]
    if (elapsed < acc) {
      idx = i
      break
    }
    idx = i + 1
  }
  const status = steps[Math.min(idx, steps.length - 1)]

  const gradingStatus: any = {
    submissionId,
    status,
    progress: Math.min(100, Math.round(((idx + 1) / steps.length) * 100)),
    stages: {
      handwritingRecognition: idx >= 2 ? "completed" : idx >= 1 ? "processing" : "pending",
      aiGrading: idx >= 4 ? "completed" : idx >= 3 ? "processing" : "pending",
      humanReview: idx >= 5 ? "completed" : idx >= 4 ? "processing" : "pending",
    },
    estimatedCompletion: status === "completed" ? null : "under a minute",
    lastUpdated: new Date().toISOString(),
  }

  if (status === "completed") {
    gradingStatus.results = {
      totalScore: 0,
      maxScore: 100,
      grade: "",
      questionScores: {},
    }
  }

  return NextResponse.json(gradingStatus)
}
