import { NextResponse } from "next/server"
import { subscribe } from "@/lib/realtime"

export const runtime = "nodejs"

export async function GET(request: Request) {
  try {
    const { searchParams } = new URL(request.url)
    const studentId = searchParams.get("studentId") || "default"
    const channel = `student:${studentId}`

    const stream = new ReadableStream({
      start(controller) {
        try {
          const send = (data: string) => {
            // ensure data is a single-line string
            const payload = typeof data === "string" ? data.replace(/\n/g, "\\n") : JSON.stringify(data)
            controller.enqueue(`data: ${payload}\n\n`)
          }

          // initial ping
          send(JSON.stringify({ type: "connected" }))

          const unsubscribe = subscribe(channel, send)
          const keepAlive = setInterval(() => controller.enqueue(`: keep-alive\n\n`), 25000)

          controller.signal.addEventListener("abort", () => {
            clearInterval(keepAlive)
            unsubscribe()
          })
        } catch (startErr) {
          // If starting the stream fails, close controller and rethrow
          try {
            controller.error(startErr)
          } catch {}
        }
      },
      cancel() {
        // noop - the abort handler in start handles cleanup
      },
    })

    return new NextResponse(stream as any, {
      headers: {
        "Content-Type": "text/event-stream",
        "Cache-Control": "no-cache, no-transform",
        Connection: "keep-alive",
      },
    })
  } catch (err) {
    console.error("SSE stream error:", err)
    return NextResponse.json({ error: "Failed to open SSE stream" }, { status: 500 })
  }
}
