"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Textarea } from "@/components/ui/textarea"
import { Separator } from "@/components/ui/separator"
import {
  Brain,
  RefreshCw,
  CheckCircle,
  XCircle,
  Star,
  BookOpen,
  MapPin,
  Vote,
  MessageSquare,
  Award,
  Target,
} from "lucide-react"
import { gradeAnswerWithAI, calculateGradeFromMarks, getGradeColor } from "@/lib/ai-grading"

interface StudentAnswer {
  id: string
  questionId: string
  question: string
  studentAnswer: string
  correctAnswer: string
  subject: "History" | "Geography" | "Political Science"
  marks: number
  totalMarks: number
  similarityScore: number
  submittedAt: string
}

interface AIGradeResult {
  totalMarks: number
  awardedMarks: number
  pointBreakdown: {
    point: string
    awarded: boolean
    reason: string
  }[]
  overallFeedback: string
  suggestions: string[]
  similarityScore: number
}

interface AIGradingInterfaceProps {
  answer: StudentAnswer
  onGradeUpdate: (answerId: string, grade: AIGradeResult) => void
}

export function AIGradingInterface({ answer, onGradeUpdate }: AIGradingInterfaceProps) {
  const [isGrading, setIsGrading] = useState(false)
  const [aiGrade, setAiGrade] = useState<AIGradeResult | null>(null)
  const [manualOverride, setManualOverride] = useState(false)
  const [overrideMarks, setOverrideMarks] = useState(answer.marks)

  const handleAIGrading = async () => {
    setIsGrading(true)
    try {
      const result = await gradeAnswerWithAI({
        question: answer.question,
        studentAnswer: answer.studentAnswer,
        correctAnswer: answer.correctAnswer,
        subject: answer.subject,
        totalMarks: answer.totalMarks,
      })

      setAiGrade(result)
      onGradeUpdate(answer.id, result)
    } catch (error) {
      console.error("Failed to grade answer:", error)
      const tokenize = (t: string) => t.toLowerCase().replace(/[^a-z0-9\s]/g, " ").split(/\s+/).filter(Boolean)
      const st = new Set(tokenize(answer.studentAnswer))
      const ct = new Set(tokenize(answer.correctAnswer))
      let overlap = 0
      for (const k of ct) if (st.has(k)) overlap++
      const sim = ct.size ? Math.round((overlap / ct.size) * 100) : 0
      const pts = [
        { point: "Understanding of concept", awarded: sim >= 35, reason: sim >= 35 ? "Core idea present" : "Core concept missing" },
        { point: "Factual accuracy", awarded: sim >= 55, reason: sim >= 55 ? "Key facts align" : "Important facts missing" },
        { point: "Analysis and evaluation", awarded: answer.studentAnswer.split(/[.,;!?]/).length > 2 && sim >= 45, reason: answer.studentAnswer.split(/[.,;!?]/).length > 2 && sim >= 45 ? "Some reasoning shown" : "Limited analysis" },
        { point: "Use of examples", awarded: /(for example|e\.g\.|such as|like)/i.test(answer.studentAnswer), reason: /(for example|e\.g\.|such as|like)/i.test(answer.studentAnswer) ? "Includes examples" : "Add specific examples" },
        { point: "Structure and clarity", awarded: answer.studentAnswer.trim().length > 0 && /\n|\./.test(answer.studentAnswer), reason: answer.studentAnswer.trim().length > 0 && /\n|\./.test(answer.studentAnswer) ? "Readable structure" : "Improve organization" },
      ]
      const weight = [0.3, 0.25, 0.2, 0.1, 0.15]
      const ratio = pts.reduce((a, p, i) => a + (p.awarded ? weight[i] : 0), 0)
      const awardedMarks = Math.round(answer.totalMarks * Math.max(0.1, Math.min(0.95, Math.max(ratio, (sim / 100) * 0.9))))
      const suggestions = [
        "Provide clearer structure with short paragraphs",
        "Support points with concise examples",
      ]
      const localResult = {
        totalMarks: answer.totalMarks,
        awardedMarks,
        pointBreakdown: pts,
        overallFeedback: `Similarity ${sim}%. ${answer.subject} answer shows ${sim >= 35 ? "basic" : "limited"} understanding.`,
        suggestions,
        similarityScore: sim,
      }
      setAiGrade(localResult)
      onGradeUpdate(answer.id, localResult)
    } finally {
      setIsGrading(false)
    }
  }

  const getSubjectIcon = (subject: string) => {
    switch (subject) {
      case "History":
        return <BookOpen className="w-4 h-4" />
      case "Geography":
        return <MapPin className="w-4 h-4" />
      case "Political Science":
        return <Vote className="w-4 h-4" />
      default:
        return <BookOpen className="w-4 h-4" />
    }
  }

  const grade = calculateGradeFromMarks(aiGrade?.awardedMarks || answer.marks, answer.totalMarks)
  const gradeColor = getGradeColor(grade)

  return (
    <div className="space-y-6">
      {/* Enhanced Answer Input Section */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <span className="flex items-center gap-2">
              {getSubjectIcon(answer.subject)}
              INPUT - {answer.subject} Assessment
            </span>
            <div className="flex items-center gap-2">
              <Badge variant="outline" className="text-lg px-3 py-1">
                <Target className="w-4 h-4 mr-1" />
                {answer.totalMarks} Marks
              </Badge>
            </div>
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 p-4 bg-muted rounded-lg">
            <div className="space-y-3">
              <div>
                <label className="text-sm font-medium">QUESTION:</label>
                <div className="mt-1 p-2 bg-background rounded border text-sm">{answer.question}</div>
              </div>

              <div>
                <label className="text-sm font-medium">STUDENT ANSWER:</label>
                <Textarea value={answer.studentAnswer} readOnly className="mt-1 min-h-24 text-sm" />
              </div>
            </div>

            <div className="space-y-3">
              <div>
                <label className="text-sm font-medium">CORRECT ANSWER:</label>
                <div className="mt-1 p-2 bg-background rounded border text-sm max-h-24 overflow-y-auto">
                  {answer.correctAnswer}
                </div>
              </div>

              <div className="grid grid-cols-2 gap-3">
                <div>
                  <label className="text-sm font-medium">SIMILARITY SCORE:</label>
                  <div className="mt-1 p-2 bg-background rounded border text-sm font-bold">
                    {aiGrade?.similarityScore || answer.similarityScore}%
                  </div>
                </div>

                <div>
                  <label className="text-sm font-medium">TOTAL MARKS:</label>
                  <div className="mt-1 p-2 bg-background rounded border text-sm font-bold">{answer.totalMarks}</div>
                </div>
              </div>
            </div>
          </div>

          <div className="flex justify-center">
            <Button onClick={handleAIGrading} disabled={isGrading} size="lg" className="min-w-48">
              {isGrading ? (
                <>
                  <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                  Analyzing with AI...
                </>
              ) : (
                <>
                  <Brain className="w-4 h-4 mr-2" />
                  Grade with AI
                </>
              )}
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Enhanced AI Grading Results */}
      {aiGrade && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center justify-between">
              <span>PROMPT - AI Grading Analysis</span>
              <div className="flex items-center gap-3">
                <div className="flex items-center gap-2 px-4 py-2 bg-primary/10 rounded-lg">
                  <Award className="w-5 h-5 text-primary" />
                  <span className="text-lg font-bold">
                    {aiGrade.awardedMarks}/{aiGrade.totalMarks}
                  </span>
                  <span className="text-sm text-muted-foreground">marks</span>
                </div>
                <Badge variant="secondary" className={`${gradeColor} font-bold text-lg px-3 py-1`}>
                  Grade: {grade}
                </Badge>
              </div>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="p-6 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg border-2 border-blue-200">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-blue-900">Marks Breakdown</h3>
                <div className="text-3xl font-bold text-blue-600">
                  {Math.round((aiGrade.awardedMarks / aiGrade.totalMarks) * 100)}%
                </div>
              </div>

              <div className="grid grid-cols-3 gap-4 mb-4">
                <div className="text-center p-3 bg-white rounded-lg">
                  <div className="text-2xl font-bold text-green-600">{aiGrade.awardedMarks}</div>
                  <div className="text-sm text-muted-foreground">Marks Awarded</div>
                </div>
                <div className="text-center p-3 bg-white rounded-lg">
                  <div className="text-2xl font-bold text-blue-600">{aiGrade.totalMarks}</div>
                  <div className="text-sm text-muted-foreground">Total Marks</div>
                </div>
                <div className="text-center p-3 bg-white rounded-lg">
                  <div className="text-2xl font-bold text-purple-600">{aiGrade.similarityScore}%</div>
                  <div className="text-sm text-muted-foreground">Similarity</div>
                </div>
              </div>

              <div className="text-center">
                <p className="text-sm font-medium text-blue-800">
                  "Based on the input provided, state reason and marks that should be awarded to the student answer"
                </p>
              </div>
            </div>

            <div className="p-4 bg-muted rounded-lg">
              <div className="mb-4">
                <h4 className="font-medium mb-3">Point-by-Point Analysis:</h4>
                <div className="space-y-3">
                  {aiGrade.pointBreakdown.map((point, index) => (
                    <div key={index} className="flex items-start gap-3 p-3 bg-background rounded border">
                      <span className="text-sm font-medium min-w-8 bg-primary text-primary-foreground rounded px-2 py-1">
                        {index + 1}
                      </span>
                      {point.awarded ? (
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                      ) : (
                        <XCircle className="w-5 h-5 text-red-600 mt-0.5" />
                      )}
                      <div className="flex-1">
                        <span className="text-sm font-medium">{point.point}</span>
                        <p className="text-xs text-muted-foreground mt-1">{point.reason}</p>
                      </div>
                      <div className="text-right">
                        <span className={`text-sm font-bold ${point.awarded ? "text-green-600" : "text-red-600"}`}>
                          {point.awarded ? "✓" : "✗"}
                        </span>
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              <div className="flex items-center justify-between pt-4 border-t-2 border-primary/20">
                <span className="text-xl font-bold text-primary">Total marks to be awarded:</span>
                <div className="flex items-center gap-3">
                  <div className="text-3xl font-bold text-primary">
                    {aiGrade.awardedMarks}/{aiGrade.totalMarks}
                  </div>
                  <div className="flex items-center gap-2 px-3 py-1 bg-yellow-100 rounded-full">
                    <Star className="w-5 h-5 text-yellow-500" />
                    <span className="font-medium text-yellow-700">
                      {Math.round((aiGrade.awardedMarks / aiGrade.totalMarks) * 100)}%
                    </span>
                  </div>
                </div>
              </div>
            </div>

            <Separator />

            <div className="space-y-4">
              <div>
                <h4 className="font-medium mb-2 flex items-center gap-2">
                  <MessageSquare className="w-4 h-4" />
                  Overall Feedback
                </h4>
                <p className="text-sm text-muted-foreground bg-muted p-3 rounded">{aiGrade.overallFeedback}</p>
              </div>

              <div>
                <h4 className="font-medium mb-2">Suggestions for Improvement</h4>
                <ul className="text-sm text-muted-foreground space-y-1">
                  {aiGrade.suggestions.map((suggestion, index) => (
                    <li key={index} className="flex items-start gap-2">
                      <span className="w-1 h-1 bg-muted-foreground rounded-full mt-2 flex-shrink-0"></span>
                      {suggestion}
                    </li>
                  ))}
                </ul>
              </div>
            </div>

            <Separator />

            <div className="flex gap-3">
              <Button
                className="flex-1 text-lg py-3"
                onClick={() => {
                  console.log("Accepting AI grade:", aiGrade.awardedMarks)
                }}
              >
                <CheckCircle className="w-5 h-5 mr-2" />
                Accept AI Grade ({aiGrade.awardedMarks}/{aiGrade.totalMarks} marks)
              </Button>

              <Button
                variant="outline"
                className="flex-1 text-lg py-3 bg-transparent"
                onClick={() => setManualOverride(!manualOverride)}
              >
                Manual Override
              </Button>
            </div>

            {manualOverride && (
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Manual Grade Override</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="flex items-center gap-4">
                    <label className="text-sm font-medium min-w-fit">Override Marks:</label>
                    <input
                      type="number"
                      min="0"
                      max={answer.totalMarks}
                      value={overrideMarks}
                      onChange={(e) => setOverrideMarks(Number(e.target.value))}
                      className="flex-1 px-3 py-2 border rounded-md text-lg font-bold"
                    />
                    <span className="text-lg font-medium">/ {answer.totalMarks}</span>
                    <Badge variant="outline" className="text-sm">
                      {Math.round((overrideMarks / answer.totalMarks) * 100)}%
                    </Badge>
                  </div>

                  <div className="flex gap-2">
                    <Button
                      size="sm"
                      onClick={() => {
                        console.log("Manual override applied:", overrideMarks)
                        setManualOverride(false)
                      }}
                    >
                      Apply Override ({overrideMarks}/{answer.totalMarks})
                    </Button>
                    <Button variant="outline" size="sm" onClick={() => setManualOverride(false)}>
                      Cancel
                    </Button>
                  </div>
                </CardContent>
              </Card>
            )}
          </CardContent>
        </Card>
      )}
    </div>
  )
}
