"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Progress } from "@/components/ui/progress"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  BarChart,
  Bar,
  LineChart,
  Line,
  PieChart,
  Pie,
  Cell,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  ResponsiveContainer,
} from "recharts"
import { TrendingUp, TrendingDown, Users, Clock, Target, Award, AlertTriangle, Download } from "lucide-react"

interface PerformanceData {
  subject: string
  averageScore: number
  totalStudents: number
  passRate: number
  trend: "up" | "down" | "stable"
}

interface QuestionAnalysis {
  questionId: number
  questionText: string
  averageScore: number
  difficulty: "easy" | "medium" | "hard"
  commonMistakes: string[]
  timeSpent: number
}

const performanceData: PerformanceData[] = [
  { subject: "Advanced Mathematics", averageScore: 78.5, totalStudents: 45, passRate: 82.2, trend: "up" },
  { subject: "Physics Mechanics", averageScore: 71.2, totalStudents: 38, passRate: 76.3, trend: "down" },
  { subject: "Organic Chemistry", averageScore: 84.1, totalStudents: 52, passRate: 88.5, trend: "up" },
  { subject: "Data Structures", averageScore: 69.8, totalStudents: 41, passRate: 73.2, trend: "stable" },
]

const scoreDistributionData = [
  { range: "90-100", count: 15, percentage: 12.5 },
  { range: "80-89", count: 28, percentage: 23.3 },
  { range: "70-79", count: 35, percentage: 29.2 },
  { range: "60-69", count: 25, percentage: 20.8 },
  { range: "50-59", count: 12, percentage: 10.0 },
  { range: "0-49", count: 5, percentage: 4.2 },
]

const timeSeriesData = [
  { month: "Jan", averageScore: 75.2, passRate: 78.5 },
  { month: "Feb", averageScore: 77.8, passRate: 81.2 },
  { month: "Mar", averageScore: 76.5, passRate: 79.8 },
  { month: "Apr", averageScore: 79.1, passRate: 83.4 },
  { month: "May", averageScore: 78.3, passRate: 82.1 },
  { month: "Jun", averageScore: 80.2, passRate: 85.6 },
]

const questionAnalysis: QuestionAnalysis[] = [
  {
    questionId: 1,
    questionText: "Solve the differential equation dy/dx = x²y",
    averageScore: 82.5,
    difficulty: "medium",
    commonMistakes: ["Incorrect integration", "Missing constant of integration"],
    timeSpent: 18.5,
  },
  {
    questionId: 2,
    questionText: "Find the area bounded by curves y = x² and y = 2x - x²",
    averageScore: 74.2,
    difficulty: "hard",
    commonMistakes: ["Wrong intersection points", "Incorrect integral setup"],
    timeSpent: 22.3,
  },
  {
    questionId: 3,
    questionText: "Prove series convergence using Basel problem approach",
    averageScore: 65.8,
    difficulty: "hard",
    commonMistakes: ["Incomplete proof", "Wrong convergence test"],
    timeSpent: 25.7,
  },
]

const ACCESSIBLE_COLORS = [
  "hsl(var(--chart-1))", // Professional blue
  "hsl(var(--chart-2))", // Vibrant orange
  "hsl(var(--chart-3))", // Success green
  "hsl(var(--chart-4))", // Alert red
  "hsl(var(--chart-5))", // Purple accent
  "hsl(var(--chart-6))", // Warm brown
  "hsl(var(--chart-7))", // Pink highlight
  "hsl(var(--chart-8))", // Neutral gray
  "hsl(var(--chart-9))", // Olive green
  "hsl(var(--chart-10))", // Cyan blue
]

const getTrendIcon = (trend: "up" | "down" | "stable") => {
  switch (trend) {
    case "up":
      return <TrendingUp className="w-4 h-4 text-green-600" />
    case "down":
      return <TrendingDown className="w-4 h-4 text-red-600" />
    case "stable":
      return <div className="w-4 h-4 bg-gray-400 rounded-full" />
  }
}

const getDifficultyColor = (difficulty: "easy" | "medium" | "hard") => {
  switch (difficulty) {
    case "easy":
      return "bg-green-100 text-green-800"
    case "medium":
      return "bg-yellow-100 text-yellow-800"
    case "hard":
      return "bg-red-100 text-red-800"
  }
}

export function AnalyticsDashboard() {
  const [selectedSubject, setSelectedSubject] = useState("all")
  const [timeRange, setTimeRange] = useState("6months")

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">Analytics Dashboard</h1>
              <p className="text-muted-foreground">Comprehensive exam performance insights and trends</p>
            </div>
            <div className="flex items-center gap-4">
              <Select value={timeRange} onValueChange={setTimeRange}>
                <SelectTrigger className="w-32">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="1month">1 Month</SelectItem>
                  <SelectItem value="3months">3 Months</SelectItem>
                  <SelectItem value="6months">6 Months</SelectItem>
                  <SelectItem value="1year">1 Year</SelectItem>
                </SelectContent>
              </Select>
              <Button variant="outline" size="sm">
                <Download className="w-4 h-4 mr-2" />
                Export
              </Button>
            </div>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-6 py-6">
        <Tabs defaultValue="overview" className="space-y-6">
          <TabsList className="grid w-full grid-cols-5">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="performance">Performance</TabsTrigger>
            <TabsTrigger value="questions">Question Analysis</TabsTrigger>
            <TabsTrigger value="trends">Trends</TabsTrigger>
            <TabsTrigger value="insights">AI Insights</TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="space-y-6">
            {/* Key Metrics */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Students</CardTitle>
                  <Users className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">176</div>
                  <p className="text-xs text-muted-foreground">+12% from last month</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Average Score</CardTitle>
                  <Target className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">76.4%</div>
                  <p className="text-xs text-muted-foreground">+2.1% from last month</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Pass Rate</CardTitle>
                  <Award className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">80.1%</div>
                  <p className="text-xs text-muted-foreground">+3.2% from last month</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Avg. Time</CardTitle>
                  <Clock className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">142m</div>
                  <p className="text-xs text-muted-foreground">-5m from last month</p>
                </CardContent>
              </Card>
            </div>

            {/* Score Distribution */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle>Score Distribution</CardTitle>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={300}>
                    <BarChart data={scoreDistributionData}>
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="range" />
                      <YAxis />
                      <Tooltip />
                      <Bar dataKey="count" fill="hsl(var(--chart-1))" />
                    </BarChart>
                  </ResponsiveContainer>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Subject Performance</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {performanceData.map((subject, index) => (
                      <div key={subject.subject} className="flex items-center justify-between">
                        <div className="flex items-center gap-3">
                          {getTrendIcon(subject.trend)}
                          <div>
                            <p className="font-medium">{subject.subject}</p>
                            <p className="text-sm text-muted-foreground">{subject.totalStudents} students</p>
                          </div>
                        </div>
                        <div className="text-right">
                          <p className="font-bold">{subject.averageScore}%</p>
                          <p className="text-sm text-muted-foreground">{subject.passRate}% pass</p>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="performance" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle>Performance Trends</CardTitle>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={300}>
                    <LineChart data={timeSeriesData}>
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="month" />
                      <YAxis />
                      <Tooltip />
                      <Legend />
                      <Line
                        type="monotone"
                        dataKey="averageScore"
                        stroke="hsl(var(--chart-1))"
                        name="Average Score"
                        strokeWidth={2}
                      />
                      <Line
                        type="monotone"
                        dataKey="passRate"
                        stroke="hsl(var(--chart-3))"
                        name="Pass Rate"
                        strokeWidth={2}
                      />
                    </LineChart>
                  </ResponsiveContainer>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Grade Distribution</CardTitle>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={300}>
                    <PieChart>
                      <Pie
                        data={scoreDistributionData}
                        cx="50%"
                        cy="50%"
                        labelLine={false}
                        label={({ range, percentage }) => `${range}: ${percentage}%`}
                        outerRadius={80}
                        fill="#8884d8"
                        dataKey="count"
                      >
                        {scoreDistributionData.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={ACCESSIBLE_COLORS[index % ACCESSIBLE_COLORS.length]} />
                        ))}
                      </Pie>
                      <Tooltip />
                    </PieChart>
                  </ResponsiveContainer>
                </CardContent>
              </Card>
            </div>

            {/* Subject Comparison */}
            <Card>
              <CardHeader>
                <CardTitle>Subject Performance Comparison</CardTitle>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={400}>
                  <BarChart data={performanceData} layout="horizontal">
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis type="number" />
                    <YAxis dataKey="subject" type="category" width={150} />
                    <Tooltip />
                    <Legend />
                    <Bar dataKey="averageScore" fill="hsl(var(--chart-1))" name="Average Score" />
                    <Bar dataKey="passRate" fill="hsl(var(--chart-3))" name="Pass Rate" />
                  </BarChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="questions" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Question-Level Analysis</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-6">
                  {questionAnalysis.map((question) => (
                    <div key={question.questionId} className="border rounded-lg p-4">
                      <div className="flex items-start justify-between mb-4">
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-2">
                            <h3 className="font-semibold">Question {question.questionId}</h3>
                            <Badge className={getDifficultyColor(question.difficulty)}>{question.difficulty}</Badge>
                          </div>
                          <p className="text-sm text-muted-foreground mb-3">{question.questionText}</p>
                        </div>
                        <div className="text-right">
                          <p className="text-2xl font-bold">{question.averageScore}%</p>
                          <p className="text-sm text-muted-foreground">Avg. Score</p>
                        </div>
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                          <h4 className="font-medium mb-2">Common Mistakes</h4>
                          <ul className="space-y-1">
                            {question.commonMistakes.map((mistake, index) => (
                              <li key={index} className="text-sm text-muted-foreground flex items-center gap-2">
                                <AlertTriangle className="w-3 h-3 text-orange-500" />
                                {mistake}
                              </li>
                            ))}
                          </ul>
                        </div>
                        <div>
                          <h4 className="font-medium mb-2">Time Analysis</h4>
                          <div className="flex items-center gap-2">
                            <Clock className="w-4 h-4 text-muted-foreground" />
                            <span className="text-sm">Average: {question.timeSpent} minutes</span>
                          </div>
                          <Progress value={(question.timeSpent / 30) * 100} className="mt-2" />
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="trends" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Long-term Trends</CardTitle>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={400}>
                  <LineChart data={timeSeriesData}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="month" />
                    <YAxis />
                    <Tooltip />
                    <Legend />
                    <Line type="monotone" dataKey="averageScore" stroke="hsl(var(--chart-1))" strokeWidth={3} />
                    <Line type="monotone" dataKey="passRate" stroke="hsl(var(--chart-3))" strokeWidth={3} />
                  </LineChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Improvement Areas</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Series Convergence</span>
                      <Badge variant="destructive">-5.2%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Complex Integration</span>
                      <Badge variant="destructive">-3.1%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Proof Techniques</span>
                      <Badge variant="destructive">-2.8%</Badge>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Strong Areas</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Basic Calculus</span>
                      <Badge className="bg-green-100 text-green-800">+8.4%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Linear Algebra</span>
                      <Badge className="bg-green-100 text-green-800">+6.2%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Derivatives</span>
                      <Badge className="bg-green-100 text-green-800">+4.7%</Badge>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Recommendations</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3 text-sm">
                    <p>• Focus on series convergence practice problems</p>
                    <p>• Provide additional proof technique workshops</p>
                    <p>• Create targeted review sessions for complex integration</p>
                    <p>• Maintain current approach for basic calculus topics</p>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="insights" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>AI-Generated Insights</CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="border-l-4 border-blue-500 pl-4">
                  <h3 className="font-semibold text-blue-700">Performance Pattern</h3>
                  <p className="text-sm text-muted-foreground mt-1">
                    Students who spend more than 20 minutes on Question 3 (Series Convergence) show 15% higher accuracy,
                    suggesting the need for more time allocation on complex proof questions.
                  </p>
                </div>

                <div className="border-l-4 border-green-500 pl-4">
                  <h3 className="font-semibold text-green-700">Success Factor</h3>
                  <p className="text-sm text-muted-foreground mt-1">
                    Students with consistent handwriting quality (OCR confidence &gt;90%) score an average of 8.3%
                    higher, indicating the importance of clear mathematical notation.
                  </p>
                </div>

                <div className="border-l-4 border-orange-500 pl-4">
                  <h3 className="font-semibold text-orange-700">Risk Indicator</h3>
                  <p className="text-sm text-muted-foreground mt-1">
                    Questions with vector retrieval cache miss rates &gt;30% show 12% lower average scores, suggesting
                    gaps in the knowledge base for certain topic areas.
                  </p>
                </div>

                <div className="border-l-4 border-purple-500 pl-4">
                  <h3 className="font-semibold text-purple-700">Optimization Opportunity</h3>
                  <p className="text-sm text-muted-foreground mt-1">
                    LLM confidence scores below 85% correlate with 23% higher appeal rates, indicating potential areas
                    for model fine-tuning and human review triggers.
                  </p>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
