"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Textarea } from "@/components/ui/textarea"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { AlertCircle, CheckCircle, Clock, Eye, FileText, Scale, Download, X, Check } from "lucide-react"

interface Appeal {
  id: string
  studentName: string
  studentEmail: string
  examSubject: string
  questionId: number
  originalScore: number
  requestedScore?: number
  status: "pending" | "under-review" | "approved" | "rejected" | "escalated"
  priority: "low" | "medium" | "high"
  submittedAt: string
  reviewedAt?: string
  reviewedBy?: string
  reason: string
  studentEvidence: string
  aiGradingDetails: {
    ocrConfidence: number
    vectorMatches: number
    llmScore: number
    llmConfidence: number
    llmExplanation: string
  }
  reviewerNotes?: string
  finalDecision?: string
}

const mockAppeals: Appeal[] = [
  {
    id: "APP-001",
    studentName: "Alice Johnson",
    studentEmail: "alice@university.edu",
    examSubject: "Advanced Mathematics",
    questionId: 3,
    originalScore: 65,
    requestedScore: 85,
    status: "pending",
    priority: "high",
    submittedAt: "2024-01-15T14:30:00Z",
    reason: "Partial credit not awarded",
    studentEvidence:
      "I believe my approach to the series convergence proof was correct, but the AI system may not have recognized the alternative method I used. I followed the Cauchy condensation test which should be equivalent to the Basel problem approach mentioned in the question.",
    aiGradingDetails: {
      ocrConfidence: 87,
      vectorMatches: 4,
      llmScore: 65,
      llmConfidence: 78,
      llmExplanation:
        "Student attempted series convergence proof but used non-standard approach. Partial credit awarded for correct setup but deducted points for incomplete justification of convergence test validity.",
    },
  },
  {
    id: "APP-002",
    studentName: "Bob Smith",
    studentEmail: "bob@university.edu",
    examSubject: "Advanced Mathematics",
    questionId: 1,
    originalScore: 72,
    status: "under-review",
    priority: "medium",
    submittedAt: "2024-01-15T13:15:00Z",
    reviewedAt: "2024-01-15T15:45:00Z",
    reviewedBy: "Dr. Sarah Wilson",
    reason: "OCR misinterpretation",
    studentEvidence:
      "The handwriting recognition system appears to have misread my mathematical notation, particularly the integral bounds and the differential equation setup.",
    aiGradingDetails: {
      ocrConfidence: 82,
      vectorMatches: 6,
      llmScore: 72,
      llmConfidence: 85,
      llmExplanation:
        "Differential equation solution shows correct method but some notation unclear. Partial credit given for approach and final answer verification.",
    },
    reviewerNotes: "Reviewing original handwritten answer. OCR did miss some notation. Checking against rubric.",
  },
  {
    id: "APP-003",
    studentName: "Carol Davis",
    studentEmail: "carol@university.edu",
    examSubject: "Physics Mechanics",
    questionId: 2,
    originalScore: 58,
    requestedScore: 75,
    status: "approved",
    priority: "medium",
    submittedAt: "2024-01-14T16:20:00Z",
    reviewedAt: "2024-01-15T09:30:00Z",
    reviewedBy: "Prof. Michael Chen",
    reason: "Alternative solution method",
    studentEvidence:
      "I used the energy conservation approach instead of the force analysis method. Both should yield the same result and deserve equal credit.",
    aiGradingDetails: {
      ocrConfidence: 94,
      vectorMatches: 3,
      llmScore: 58,
      llmConfidence: 72,
      llmExplanation:
        "Student used energy method but AI expected force analysis. Correct final answer but methodology not recognized in knowledge base.",
    },
    reviewerNotes:
      "Student is correct. Energy conservation method is valid and well-executed. Updating score to 75/100.",
    finalDecision: "Appeal approved. Score updated from 58 to 75. Alternative solution method was correctly applied.",
  },
]

export function AppealsManagementSystem() {
  const [appeals, setAppeals] = useState(mockAppeals)
  const [selectedAppeal, setSelectedAppeal] = useState<Appeal | null>(null)
  const [filterStatus, setFilterStatus] = useState("all")
  const [filterPriority, setFilterPriority] = useState("all")

  const getStatusColor = (status: Appeal["status"]) => {
    switch (status) {
      case "pending":
        return "bg-yellow-100 text-yellow-800"
      case "under-review":
        return "bg-blue-100 text-blue-800"
      case "approved":
        return "bg-green-100 text-green-800"
      case "rejected":
        return "bg-red-100 text-red-800"
      case "escalated":
        return "bg-purple-100 text-purple-800"
    }
  }

  const getPriorityColor = (priority: Appeal["priority"]) => {
    switch (priority) {
      case "low":
        return "bg-gray-100 text-gray-800"
      case "medium":
        return "bg-orange-100 text-orange-800"
      case "high":
        return "bg-red-100 text-red-800"
    }
  }

  const getStatusIcon = (status: Appeal["status"]) => {
    switch (status) {
      case "pending":
        return <Clock className="w-4 h-4" />
      case "under-review":
        return <Eye className="w-4 h-4" />
      case "approved":
        return <CheckCircle className="w-4 h-4" />
      case "rejected":
        return <X className="w-4 h-4" />
      case "escalated":
        return <AlertCircle className="w-4 h-4" />
    }
  }

  const handleStatusUpdate = (appealId: string, newStatus: Appeal["status"], reviewerNotes?: string) => {
    setAppeals((prev) =>
      prev.map((appeal) =>
        appeal.id === appealId
          ? {
              ...appeal,
              status: newStatus,
              reviewedAt: new Date().toISOString(),
              reviewedBy: "Current Reviewer",
              reviewerNotes: reviewerNotes || appeal.reviewerNotes,
            }
          : appeal,
      ),
    )
  }

  const filteredAppeals = appeals.filter((appeal) => {
    const statusMatch = filterStatus === "all" || appeal.status === filterStatus
    const priorityMatch = filterPriority === "all" || appeal.priority === filterPriority
    return statusMatch && priorityMatch
  })

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    })
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">Appeals Management System</h1>
              <p className="text-muted-foreground">Review and manage student grade appeals</p>
            </div>
            <div className="flex items-center gap-4">
              <Select value={filterStatus} onValueChange={setFilterStatus}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Filter by status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="pending">Pending</SelectItem>
                  <SelectItem value="under-review">Under Review</SelectItem>
                  <SelectItem value="approved">Approved</SelectItem>
                  <SelectItem value="rejected">Rejected</SelectItem>
                  <SelectItem value="escalated">Escalated</SelectItem>
                </SelectContent>
              </Select>
              <Select value={filterPriority} onValueChange={setFilterPriority}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Filter by priority" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Priority</SelectItem>
                  <SelectItem value="high">High</SelectItem>
                  <SelectItem value="medium">Medium</SelectItem>
                  <SelectItem value="low">Low</SelectItem>
                </SelectContent>
              </Select>
              <Button variant="outline" size="sm">
                <Download className="w-4 h-4 mr-2" />
                Export
              </Button>
            </div>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-6 py-6">
        <Tabs defaultValue="queue" className="space-y-6">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="queue">Appeal Queue</TabsTrigger>
            <TabsTrigger value="review">Under Review</TabsTrigger>
            <TabsTrigger value="completed">Completed</TabsTrigger>
            <TabsTrigger value="statistics">Statistics</TabsTrigger>
          </TabsList>

          <TabsContent value="queue" className="space-y-6">
            {/* Stats Cards */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Appeals</CardTitle>
                  <Scale className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{appeals.length}</div>
                  <p className="text-xs text-muted-foreground">This month</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Pending</CardTitle>
                  <Clock className="h-4 w-4 text-yellow-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-yellow-600">
                    {appeals.filter((a) => a.status === "pending").length}
                  </div>
                  <p className="text-xs text-muted-foreground">Awaiting review</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Approved</CardTitle>
                  <CheckCircle className="h-4 w-4 text-green-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-green-600">
                    {appeals.filter((a) => a.status === "approved").length}
                  </div>
                  <p className="text-xs text-muted-foreground">Score updated</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">High Priority</CardTitle>
                  <AlertCircle className="h-4 w-4 text-red-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-red-600">
                    {appeals.filter((a) => a.priority === "high").length}
                  </div>
                  <p className="text-xs text-muted-foreground">Urgent review</p>
                </CardContent>
              </Card>
            </div>

            {/* Appeals List */}
            <Card>
              <CardHeader>
                <CardTitle>Appeal Queue</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {filteredAppeals.map((appeal) => (
                    <div key={appeal.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <Avatar>
                          <AvatarImage src={`/generic-placeholder-graphic.png?key=${appeal.id}&height=40&width=40`} />
                          <AvatarFallback>
                            {appeal.studentName
                              .split(" ")
                              .map((n) => n[0])
                              .join("")}
                          </AvatarFallback>
                        </Avatar>
                        <div>
                          <h3 className="font-semibold">{appeal.studentName}</h3>
                          <p className="text-sm text-muted-foreground">
                            {appeal.examSubject} - Question {appeal.questionId}
                          </p>
                          <p className="text-xs text-muted-foreground">Submitted {formatDate(appeal.submittedAt)}</p>
                        </div>
                      </div>

                      <div className="flex items-center gap-6">
                        <div className="text-center">
                          <p className="text-lg font-bold">{appeal.originalScore}%</p>
                          <p className="text-xs text-muted-foreground">Current</p>
                        </div>

                        {appeal.requestedScore && (
                          <div className="text-center">
                            <p className="text-lg font-bold text-blue-600">{appeal.requestedScore}%</p>
                            <p className="text-xs text-muted-foreground">Requested</p>
                          </div>
                        )}

                        <div className="flex flex-col gap-2">
                          <Badge className={getStatusColor(appeal.status)}>
                            {getStatusIcon(appeal.status)}
                            <span className="ml-1">{appeal.status.replace("-", " ")}</span>
                          </Badge>
                          <Badge className={getPriorityColor(appeal.priority)}>{appeal.priority}</Badge>
                        </div>

                        <Dialog>
                          <DialogTrigger asChild>
                            <Button variant="outline" size="sm" onClick={() => setSelectedAppeal(appeal)}>
                              <Eye className="w-4 h-4 mr-2" />
                              Review
                            </Button>
                          </DialogTrigger>
                          <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
                            <DialogHeader>
                              <DialogTitle>Appeal Review - {appeal.id}</DialogTitle>
                            </DialogHeader>

                            {selectedAppeal && (
                              <div className="space-y-6">
                                {/* Student Information */}
                                <div className="grid grid-cols-2 gap-4">
                                  <div>
                                    <h3 className="font-semibold mb-2">Student Information</h3>
                                    <p>
                                      <strong>Name:</strong> {selectedAppeal.studentName}
                                    </p>
                                    <p>
                                      <strong>Email:</strong> {selectedAppeal.studentEmail}
                                    </p>
                                    <p>
                                      <strong>Subject:</strong> {selectedAppeal.examSubject}
                                    </p>
                                    <p>
                                      <strong>Question:</strong> {selectedAppeal.questionId}
                                    </p>
                                  </div>
                                  <div>
                                    <h3 className="font-semibold mb-2">Appeal Details</h3>
                                    <p>
                                      <strong>Current Score:</strong> {selectedAppeal.originalScore}%
                                    </p>
                                    {selectedAppeal.requestedScore && (
                                      <p>
                                        <strong>Requested Score:</strong> {selectedAppeal.requestedScore}%
                                      </p>
                                    )}
                                    <p>
                                      <strong>Priority:</strong> {selectedAppeal.priority}
                                    </p>
                                    <p>
                                      <strong>Submitted:</strong> {formatDate(selectedAppeal.submittedAt)}
                                    </p>
                                  </div>
                                </div>

                                {/* Student Evidence */}
                                <div>
                                  <h3 className="font-semibold mb-2">Student's Reason for Appeal</h3>
                                  <p className="text-sm mb-2">
                                    <strong>Category:</strong> {selectedAppeal.reason}
                                  </p>
                                  <div className="bg-muted p-3 rounded-lg">
                                    <p className="text-sm">{selectedAppeal.studentEvidence}</p>
                                  </div>
                                </div>

                                {/* AI Grading Details */}
                                <div>
                                  <h3 className="font-semibold mb-2">AI Grading Analysis</h3>
                                  <div className="grid grid-cols-2 gap-4 mb-3">
                                    <div>
                                      <p className="text-sm">
                                        <strong>OCR Confidence:</strong> {selectedAppeal.aiGradingDetails.ocrConfidence}
                                        %
                                      </p>
                                      <p className="text-sm">
                                        <strong>Vector Matches:</strong> {selectedAppeal.aiGradingDetails.vectorMatches}
                                      </p>
                                    </div>
                                    <div>
                                      <p className="text-sm">
                                        <strong>LLM Score:</strong> {selectedAppeal.aiGradingDetails.llmScore}%
                                      </p>
                                      <p className="text-sm">
                                        <strong>LLM Confidence:</strong> {selectedAppeal.aiGradingDetails.llmConfidence}
                                        %
                                      </p>
                                    </div>
                                  </div>
                                  <div className="bg-muted p-3 rounded-lg">
                                    <p className="text-sm">
                                      <strong>AI Explanation:</strong>
                                    </p>
                                    <p className="text-sm mt-1">{selectedAppeal.aiGradingDetails.llmExplanation}</p>
                                  </div>
                                </div>

                                {/* Reviewer Section */}
                                <div>
                                  <h3 className="font-semibold mb-2">Reviewer Notes</h3>
                                  <Textarea
                                    placeholder="Add your review notes here..."
                                    value={selectedAppeal.reviewerNotes || ""}
                                    onChange={(e) => {
                                      if (selectedAppeal) {
                                        setSelectedAppeal({
                                          ...selectedAppeal,
                                          reviewerNotes: e.target.value,
                                        })
                                      }
                                    }}
                                    className="mb-3"
                                  />

                                  <div className="flex gap-2">
                                    <Button
                                      onClick={() =>
                                        handleStatusUpdate(selectedAppeal.id, "approved", selectedAppeal.reviewerNotes)
                                      }
                                      className="bg-green-600 hover:bg-green-700"
                                    >
                                      <Check className="w-4 h-4 mr-2" />
                                      Approve Appeal
                                    </Button>
                                    <Button
                                      onClick={() =>
                                        handleStatusUpdate(selectedAppeal.id, "rejected", selectedAppeal.reviewerNotes)
                                      }
                                      variant="destructive"
                                    >
                                      <X className="w-4 h-4 mr-2" />
                                      Reject Appeal
                                    </Button>
                                    <Button
                                      onClick={() =>
                                        handleStatusUpdate(selectedAppeal.id, "escalated", selectedAppeal.reviewerNotes)
                                      }
                                      variant="outline"
                                    >
                                      <AlertCircle className="w-4 h-4 mr-2" />
                                      Escalate
                                    </Button>
                                  </div>
                                </div>
                              </div>
                            )}
                          </DialogContent>
                        </Dialog>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="review" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Appeals Under Review</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {appeals
                    .filter((a) => a.status === "under-review")
                    .map((appeal) => (
                      <div key={appeal.id} className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="flex items-center gap-4">
                          <Avatar>
                            <AvatarImage src={`/generic-placeholder-graphic.png?key=${appeal.id}&height=40&width=40`} />
                            <AvatarFallback>
                              {appeal.studentName
                                .split(" ")
                                .map((n) => n[0])
                                .join("")}
                            </AvatarFallback>
                          </Avatar>
                          <div>
                            <h3 className="font-semibold">{appeal.studentName}</h3>
                            <p className="text-sm text-muted-foreground">
                              {appeal.examSubject} - Question {appeal.questionId}
                            </p>
                            <p className="text-xs text-muted-foreground">
                              Reviewed by {appeal.reviewedBy} • {formatDate(appeal.reviewedAt!)}
                            </p>
                          </div>
                        </div>
                        <div className="flex items-center gap-4">
                          <Badge className={getStatusColor(appeal.status)}>Under Review</Badge>
                          <Button variant="outline" size="sm">
                            <Eye className="w-4 h-4 mr-2" />
                            Continue Review
                          </Button>
                        </div>
                      </div>
                    ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="completed" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Completed Appeals</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {appeals
                    .filter((a) => a.status === "approved" || a.status === "rejected")
                    .map((appeal) => (
                      <div key={appeal.id} className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="flex items-center gap-4">
                          <Avatar>
                            <AvatarImage src={`/generic-placeholder-graphic.png?key=${appeal.id}&height=40&width=40`} />
                            <AvatarFallback>
                              {appeal.studentName
                                .split(" ")
                                .map((n) => n[0])
                                .join("")}
                            </AvatarFallback>
                          </Avatar>
                          <div>
                            <h3 className="font-semibold">{appeal.studentName}</h3>
                            <p className="text-sm text-muted-foreground">
                              {appeal.examSubject} - Question {appeal.questionId}
                            </p>
                            {appeal.finalDecision && (
                              <p className="text-xs text-muted-foreground mt-1">{appeal.finalDecision}</p>
                            )}
                          </div>
                        </div>
                        <div className="flex items-center gap-4">
                          <Badge className={getStatusColor(appeal.status)}>
                            {appeal.status === "approved" ? "Approved" : "Rejected"}
                          </Badge>
                          <Button variant="outline" size="sm">
                            <FileText className="w-4 h-4 mr-2" />
                            View Details
                          </Button>
                        </div>
                      </div>
                    ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="statistics" className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Appeal Outcomes</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Approved</span>
                      <Badge className="bg-green-100 text-green-800">33%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Rejected</span>
                      <Badge className="bg-red-100 text-red-800">0%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Pending</span>
                      <Badge className="bg-yellow-100 text-yellow-800">67%</Badge>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Common Reasons</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">OCR Issues</span>
                      <Badge variant="outline">33%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Alternative Methods</span>
                      <Badge variant="outline">33%</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Partial Credit</span>
                      <Badge variant="outline">34%</Badge>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Review Metrics</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div>
                      <span className="text-sm text-muted-foreground">Avg. Review Time</span>
                      <p className="font-semibold">2.3 hours</p>
                    </div>
                    <div>
                      <span className="text-sm text-muted-foreground">Score Change (Avg.)</span>
                      <p className="font-semibold">+8.5 points</p>
                    </div>
                    <div>
                      <span className="text-sm text-muted-foreground">Appeals This Month</span>
                      <p className="font-semibold">3</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
