"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Progress } from "@/components/ui/progress"
import { Checkbox } from "@/components/ui/checkbox"
import { ScrollArea } from "@/components/ui/scroll-area"
import { Brain, Play, Pause, CheckCircle, Clock, AlertCircle } from "lucide-react"

interface BatchGradingItem {
  id: string
  studentName: string
  question: string
  subject: "History" | "Geography" | "Political Science"
  status: "pending" | "processing" | "completed" | "error"
  estimatedTime: number
  actualTime?: number
  marks?: number
  totalMarks: number
}

interface BatchGradingInterfaceProps {
  items: BatchGradingItem[]
  onStartBatch: (selectedIds: string[]) => void
  onPauseBatch: () => void
  isProcessing: boolean
}

export function BatchGradingInterface({ items, onStartBatch, onPauseBatch, isProcessing }: BatchGradingInterfaceProps) {
  const [selectedItems, setSelectedItems] = useState<string[]>([])
  const [currentProcessing, setCurrentProcessing] = useState<string | null>(null)

  const completedItems = items.filter((item) => item.status === "completed")
  const pendingItems = items.filter((item) => item.status === "pending")
  const processingItems = items.filter((item) => item.status === "processing")
  const errorItems = items.filter((item) => item.status === "error")

  const totalEstimatedTime = selectedItems.reduce((total, id) => {
    const item = items.find((i) => i.id === id)
    return total + (item?.estimatedTime || 0)
  }, 0)

  const getStatusIcon = (status: BatchGradingItem["status"]) => {
    switch (status) {
      case "pending":
        return <Clock className="w-4 h-4 text-gray-500" />
      case "processing":
        return <Brain className="w-4 h-4 text-blue-500 animate-pulse" />
      case "completed":
        return <CheckCircle className="w-4 h-4 text-green-500" />
      case "error":
        return <AlertCircle className="w-4 h-4 text-red-500" />
    }
  }

  const getStatusColor = (status: BatchGradingItem["status"]) => {
    switch (status) {
      case "pending":
        return "bg-gray-100 text-gray-800"
      case "processing":
        return "bg-blue-100 text-blue-800"
      case "completed":
        return "bg-green-100 text-green-800"
      case "error":
        return "bg-red-100 text-red-800"
    }
  }

  const handleSelectAll = () => {
    if (selectedItems.length === pendingItems.length) {
      setSelectedItems([])
    } else {
      setSelectedItems(pendingItems.map((item) => item.id))
    }
  }

  const handleItemSelect = (itemId: string) => {
    setSelectedItems((prev) => (prev.includes(itemId) ? prev.filter((id) => id !== itemId) : [...prev, itemId]))
  }

  return (
    <div className="space-y-6">
      {/* Batch Control Panel */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <span className="flex items-center gap-2">
              <Brain className="w-5 h-5" />
              Batch AI Grading
            </span>
            <div className="flex items-center gap-2">
              {isProcessing ? (
                <Button variant="outline" onClick={onPauseBatch}>
                  <Pause className="w-4 h-4 mr-2" />
                  Pause
                </Button>
              ) : (
                <Button onClick={() => onStartBatch(selectedItems)} disabled={selectedItems.length === 0}>
                  <Play className="w-4 h-4 mr-2" />
                  Start Grading ({selectedItems.length})
                </Button>
              )}
            </div>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
            <div className="text-center">
              <div className="text-2xl font-bold text-gray-600">{pendingItems.length}</div>
              <div className="text-sm text-muted-foreground">Pending</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">{processingItems.length}</div>
              <div className="text-sm text-muted-foreground">Processing</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">{completedItems.length}</div>
              <div className="text-sm text-muted-foreground">Completed</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-red-600">{errorItems.length}</div>
              <div className="text-sm text-muted-foreground">Errors</div>
            </div>
          </div>

          {selectedItems.length > 0 && (
            <div className="p-4 bg-muted rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="font-medium">{selectedItems.length} items selected</p>
                  <p className="text-sm text-muted-foreground">
                    Estimated time: {Math.ceil(totalEstimatedTime / 60)} minutes
                  </p>
                </div>
                <div className="text-right">
                  <p className="text-sm text-muted-foreground">Total marks to grade:</p>
                  <p className="font-bold">
                    {selectedItems.reduce((total, id) => {
                      const item = items.find((i) => i.id === id)
                      return total + (item?.totalMarks || 0)
                    }, 0)}{" "}
                    marks
                  </p>
                </div>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Progress Overview */}
      {isProcessing && (
        <Card>
          <CardHeader>
            <CardTitle>Grading Progress</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex items-center justify-between text-sm">
                <span>Overall Progress</span>
                <span>
                  {completedItems.length}/{items.length} completed
                </span>
              </div>
              <Progress value={(completedItems.length / items.length) * 100} className="h-3" />

              {currentProcessing && (
                <div className="p-3 bg-blue-50 border border-blue-200 rounded">
                  <div className="flex items-center gap-2">
                    <Brain className="w-4 h-4 text-blue-600 animate-pulse" />
                    <span className="text-sm font-medium">Currently processing:</span>
                  </div>
                  <p className="text-sm text-blue-700 mt-1">
                    {items.find((item) => item.id === currentProcessing)?.studentName} -
                    {items.find((item) => item.id === currentProcessing)?.subject}
                  </p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Items List */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <span>Grading Queue</span>
            <div className="flex items-center gap-2">
              <Checkbox
                checked={selectedItems.length === pendingItems.length && pendingItems.length > 0}
                onCheckedChange={handleSelectAll}
              />
              <span className="text-sm">Select All Pending</span>
            </div>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <ScrollArea className="h-96">
            <div className="space-y-2">
              {items.map((item) => (
                <div
                  key={item.id}
                  className={`p-4 border rounded-lg transition-colors ${
                    selectedItems.includes(item.id) ? "border-primary bg-primary/5" : ""
                  }`}
                >
                  <div className="flex items-center gap-4">
                    {item.status === "pending" && (
                      <Checkbox
                        checked={selectedItems.includes(item.id)}
                        onCheckedChange={() => handleItemSelect(item.id)}
                      />
                    )}

                    <div className="flex-1">
                      <div className="flex items-center justify-between mb-2">
                        <div className="flex items-center gap-2">
                          <h4 className="font-medium">{item.studentName}</h4>
                          <Badge variant="outline">{item.subject}</Badge>
                          <Badge className={getStatusColor(item.status)}>
                            {getStatusIcon(item.status)}
                            <span className="ml-1">{item.status}</span>
                          </Badge>
                        </div>

                        <div className="text-right">
                          {item.marks !== undefined ? (
                            <div>
                              <span className="font-bold">
                                {item.marks}/{item.totalMarks}
                              </span>
                              <div className="text-xs text-muted-foreground">
                                {item.actualTime ? `${item.actualTime}s` : `~${item.estimatedTime}s`}
                              </div>
                            </div>
                          ) : (
                            <div className="text-sm text-muted-foreground">
                              {item.totalMarks} marks • ~{item.estimatedTime}s
                            </div>
                          )}
                        </div>
                      </div>

                      <p className="text-sm text-muted-foreground line-clamp-2">{item.question}</p>

                      {item.status === "processing" && (
                        <div className="mt-2">
                          <Progress value={50} className="h-1" />
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </ScrollArea>
        </CardContent>
      </Card>
    </div>
  )
}
