"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog"
import {
  Play,
  Pause,
  Square,
  RotateCcw,
  Download,
  Upload,
  Share2,
  Settings,
  HelpCircle,
  Maximize2,
  Minimize2,
  Volume2,
  VolumeX,
  Bell,
  BellOff,
  Eye,
  EyeOff,
  Lock,
  Unlock,
  RefreshCw,
  CheckCircle,
  AlertTriangle,
  Info,
} from "lucide-react"

interface EnhancedButtonControlsProps {
  onAction?: (action: string, data?: any) => void
  disabled?: boolean
  context?: "student" | "teacher" | "admin"
}

export function EnhancedButtonControls({
  onAction,
  disabled = false,
  context = "student",
}: EnhancedButtonControlsProps) {
  const [isRecording, setIsRecording] = useState(false)
  const [isFullscreen, setIsFullscreen] = useState(false)
  const [audioEnabled, setAudioEnabled] = useState(true)
  const [notificationsEnabled, setNotificationsEnabled] = useState(true)
  const [isLocked, setIsLocked] = useState(false)
  const [isVisible, setIsVisible] = useState(true)
  const [isLoading, setIsLoading] = useState(false)

  const handleAction = async (action: string, data?: any) => {
    console.log(`[v0] Button action triggered: ${action}`, data)
    setIsLoading(true)

    try {
      // Simulate async operation
      await new Promise((resolve) => setTimeout(resolve, 1000))

      switch (action) {
        case "record":
          setIsRecording(!isRecording)
          break
        case "fullscreen":
          setIsFullscreen(!isFullscreen)
          break
        case "audio":
          setAudioEnabled(!audioEnabled)
          break
        case "notifications":
          setNotificationsEnabled(!notificationsEnabled)
          break
        case "lock":
          setIsLocked(!isLocked)
          break
        case "visibility":
          setIsVisible(!isVisible)
          break
        case "reset":
          // Reset all states
          setIsRecording(false)
          setIsFullscreen(false)
          setAudioEnabled(true)
          setNotificationsEnabled(true)
          setIsLocked(false)
          setIsVisible(true)
          break
      }

      onAction?.(action, data)
    } catch (error) {
      console.error(`[v0] Error executing action ${action}:`, error)
    } finally {
      setIsLoading(false)
    }
  }

  const getButtonVariant = (isActive: boolean) => (isActive ? "default" : "outline")
  const getStatusColor = (isActive: boolean) => (isActive ? "text-green-600" : "text-muted-foreground")

  return (
    <TooltipProvider>
      <Card className="glass-effect">
        <CardHeader className="pb-4">
          <div className="flex items-center justify-between">
            <CardTitle className="text-lg">Enhanced Controls</CardTitle>
            <Badge variant={disabled ? "secondary" : "default"}>{disabled ? "Disabled" : "Active"}</Badge>
          </div>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Recording Controls */}
          <div className="space-y-3">
            <h4 className="font-medium text-sm">Recording & Monitoring</h4>
            <div className="flex flex-wrap gap-2">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant={getButtonVariant(isRecording)}
                    size="sm"
                    onClick={() => handleAction("record")}
                    disabled={disabled || isLoading}
                    className="flex items-center gap-2"
                  >
                    {isRecording ? (
                      <>
                        <Square className="h-4 w-4" />
                        Stop Recording
                      </>
                    ) : (
                      <>
                        <Play className="h-4 w-4" />
                        Start Recording
                      </>
                    )}
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>{isRecording ? "Stop screen recording" : "Start screen recording"}</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleAction("pause")}
                    disabled={disabled || !isRecording || isLoading}
                  >
                    <Pause className="h-4 w-4" />
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Pause recording</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant={getButtonVariant(isVisible)}
                    size="sm"
                    onClick={() => handleAction("visibility")}
                    disabled={disabled || isLoading}
                  >
                    {isVisible ? <Eye className="h-4 w-4" /> : <EyeOff className="h-4 w-4" />}
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>{isVisible ? "Hide interface" : "Show interface"}</p>
                </TooltipContent>
              </Tooltip>
            </div>
          </div>

          {/* Display Controls */}
          <div className="space-y-3">
            <h4 className="font-medium text-sm">Display & Audio</h4>
            <div className="flex flex-wrap gap-2">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant={getButtonVariant(isFullscreen)}
                    size="sm"
                    onClick={() => handleAction("fullscreen")}
                    disabled={disabled || isLoading}
                  >
                    {isFullscreen ? (
                      <>
                        <Minimize2 className="h-4 w-4 mr-2" />
                        Exit Fullscreen
                      </>
                    ) : (
                      <>
                        <Maximize2 className="h-4 w-4 mr-2" />
                        Fullscreen
                      </>
                    )}
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>{isFullscreen ? "Exit fullscreen mode" : "Enter fullscreen mode"}</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant={getButtonVariant(audioEnabled)}
                    size="sm"
                    onClick={() => handleAction("audio")}
                    disabled={disabled || isLoading}
                  >
                    {audioEnabled ? <Volume2 className="h-4 w-4" /> : <VolumeX className="h-4 w-4" />}
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>{audioEnabled ? "Mute audio" : "Unmute audio"}</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant={getButtonVariant(notificationsEnabled)}
                    size="sm"
                    onClick={() => handleAction("notifications")}
                    disabled={disabled || isLoading}
                  >
                    {notificationsEnabled ? <Bell className="h-4 w-4" /> : <BellOff className="h-4 w-4" />}
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>{notificationsEnabled ? "Disable notifications" : "Enable notifications"}</p>
                </TooltipContent>
              </Tooltip>
            </div>
          </div>

          {/* File Operations */}
          <div className="space-y-3">
            <h4 className="font-medium text-sm">File Operations</h4>
            <div className="flex flex-wrap gap-2">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleAction("download")}
                    disabled={disabled || isLoading}
                  >
                    <Download className="h-4 w-4 mr-2" />
                    Download
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Download current work</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleAction("upload")}
                    disabled={disabled || isLoading}
                  >
                    <Upload className="h-4 w-4 mr-2" />
                    Upload
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Upload files</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleAction("share")}
                    disabled={disabled || isLoading}
                  >
                    <Share2 className="h-4 w-4 mr-2" />
                    Share
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Share with teacher</p>
                </TooltipContent>
              </Tooltip>
            </div>
          </div>

          {/* Security & System */}
          <div className="space-y-3">
            <h4 className="font-medium text-sm">Security & System</h4>
            <div className="flex flex-wrap gap-2">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant={getButtonVariant(isLocked)}
                    size="sm"
                    onClick={() => handleAction("lock")}
                    disabled={disabled || isLoading}
                  >
                    {isLocked ? (
                      <>
                        <Unlock className="h-4 w-4 mr-2" />
                        Unlock
                      </>
                    ) : (
                      <>
                        <Lock className="h-4 w-4 mr-2" />
                        Lock
                      </>
                    )}
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>{isLocked ? "Unlock interface" : "Lock interface"}</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleAction("refresh")}
                    disabled={disabled || isLoading}
                  >
                    <RefreshCw className={`h-4 w-4 mr-2 ${isLoading ? "animate-spin" : ""}`} />
                    Refresh
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Refresh connection</p>
                </TooltipContent>
              </Tooltip>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleAction("settings")}
                    disabled={disabled || isLoading}
                  >
                    <Settings className="h-4 w-4 mr-2" />
                    Settings
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Open settings</p>
                </TooltipContent>
              </Tooltip>
            </div>
          </div>

          {/* Action Buttons with Confirmations */}
          <div className="space-y-3">
            <h4 className="font-medium text-sm">Actions</h4>
            <div className="flex flex-wrap gap-2">
              <AlertDialog>
                <AlertDialogTrigger asChild>
                  <Button variant="outline" size="sm" disabled={disabled || isLoading}>
                    <RotateCcw className="h-4 w-4 mr-2" />
                    Reset
                  </Button>
                </AlertDialogTrigger>
                <AlertDialogContent>
                  <AlertDialogHeader>
                    <AlertDialogTitle>Reset All Settings</AlertDialogTitle>
                    <AlertDialogDescription>
                      This will reset all interface settings to their default values. This action cannot be undone.
                    </AlertDialogDescription>
                  </AlertDialogHeader>
                  <AlertDialogFooter>
                    <AlertDialogCancel>Cancel</AlertDialogCancel>
                    <AlertDialogAction onClick={() => handleAction("reset")}>Reset</AlertDialogAction>
                  </AlertDialogFooter>
                </AlertDialogContent>
              </AlertDialog>

              <Tooltip>
                <TooltipTrigger asChild>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleAction("help")}
                    disabled={disabled || isLoading}
                  >
                    <HelpCircle className="h-4 w-4 mr-2" />
                    Help
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Get help and support</p>
                </TooltipContent>
              </Tooltip>
            </div>
          </div>

          {/* Status Indicators */}
          <div className="space-y-3">
            <h4 className="font-medium text-sm">Status</h4>
            <div className="grid grid-cols-2 gap-3 text-xs">
              <div className="flex items-center gap-2">
                <CheckCircle className={`h-3 w-3 ${getStatusColor(isRecording)}`} />
                <span>Recording: {isRecording ? "Active" : "Inactive"}</span>
              </div>
              <div className="flex items-center gap-2">
                <CheckCircle className={`h-3 w-3 ${getStatusColor(audioEnabled)}`} />
                <span>Audio: {audioEnabled ? "On" : "Off"}</span>
              </div>
              <div className="flex items-center gap-2">
                <CheckCircle className={`h-3 w-3 ${getStatusColor(notificationsEnabled)}`} />
                <span>Notifications: {notificationsEnabled ? "On" : "Off"}</span>
              </div>
              <div className="flex items-center gap-2">
                <CheckCircle className={`h-3 w-3 ${getStatusColor(!isLocked)}`} />
                <span>Interface: {isLocked ? "Locked" : "Unlocked"}</span>
              </div>
            </div>
          </div>

          {/* Context-specific buttons */}
          {context === "teacher" && (
            <div className="space-y-3">
              <h4 className="font-medium text-sm">Teacher Controls</h4>
              <div className="flex flex-wrap gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleAction("monitor-all")}
                  disabled={disabled || isLoading}
                >
                  <Eye className="h-4 w-4 mr-2" />
                  Monitor All Students
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleAction("send-alert")}
                  disabled={disabled || isLoading}
                >
                  <AlertTriangle className="h-4 w-4 mr-2" />
                  Send Alert
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleAction("grade-all")}
                  disabled={disabled || isLoading}
                >
                  <CheckCircle className="h-4 w-4 mr-2" />
                  Grade All
                </Button>
              </div>
            </div>
          )}

          {context === "admin" && (
            <div className="space-y-3">
              <h4 className="font-medium text-sm">Admin Controls</h4>
              <div className="flex flex-wrap gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleAction("system-status")}
                  disabled={disabled || isLoading}
                >
                  <Info className="h-4 w-4 mr-2" />
                  System Status
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleAction("backup")}
                  disabled={disabled || isLoading}
                >
                  <Download className="h-4 w-4 mr-2" />
                  Backup Data
                </Button>
              </div>
            </div>
          )}
        </CardContent>
      </Card>
    </TooltipProvider>
  )
}
