"use client"

import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Progress } from "@/components/ui/progress"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Brain, User, BarChart3, TrendingUp, TrendingDown, Minus } from "lucide-react"

interface GradeComparison {
  studentName: string
  question: string
  subject: "History" | "Geography" | "Political Science"
  humanGrade: {
    marks: number
    totalMarks: number
    feedback: string
    gradedBy: string
  }
  aiGrade: {
    marks: number
    totalMarks: number
    feedback: string
    confidence: number
  }
  finalGrade: {
    marks: number
    totalMarks: number
    method: "human" | "ai" | "hybrid"
  }
}

interface GradeComparisonViewProps {
  comparison: GradeComparison
}

export function GradeComparisonView({ comparison }: GradeComparisonViewProps) {
  const humanPercentage = (comparison.humanGrade.marks / comparison.humanGrade.totalMarks) * 100
  const aiPercentage = (comparison.aiGrade.marks / comparison.aiGrade.totalMarks) * 100
  const finalPercentage = (comparison.finalGrade.marks / comparison.finalGrade.totalMarks) * 100

  const difference = aiPercentage - humanPercentage
  const getDifferenceIcon = () => {
    if (Math.abs(difference) < 5) return <Minus className="w-4 h-4 text-gray-600" />
    if (difference > 0) return <TrendingUp className="w-4 h-4 text-green-600" />
    return <TrendingDown className="w-4 h-4 text-red-600" />
  }

  const getDifferenceColor = () => {
    if (Math.abs(difference) < 5) return "text-gray-600"
    if (difference > 0) return "text-green-600"
    return "text-red-600"
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          <span>Grade Comparison - {comparison.studentName}</span>
          <Badge variant="outline">{comparison.subject}</Badge>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-6">
          {/* Question Display */}
          <div className="p-4 bg-muted rounded-lg">
            <h4 className="font-medium mb-2">Question:</h4>
            <p className="text-sm text-muted-foreground">{comparison.question}</p>
          </div>

          <Tabs defaultValue="comparison" className="space-y-4">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="comparison">Comparison</TabsTrigger>
              <TabsTrigger value="human">Human Grade</TabsTrigger>
              <TabsTrigger value="ai">AI Grade</TabsTrigger>
            </TabsList>

            <TabsContent value="comparison" className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                {/* Human Grade */}
                <Card>
                  <CardHeader className="pb-3">
                    <CardTitle className="text-base flex items-center gap-2">
                      <User className="w-4 h-4" />
                      Human Grade
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-center">
                      <div className="text-2xl font-bold text-blue-600">
                        {comparison.humanGrade.marks}/{comparison.humanGrade.totalMarks}
                      </div>
                      <div className="text-sm text-muted-foreground">{humanPercentage.toFixed(1)}%</div>
                      <Progress value={humanPercentage} className="mt-2 h-2" />
                      <div className="text-xs text-muted-foreground mt-2">By: {comparison.humanGrade.gradedBy}</div>
                    </div>
                  </CardContent>
                </Card>

                {/* AI Grade */}
                <Card>
                  <CardHeader className="pb-3">
                    <CardTitle className="text-base flex items-center gap-2">
                      <Brain className="w-4 h-4" />
                      AI Grade
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-center">
                      <div className="text-2xl font-bold text-purple-600">
                        {comparison.aiGrade.marks}/{comparison.aiGrade.totalMarks}
                      </div>
                      <div className="text-sm text-muted-foreground">{aiPercentage.toFixed(1)}%</div>
                      <Progress value={aiPercentage} className="mt-2 h-2" />
                      <div className="text-xs text-muted-foreground mt-2">
                        Confidence: {comparison.aiGrade.confidence}%
                      </div>
                    </div>
                  </CardContent>
                </Card>

                {/* Final Grade */}
                <Card>
                  <CardHeader className="pb-3">
                    <CardTitle className="text-base flex items-center gap-2">
                      <BarChart3 className="w-4 h-4" />
                      Final Grade
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-center">
                      <div className="text-2xl font-bold text-green-600">
                        {comparison.finalGrade.marks}/{comparison.finalGrade.totalMarks}
                      </div>
                      <div className="text-sm text-muted-foreground">{finalPercentage.toFixed(1)}%</div>
                      <Progress value={finalPercentage} className="mt-2 h-2" />
                      <Badge variant="outline" className="mt-2">
                        {comparison.finalGrade.method}
                      </Badge>
                    </div>
                  </CardContent>
                </Card>
              </div>

              {/* Difference Analysis */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Grade Analysis</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="flex items-center justify-between p-4 bg-muted rounded-lg">
                    <div className="flex items-center gap-2">
                      {getDifferenceIcon()}
                      <span className="font-medium">Grade Difference:</span>
                    </div>
                    <div className={`font-bold ${getDifferenceColor()}`}>
                      {difference > 0 ? "+" : ""}
                      {difference.toFixed(1)}%
                    </div>
                  </div>

                  <div className="mt-4 space-y-2">
                    <div className="text-sm">
                      <strong>Agreement Level:</strong>
                      <span
                        className={`ml-2 ${Math.abs(difference) < 10 ? "text-green-600" : Math.abs(difference) < 20 ? "text-yellow-600" : "text-red-600"}`}
                      >
                        {Math.abs(difference) < 10
                          ? "High Agreement"
                          : Math.abs(difference) < 20
                            ? "Moderate Agreement"
                            : "Low Agreement"}
                      </span>
                    </div>

                    {Math.abs(difference) > 15 && (
                      <div className="p-3 bg-yellow-50 border border-yellow-200 rounded text-sm">
                        <strong>Review Recommended:</strong> Significant difference detected. Consider manual review.
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="human" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <User className="w-5 h-5" />
                    Human Evaluation
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <label className="text-sm font-medium">Marks Awarded</label>
                        <div className="text-2xl font-bold text-blue-600">
                          {comparison.humanGrade.marks}/{comparison.humanGrade.totalMarks}
                        </div>
                      </div>
                      <div>
                        <label className="text-sm font-medium">Percentage</label>
                        <div className="text-2xl font-bold">{humanPercentage.toFixed(1)}%</div>
                      </div>
                    </div>

                    <div>
                      <label className="text-sm font-medium">Graded By</label>
                      <p className="text-sm text-muted-foreground">{comparison.humanGrade.gradedBy}</p>
                    </div>

                    <div>
                      <label className="text-sm font-medium">Feedback</label>
                      <p className="text-sm text-muted-foreground bg-muted p-3 rounded">
                        {comparison.humanGrade.feedback}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="ai" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Brain className="w-5 h-5" />
                    AI Evaluation
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="grid grid-cols-3 gap-4">
                      <div>
                        <label className="text-sm font-medium">Marks Awarded</label>
                        <div className="text-2xl font-bold text-purple-600">
                          {comparison.aiGrade.marks}/{comparison.aiGrade.totalMarks}
                        </div>
                      </div>
                      <div>
                        <label className="text-sm font-medium">Percentage</label>
                        <div className="text-2xl font-bold">{aiPercentage.toFixed(1)}%</div>
                      </div>
                      <div>
                        <label className="text-sm font-medium">Confidence</label>
                        <div className="text-2xl font-bold text-green-600">{comparison.aiGrade.confidence}%</div>
                      </div>
                    </div>

                    <div>
                      <label className="text-sm font-medium">AI Feedback</label>
                      <p className="text-sm text-muted-foreground bg-muted p-3 rounded">
                        {comparison.aiGrade.feedback}
                      </p>
                    </div>

                    <div className="p-3 bg-purple-50 border border-purple-200 rounded">
                      <h4 className="font-medium text-purple-800 mb-2">AI Analysis Highlights</h4>
                      <ul className="text-sm text-purple-700 space-y-1">
                        <li>• Processed answer in 2.3 seconds</li>
                        <li>• Analyzed {comparison.question.split(" ").length} question words</li>
                        <li>• Evaluated against {comparison.subject} curriculum standards</li>
                        <li>• Cross-referenced with model answers database</li>
                      </ul>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </CardContent>
    </Card>
  )
}
