"use client"

import { useState, useEffect, useCallback } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Progress } from "@/components/ui/progress"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Alert, AlertDescription } from "@/components/ui/alert"
import {
  Activity,
  Brain,
  Database,
  FileText,
  CheckCircle,
  AlertTriangle,
  Clock,
  TrendingUp,
  Server,
  Eye,
  RefreshCw,
  Pause,
  Play,
  Wifi,
  WifiOff,
  Zap,
  Users,
} from "lucide-react"

interface GradingJob {
  id: string
  studentName: string
  studentId: string
  examId: string
  questionId: number
  status: "queued" | "ocr-processing" | "vector-retrieval" | "llm-scoring" | "completed" | "failed"
  progress: number
  startTime: string
  endTime?: string
  ocrConfidence?: number
  vectorMatches?: number
  llmScore?: number
  llmConfidence?: number
  processingTime?: number
  errorMessage?: string
}

interface PipelineStats {
  totalJobs: number
  completed: number
  failed: number
  processing: number
  queued: number
  avgProcessingTime: number
  ocrAccuracy: number
  llmConfidence: number
  cacheHitRate: number
  throughputPerMinute: number
  activeConnections: number
}

interface SystemMetrics {
  cpu: number
  memory: number
  gpu: number
  diskUsage: number
  networkLatency: number
  queueDepth: number
}

const mockGradingJobs: GradingJob[] = [
  {
    id: "1",
    studentName: "Alice Johnson",
    studentId: "STU-2024-001",
    examId: "EXAM-MATH-001",
    questionId: 1,
    status: "completed",
    progress: 100,
    startTime: "2024-01-15T10:30:00Z",
    endTime: "2024-01-15T10:32:15Z",
    ocrConfidence: 94,
    vectorMatches: 8,
    llmScore: 87,
    llmConfidence: 92,
    processingTime: 135,
  },
  {
    id: "2",
    studentName: "Bob Smith",
    studentId: "STU-2024-002",
    examId: "EXAM-MATH-001",
    questionId: 1,
    status: "llm-scoring",
    progress: 75,
    startTime: "2024-01-15T10:31:00Z",
    ocrConfidence: 89,
    vectorMatches: 6,
  },
  {
    id: "3",
    studentName: "Carol Davis",
    studentId: "STU-2024-003",
    examId: "EXAM-MATH-001",
    questionId: 2,
    status: "vector-retrieval",
    progress: 50,
    startTime: "2024-01-15T10:31:30Z",
    ocrConfidence: 96,
  },
  {
    id: "4",
    studentName: "David Wilson",
    studentId: "STU-2024-004",
    examId: "EXAM-MATH-001",
    questionId: 1,
    status: "ocr-processing",
    progress: 25,
    startTime: "2024-01-15T10:32:00Z",
  },
  {
    id: "5",
    studentName: "Eva Brown",
    studentId: "STU-2024-005",
    examId: "EXAM-MATH-001",
    questionId: 3,
    status: "queued",
    progress: 0,
    startTime: "2024-01-15T10:32:15Z",
  },
]

const mockStats: PipelineStats = {
  totalJobs: 125,
  completed: 98,
  failed: 2,
  processing: 15,
  queued: 10,
  avgProcessingTime: 2.3,
  ocrAccuracy: 94.2,
  llmConfidence: 89.7,
  cacheHitRate: 76.8,
  throughputPerMinute: 12.5,
  activeConnections: 8,
}

const mockSystemMetrics: SystemMetrics = {
  cpu: 67,
  memory: 45,
  gpu: 82,
  diskUsage: 34,
  networkLatency: 23,
  queueDepth: 15,
}

export function GradingPipelineMonitor() {
  const [jobs, setJobs] = useState(mockGradingJobs)
  const [stats, setStats] = useState(mockStats)
  const [systemMetrics, setSystemMetrics] = useState(mockSystemMetrics)
  const [isRunning, setIsRunning] = useState(true)
  const [selectedJob, setSelectedJob] = useState<GradingJob | null>(null)
  const [isConnected, setIsConnected] = useState(true)
  const [logs, setLogs] = useState<Array<{ timestamp: string; level: string; message: string }>>([])

  useEffect(() => {
    if (!isRunning) return

    const interval = setInterval(() => {
      // Update jobs
      setJobs((prevJobs) =>
        prevJobs.map((job) => {
          if (job.status === "completed" || job.status === "failed") return job

          const newProgress = Math.min(job.progress + Math.random() * 15, 100)
          let newStatus = job.status

          if (newProgress >= 100) {
            newStatus = "completed"
            // Add completion log
            setLogs((prev) => [
              {
                timestamp: new Date().toLocaleTimeString(),
                level: "INFO",
                message: `Grading completed for ${job.studentName} - Question ${job.questionId} (Score: ${
                  Math.floor(Math.random() * 40) + 60
                }%)`,
              },
              ...prev.slice(0, 19),
            ])
          } else if (newProgress >= 75 && job.status !== "llm-scoring") {
            newStatus = "llm-scoring"
            setLogs((prev) => [
              {
                timestamp: new Date().toLocaleTimeString(),
                level: "INFO",
                message: `LLM scoring started for ${job.studentName} - Question ${job.questionId}`,
              },
              ...prev.slice(0, 19),
            ])
          } else if (newProgress >= 50 && job.status !== "vector-retrieval") {
            newStatus = "vector-retrieval"
            setLogs((prev) => [
              {
                timestamp: new Date().toLocaleTimeString(),
                level: "INFO",
                message: `Vector retrieval: ${Math.floor(Math.random() * 5) + 5} matches found for ${job.studentName}`,
              },
              ...prev.slice(0, 19),
            ])
          } else if (newProgress >= 25 && job.status !== "ocr-processing") {
            newStatus = "ocr-processing"
            setLogs((prev) => [
              {
                timestamp: new Date().toLocaleTimeString(),
                level: "INFO",
                message: `OCR processing started for ${job.studentName} - Question ${job.questionId}`,
              },
              ...prev.slice(0, 19),
            ])
          }

          return {
            ...job,
            progress: newProgress,
            status: newStatus,
            ...(newStatus === "completed" && {
              endTime: new Date().toISOString(),
              llmScore: Math.floor(Math.random() * 40) + 60,
              llmConfidence: Math.floor(Math.random() * 20) + 80,
              processingTime: Math.floor(Math.random() * 60) + 90,
            }),
          }
        }),
      )

      // Update stats
      setStats((prev) => ({
        ...prev,
        processing: Math.max(0, prev.processing + Math.floor(Math.random() * 3) - 1),
        completed: prev.completed + Math.floor(Math.random() * 2),
        throughputPerMinute: Math.max(8, prev.throughputPerMinute + (Math.random() - 0.5) * 2),
        activeConnections: Math.max(1, prev.activeConnections + Math.floor(Math.random() * 3) - 1),
      }))

      // Update system metrics
      setSystemMetrics((prev) => ({
        cpu: Math.max(20, Math.min(95, prev.cpu + (Math.random() - 0.5) * 10)),
        memory: Math.max(20, Math.min(90, prev.memory + (Math.random() - 0.5) * 8)),
        gpu: Math.max(30, Math.min(100, prev.gpu + (Math.random() - 0.5) * 15)),
        diskUsage: Math.max(10, Math.min(80, prev.diskUsage + (Math.random() - 0.5) * 2)),
        networkLatency: Math.max(10, Math.min(100, prev.networkLatency + (Math.random() - 0.5) * 5)),
        queueDepth: Math.max(0, Math.min(50, prev.queueDepth + Math.floor(Math.random() * 5) - 2)),
      }))

      // Simulate connection status
      setIsConnected(Math.random() > 0.05) // 95% uptime
    }, 2000)

    return () => clearInterval(interval)
  }, [isRunning])

  useEffect(() => {
    setLogs([
      { timestamp: new Date().toLocaleTimeString(), level: "INFO", message: "Pipeline monitor initialized" },
      { timestamp: new Date().toLocaleTimeString(), level: "INFO", message: "Connected to grading service" },
      { timestamp: new Date().toLocaleTimeString(), level: "INFO", message: "Real-time monitoring active" },
    ])
  }, [])

  const handleRefresh = useCallback(async () => {
    console.log("[v0] Refreshing pipeline data...")
    // In real implementation, this would fetch fresh data from API
    setLogs((prev) => [
      {
        timestamp: new Date().toLocaleTimeString(),
        level: "INFO",
        message: "Pipeline data refreshed",
      },
      ...prev.slice(0, 19),
    ])
  }, [])

  const handlePauseResume = useCallback(() => {
    setIsRunning(!isRunning)
    setLogs((prev) => [
      {
        timestamp: new Date().toLocaleTimeString(),
        level: "INFO",
        message: `Pipeline monitoring ${!isRunning ? "resumed" : "paused"}`,
      },
      ...prev.slice(0, 19),
    ])
  }, [isRunning])

  const getStatusIcon = (status: GradingJob["status"]) => {
    switch (status) {
      case "queued":
        return <Clock className="w-4 h-4" />
      case "ocr-processing":
        return <FileText className="w-4 h-4" />
      case "vector-retrieval":
        return <Database className="w-4 h-4" />
      case "llm-scoring":
        return <Brain className="w-4 h-4" />
      case "completed":
        return <CheckCircle className="w-4 h-4 text-green-600" />
      case "failed":
        return <AlertTriangle className="w-4 h-4 text-red-600" />
    }
  }

  const getStatusColor = (status: GradingJob["status"]) => {
    switch (status) {
      case "queued":
        return "bg-gray-100 text-gray-800"
      case "ocr-processing":
        return "bg-blue-100 text-blue-800"
      case "vector-retrieval":
        return "bg-purple-100 text-purple-800"
      case "llm-scoring":
        return "bg-orange-100 text-orange-800"
      case "completed":
        return "bg-green-100 text-green-800"
      case "failed":
        return "bg-red-100 text-red-800"
    }
  }

  const formatDuration = (start: string, end?: string) => {
    const startTime = new Date(start)
    const endTime = end ? new Date(end) : new Date()
    const duration = (endTime.getTime() - startTime.getTime()) / 1000
    return `${duration.toFixed(1)}s`
  }

  const getMetricColor = (value: number, thresholds: { warning: number; critical: number }) => {
    if (value >= thresholds.critical) return "text-red-600"
    if (value >= thresholds.warning) return "text-yellow-600"
    return "text-green-600"
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Enhanced Header */}
      <div className="border-b bg-card">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">TrueGradeAI Pipeline Monitor</h1>
              <p className="text-muted-foreground">Real-time AI-powered grading system monitoring</p>
            </div>
            <div className="flex items-center gap-4">
              {/* Connection Status */}
              <div className="flex items-center gap-2">
                {isConnected ? (
                  <Wifi className="w-4 h-4 text-green-600" />
                ) : (
                  <WifiOff className="w-4 h-4 text-red-600" />
                )}
                <span className="text-sm">{isConnected ? "Connected" : "Disconnected"}</span>
              </div>

              <Button variant="outline" size="sm" onClick={handlePauseResume}>
                {isRunning ? <Pause className="w-4 h-4 mr-2" /> : <Play className="w-4 h-4 mr-2" />}
                {isRunning ? "Pause" : "Resume"}
              </Button>
              <Button variant="outline" size="sm" onClick={handleRefresh}>
                <RefreshCw className="w-4 h-4 mr-2" />
                Refresh
              </Button>
            </div>
          </div>

          {/* Connection Alert */}
          {!isConnected && (
            <Alert className="mt-4 border-red-200 bg-red-50">
              <AlertTriangle className="h-4 w-4 text-red-600" />
              <AlertDescription className="text-red-800">
                Connection to grading service lost. Attempting to reconnect...
              </AlertDescription>
            </Alert>
          )}
        </div>
      </div>

      <div className="container mx-auto px-6 py-6">
        <Tabs defaultValue="overview" className="space-y-6">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="pipeline">Pipeline Status</TabsTrigger>
            <TabsTrigger value="performance">Performance</TabsTrigger>
            <TabsTrigger value="logs">System Logs</TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="space-y-6">
            {/* Enhanced Stats Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Jobs</CardTitle>
                  <Activity className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats.totalJobs}</div>
                  <p className="text-xs text-muted-foreground">
                    {stats.completed} completed, {stats.failed} failed
                  </p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Throughput</CardTitle>
                  <Zap className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats.throughputPerMinute.toFixed(1)}/min</div>
                  <p className="text-xs text-muted-foreground">Jobs processed per minute</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">OCR Accuracy</CardTitle>
                  <FileText className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats.ocrAccuracy}%</div>
                  <p className="text-xs text-muted-foreground">Handwriting recognition</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Active Connections</CardTitle>
                  <Users className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats.activeConnections}</div>
                  <p className="text-xs text-muted-foreground">Real-time connections</p>
                </CardContent>
              </Card>
            </div>

            {/* Enhanced Pipeline Flow Visualization */}
            <Card>
              <CardHeader>
                <CardTitle>TrueGradeAI Grading Pipeline Flow</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex items-center justify-between p-6">
                  <div className="flex flex-col items-center gap-2">
                    <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                      <FileText className="w-6 h-6 text-blue-600" />
                    </div>
                    <span className="text-sm font-medium">TrOCR</span>
                    <span className="text-xs text-muted-foreground">Handwriting → Text</span>
                    <Badge variant="outline" className="text-xs">
                      {stats.ocrAccuracy}% accuracy
                    </Badge>
                  </div>

                  <div className="flex-1 h-px bg-border mx-4 relative">
                    <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 bg-background px-2">
                      <div className="w-2 h-2 bg-blue-500 rounded-full animate-pulse"></div>
                    </div>
                  </div>

                  <div className="flex flex-col items-center gap-2">
                    <div className="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center">
                      <Database className="w-6 h-6 text-purple-600" />
                    </div>
                    <span className="text-sm font-medium">Vector Search</span>
                    <span className="text-xs text-muted-foreground">FAISS Retrieval</span>
                    <Badge variant="outline" className="text-xs">
                      {stats.cacheHitRate}% cache hit
                    </Badge>
                  </div>

                  <div className="flex-1 h-px bg-border mx-4 relative">
                    <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 bg-background px-2">
                      <div className="w-2 h-2 bg-purple-500 rounded-full animate-pulse"></div>
                    </div>
                  </div>

                  <div className="flex flex-col items-center gap-2">
                    <div className="w-12 h-12 bg-orange-100 rounded-full flex items-center justify-center">
                      <Brain className="w-6 h-6 text-orange-600" />
                    </div>
                    <span className="text-sm font-medium">Gemini 2.5 Pro</span>
                    <span className="text-xs text-muted-foreground">LLM Scoring</span>
                    <Badge variant="outline" className="text-xs">
                      {stats.llmConfidence}% confidence
                    </Badge>
                  </div>

                  <div className="flex-1 h-px bg-border mx-4 relative">
                    <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 bg-background px-2">
                      <div className="w-2 h-2 bg-orange-500 rounded-full animate-pulse"></div>
                    </div>
                  </div>

                  <div className="flex flex-col items-center gap-2">
                    <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                      <CheckCircle className="w-6 h-6 text-green-600" />
                    </div>
                    <span className="text-sm font-medium">Result</span>
                    <span className="text-xs text-muted-foreground">Score + Explanation</span>
                    <Badge variant="outline" className="text-xs">
                      {stats.avgProcessingTime}s avg
                    </Badge>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Queue Status */}
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Queue Status</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex justify-between">
                      <span className="text-sm">Queued</span>
                      <Badge variant="secondary">{stats.queued}</Badge>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm">Processing</span>
                      <Badge variant="default">{stats.processing}</Badge>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm">Queue Depth</span>
                      <Badge variant="outline">{systemMetrics.queueDepth}</Badge>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">System Health</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex justify-between items-center">
                      <span className="text-sm">CPU</span>
                      <span
                        className={`text-sm font-medium ${getMetricColor(systemMetrics.cpu, { warning: 70, critical: 85 })}`}
                      >
                        {systemMetrics.cpu}%
                      </span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="text-sm">Memory</span>
                      <span
                        className={`text-sm font-medium ${getMetricColor(systemMetrics.memory, { warning: 75, critical: 90 })}`}
                      >
                        {systemMetrics.memory}%
                      </span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="text-sm">GPU</span>
                      <span
                        className={`text-sm font-medium ${getMetricColor(systemMetrics.gpu, { warning: 85, critical: 95 })}`}
                      >
                        {systemMetrics.gpu}%
                      </span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Network Status</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex justify-between items-center">
                      <span className="text-sm">Latency</span>
                      <span
                        className={`text-sm font-medium ${getMetricColor(systemMetrics.networkLatency, {
                          warning: 50,
                          critical: 100,
                        })}`}
                      >
                        {systemMetrics.networkLatency}ms
                      </span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="text-sm">Connections</span>
                      <span className="text-sm font-medium text-green-600">{stats.activeConnections}</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="text-sm">Status</span>
                      <Badge variant={isConnected ? "default" : "destructive"}>
                        {isConnected ? "Online" : "Offline"}
                      </Badge>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="pipeline" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <span>Active Grading Jobs</span>
                  <Badge variant="secondary">
                    {jobs.filter((j) => j.status !== "completed" && j.status !== "failed").length} Active
                  </Badge>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {jobs.map((job) => (
                    <div key={job.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <div className="flex items-center gap-2">
                          {getStatusIcon(job.status)}
                          <div>
                            <h3 className="font-semibold">{job.studentName}</h3>
                            <p className="text-sm text-muted-foreground">Question {job.questionId}</p>
                          </div>
                        </div>
                      </div>

                      <div className="flex items-center gap-6">
                        <Badge className={getStatusColor(job.status)}>{job.status.replace("-", " ")}</Badge>

                        <div className="w-32">
                          <div className="flex items-center justify-between text-xs mb-1">
                            <span>Progress</span>
                            <span>{Math.round(job.progress)}%</span>
                          </div>
                          <Progress value={job.progress} className="h-2" />
                        </div>

                        <div className="text-right text-sm">
                          <p className="font-medium">{formatDuration(job.startTime, job.endTime)}</p>
                          <p className="text-muted-foreground">Duration</p>
                        </div>

                        {job.llmScore && (
                          <div className="text-right">
                            <p className="text-lg font-bold">{job.llmScore}%</p>
                            <p className="text-xs text-muted-foreground">Score</p>
                          </div>
                        )}

                        <Button variant="outline" size="sm" onClick={() => setSelectedJob(job)}>
                          <Eye className="w-4 h-4 mr-2" />
                          Details
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Job Details Modal */}
            {selectedJob && (
              <Card>
                <CardHeader>
                  <CardTitle>Job Details - {selectedJob.studentName}</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <span className="text-sm text-muted-foreground">OCR Confidence:</span>
                      <p className="font-semibold">{selectedJob.ocrConfidence}%</p>
                    </div>
                    <div>
                      <span className="text-sm text-muted-foreground">Vector Matches:</span>
                      <p className="font-semibold">{selectedJob.vectorMatches}</p>
                    </div>
                    <div>
                      <span className="text-sm text-muted-foreground">LLM Confidence:</span>
                      <p className="font-semibold">{selectedJob.llmConfidence}%</p>
                    </div>
                    <div>
                      <span className="text-sm text-muted-foreground">Final Score:</span>
                      <p className="font-semibold">{selectedJob.llmScore}%</p>
                    </div>
                  </div>
                  <Button variant="outline" onClick={() => setSelectedJob(null)}>
                    Close
                  </Button>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="performance" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="w-5 h-5" />
                    Processing Throughput
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="h-64 flex items-center justify-center text-muted-foreground">
                    Throughput chart would be rendered here
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Server className="w-5 h-5" />
                    System Resources
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <div className="flex items-center justify-between text-sm mb-1">
                      <span>CPU Usage</span>
                      <span>67%</span>
                    </div>
                    <Progress value={67} className="h-2" />
                  </div>
                  <div>
                    <div className="flex items-center justify-between text-sm mb-1">
                      <span>Memory Usage</span>
                      <span>45%</span>
                    </div>
                    <Progress value={45} className="h-2" />
                  </div>
                  <div>
                    <div className="flex items-center justify-between text-sm mb-1">
                      <span>GPU Utilization</span>
                      <span>82%</span>
                    </div>
                    <Progress value={82} className="h-2" />
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="logs" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <span>Real-time System Logs</span>
                  <Badge variant="outline">{logs.length} entries</Badge>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2 font-mono text-sm max-h-96 overflow-y-auto">
                  {logs.map((log, index) => (
                    <div key={index} className="flex items-center gap-2 py-1">
                      <span className="text-muted-foreground text-xs">{log.timestamp}</span>
                      <Badge
                        variant="outline"
                        className={`text-xs ${
                          log.level === "ERROR"
                            ? "border-red-200 text-red-800"
                            : log.level === "WARN"
                              ? "border-yellow-200 text-yellow-800"
                              : "border-blue-200 text-blue-800"
                        }`}
                      >
                        {log.level}
                      </Badge>
                      <span className="text-sm">{log.message}</span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
