"use client"

import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Progress } from "@/components/ui/progress"
import { CheckCircle, XCircle, AlertTriangle, Target } from "lucide-react"

interface RubricCriteria {
  name: string
  description: string
  maxPoints: number
  earnedPoints: number
  feedback: string
  status: "excellent" | "good" | "satisfactory" | "needs-improvement"
}

interface GradingRubricDisplayProps {
  criteria: RubricCriteria[]
  totalPoints: number
  earnedPoints: number
  subject: "History" | "Geography" | "Political Science"
}

export function GradingRubricDisplay({ criteria, totalPoints, earnedPoints, subject }: GradingRubricDisplayProps) {
  const percentage = (earnedPoints / totalPoints) * 100

  const getStatusIcon = (status: RubricCriteria["status"]) => {
    switch (status) {
      case "excellent":
        return <CheckCircle className="w-4 h-4 text-green-600" />
      case "good":
        return <CheckCircle className="w-4 h-4 text-blue-600" />
      case "satisfactory":
        return <AlertTriangle className="w-4 h-4 text-yellow-600" />
      case "needs-improvement":
        return <XCircle className="w-4 h-4 text-red-600" />
    }
  }

  const getStatusColor = (status: RubricCriteria["status"]) => {
    switch (status) {
      case "excellent":
        return "bg-green-100 text-green-800"
      case "good":
        return "bg-blue-100 text-blue-800"
      case "satisfactory":
        return "bg-yellow-100 text-yellow-800"
      case "needs-improvement":
        return "bg-red-100 text-red-800"
    }
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          <span className="flex items-center gap-2">
            <Target className="w-5 h-5" />
            Grading Rubric - {subject}
          </span>
          <div className="text-right">
            <div className="text-2xl font-bold">
              {earnedPoints}/{totalPoints}
            </div>
            <div className="text-sm text-muted-foreground">{percentage.toFixed(1)}%</div>
          </div>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="mb-4">
          <div className="flex items-center justify-between text-sm mb-2">
            <span>Overall Performance</span>
            <span>{percentage.toFixed(1)}%</span>
          </div>
          <Progress value={percentage} className="h-3" />
        </div>

        <div className="space-y-3">
          {criteria.map((criterion, index) => {
            const criterionPercentage = (criterion.earnedPoints / criterion.maxPoints) * 100

            return (
              <div key={index} className="p-4 border rounded-lg">
                <div className="flex items-center justify-between mb-2">
                  <div className="flex items-center gap-2">
                    {getStatusIcon(criterion.status)}
                    <h4 className="font-medium">{criterion.name}</h4>
                  </div>
                  <div className="flex items-center gap-2">
                    <Badge className={getStatusColor(criterion.status)}>{criterion.status.replace("-", " ")}</Badge>
                    <span className="text-sm font-medium">
                      {criterion.earnedPoints}/{criterion.maxPoints}
                    </span>
                  </div>
                </div>

                <p className="text-sm text-muted-foreground mb-3">{criterion.description}</p>

                <div className="space-y-2">
                  <div className="flex items-center justify-between text-xs">
                    <span>Performance</span>
                    <span>{criterionPercentage.toFixed(1)}%</span>
                  </div>
                  <Progress value={criterionPercentage} className="h-2" />
                </div>

                {criterion.feedback && (
                  <div className="mt-3 p-2 bg-muted rounded text-sm">
                    <strong>Feedback:</strong> {criterion.feedback}
                  </div>
                )}
              </div>
            )
          })}
        </div>
      </CardContent>
    </Card>
  )
}
