"use client"

import { useState, useEffect } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Progress } from "@/components/ui/progress"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import { ScrollArea } from "@/components/ui/scroll-area"
import { CameraFeed } from "@/components/camera-feed"
import { StylusAIProcessor } from "@/components/stylus-ai-processor"
import { EnhancedButtonControls } from "@/components/enhanced-button-controls"
import {
  Camera,
  Monitor,
  Mail,
  Webhook,
  Settings,
  Eye,
  Volume2,
  VolumeX,
  AlertTriangle,
  CheckCircle,
  Clock,
  Users,
  Brain,
  Maximize2,
  Minimize2,
} from "lucide-react"

interface StudentMonitorData {
  id: string
  name: string
  email: string
  status: "active" | "idle" | "flagged" | "completed"
  cameraActive: boolean
  lastActivity: Date
  suspiciousActivity: number
  currentQuestion: number
  totalQuestions: number
  timeRemaining: number
  aiProcessingActive: boolean
  lastStylusInput?: string
}

interface DeliverySettings {
  method: "display" | "email" | "webhook"
  emailAddress?: string
  webhookUrl?: string
  realTimeUpdates: boolean
  alertThreshold: number
}

const mockStudentData: StudentMonitorData[] = []

export function LiveMonitoringInterface() {
  const [selectedStudent, setSelectedStudent] = useState<string | null>(null)
  const [deliverySettings, setDeliverySettings] = useState<DeliverySettings>({
    method: "display",
    realTimeUpdates: true,
    alertThreshold: 2,
  })
  const [isFullscreen, setIsFullscreen] = useState(false)
  const [audioAlerts, setAudioAlerts] = useState(true)
  const [autoRefresh, setAutoRefresh] = useState(true)
  const [refreshInterval, setRefreshInterval] = useState(5)
  const [processedTexts, setProcessedTexts] = useState<
    Array<{
      studentId: string
      text: string
      analysis: any
      timestamp: Date
    }>
  >([])

  useEffect(() => {
    if (!selectedStudent) return
    const es = new EventSource(`/api/realtime/stream?studentId=${selectedStudent}`)
    es.onmessage = (e) => {
      try {
        const msg = JSON.parse(e.data)
        if (msg?.data?.type === "stylus-text") {
          setProcessedTexts((prev) => [
            { studentId: selectedStudent, text: msg.data.text, analysis: msg.data.analysis, timestamp: new Date() },
            ...prev,
          ])
        }
      } catch {}
    }
    es.onerror = () => {
      es.close()
    }
    return () => es.close()
  }, [selectedStudent])

  const handleTextProcessed = (studentId: string, text: string, analysis: any) => {
    console.log("[v0] Text processed for student:", studentId, text)
    const newEntry = {
      studentId,
      text,
      analysis,
      timestamp: new Date(),
    }

    setProcessedTexts((prev) => [newEntry, ...prev.slice(0, 49)]) // Keep last 50 entries

    // Handle delivery based on settings
    switch (deliverySettings.method) {
      case "email":
        if (deliverySettings.emailAddress) {
          fetch("/api/delivery/email", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ studentId, text, analysis, emailAddress: deliverySettings.emailAddress }),
          }).catch((e) => console.error("[v0] Email delivery failed", e))
        }
        break
      case "webhook":
        if (deliverySettings.webhookUrl) {
          fetch("/api/delivery/webhook", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ studentId, text, analysis, webhookUrl: deliverySettings.webhookUrl }),
          }).catch((e) => console.error("[v0] Webhook delivery failed", e))
        }
        break
      case "display":
      default:
        // Already handled by state update
        break
    }
  }

  const handleButtonAction = (action: string, data?: any) => {
    console.log("[v0] Enhanced button action:", action, data)

    switch (action) {
      case "fullscreen":
        setIsFullscreen(!isFullscreen)
        break
      case "audio":
        setAudioAlerts(!audioAlerts)
        break
      case "refresh":
        // Refresh monitoring data
        console.log("[v0] Refreshing monitoring data...")
        break
      case "monitor-all":
        console.log("[v0] Monitoring all students...")
        break
      case "send-alert":
        console.log("[v0] Sending alert to students...")
        break
      case "grade-all":
        console.log("[v0] Starting AI grading for all submissions...")
        break
      default:
        console.log("[v0] Unhandled action:", action)
    }
  }

  const getStatusColor = (status: StudentMonitorData["status"]) => {
    switch (status) {
      case "active":
        return "bg-green-100 text-green-800"
      case "idle":
        return "bg-yellow-100 text-yellow-800"
      case "flagged":
        return "bg-red-100 text-red-800"
      case "completed":
        return "bg-blue-100 text-blue-800"
    }
  }

  const formatTime = (seconds: number) => {
    const hours = Math.floor(seconds / 3600)
    const minutes = Math.floor((seconds % 3600) / 60)
    const secs = seconds % 60
    return `${hours.toString().padStart(2, "0")}:${minutes.toString().padStart(2, "0")}:${secs.toString().padStart(2, "0")}`
  }

  const getTimeSinceActivity = (lastActivity: Date) => {
    const diff = Date.now() - lastActivity.getTime()
    const minutes = Math.floor(diff / 60000)
    if (minutes < 1) return "Just now"
    if (minutes < 60) return `${minutes}m ago`
    const hours = Math.floor(minutes / 60)
    return `${hours}h ${minutes % 60}m ago`
  }

  useEffect(() => {
    if (!autoRefresh) return

    const interval = setInterval(() => {
      console.log("[v0] Auto-refreshing monitoring data...")
      // In real implementation, this would fetch fresh data
    }, refreshInterval * 1000)

    return () => clearInterval(interval)
  }, [autoRefresh, refreshInterval])

  return (
    <div className={`${isFullscreen ? "fixed inset-0 z-50 bg-background" : ""}`}>
      <div className="space-y-6 p-6">
        {/* Header Controls */}
        <div className="flex items-center justify-between">
          <div>
            <h2 className="text-2xl font-bold">Live Student Monitoring</h2>
            <p className="text-muted-foreground">Real-time camera feeds and AI stylus processing</p>
          </div>
          <div className="flex items-center gap-2">
            <Button variant="outline" size="sm" onClick={() => setAudioAlerts(!audioAlerts)}>
              {audioAlerts ? <Volume2 className="h-4 w-4" /> : <VolumeX className="h-4 w-4" />}
            </Button>
            <Button variant="outline" size="sm" onClick={() => setIsFullscreen(!isFullscreen)}>
              {isFullscreen ? <Minimize2 className="h-4 w-4" /> : <Maximize2 className="h-4 w-4" />}
            </Button>
          </div>
        </div>

        <Tabs defaultValue="grid-view" className="space-y-4">
          <TabsList className="grid w-full grid-cols-5">
            <TabsTrigger value="grid-view">Grid View</TabsTrigger>
            <TabsTrigger value="detailed-view">Detailed View</TabsTrigger>
            <TabsTrigger value="ai-processing">AI Processing</TabsTrigger>
            <TabsTrigger value="delivery-settings">Delivery Settings</TabsTrigger>
            <TabsTrigger value="enhanced-controls">Enhanced Controls</TabsTrigger>
          </TabsList>

          <TabsContent value="grid-view" className="space-y-4">
            {/* Status Overview */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center gap-2">
                    <Users className="h-4 w-4 text-green-600" />
                    <div>
                      <p className="text-2xl font-bold text-green-600">
                        {mockStudentData.filter((s) => s.status === "active").length}
                      </p>
                      <p className="text-sm text-muted-foreground">Active</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center gap-2">
                    <AlertTriangle className="h-4 w-4 text-red-600" />
                    <div>
                      <p className="text-2xl font-bold text-red-600">
                        {mockStudentData.filter((s) => s.status === "flagged").length}
                      </p>
                      <p className="text-sm text-muted-foreground">Flagged</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center gap-2">
                    <Camera className="h-4 w-4 text-blue-600" />
                    <div>
                      <p className="text-2xl font-bold text-blue-600">
                        {mockStudentData.filter((s) => s.cameraActive).length}
                      </p>
                      <p className="text-sm text-muted-foreground">Cameras Active</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center gap-2">
                    <Brain className="h-4 w-4 text-purple-600" />
                    <div>
                      <p className="text-2xl font-bold text-purple-600">
                        {mockStudentData.filter((s) => s.aiProcessingActive).length}
                      </p>
                      <p className="text-sm text-muted-foreground">AI Processing</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Camera Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {mockStudentData.map((student) => (
                <Card key={student.id} className="glass-effect">
                  <CardHeader className="pb-2">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <Avatar className="h-8 w-8">
                          <AvatarFallback className="text-xs">
                            {student.name
                              .split(" ")
                              .map((n) => n[0])
                              .join("")}
                          </AvatarFallback>
                        </Avatar>
                        <div>
                          <p className="font-medium text-sm">{student.name}</p>
                          <p className="text-xs text-muted-foreground">{student.email}</p>
                        </div>
                      </div>
                      <Badge className={getStatusColor(student.status)} variant="secondary">
                        {student.status}
                      </Badge>
                    </div>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    {/* Camera Feed */}
                    <CameraFeed isTeacherView={true} studentId={student.id} />

                    {/* Student Stats */}
                    <div className="grid grid-cols-2 gap-2 text-xs">
                      <div className="flex items-center gap-1">
                        <Clock className="h-3 w-3" />
                        <span>{formatTime(student.timeRemaining)}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <CheckCircle className="h-3 w-3" />
                        <span>
                          {student.currentQuestion}/{student.totalQuestions}
                        </span>
                      </div>
                      <div className="flex items-center gap-1">
                        <Eye className="h-3 w-3" />
                        <span>{getTimeSinceActivity(student.lastActivity)}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <AlertTriangle className="h-3 w-3" />
                        <span>{student.suspiciousActivity} alerts</span>
                      </div>
                    </div>

                    {/* Progress Bar */}
                    <div className="space-y-1">
                      <div className="flex justify-between text-xs">
                        <span>Progress</span>
                        <span>{Math.round((student.currentQuestion / student.totalQuestions) * 100)}%</span>
                      </div>
                      <Progress value={(student.currentQuestion / student.totalQuestions) * 100} className="h-2" />
                    </div>

                    {/* Action Buttons */}
                    <div className="flex gap-2">
                      <Button
                        size="sm"
                        variant="outline"
                        className="flex-1 bg-transparent"
                        onClick={() => setSelectedStudent(student.id)}
                      >
                        <Monitor className="h-3 w-3 mr-1" />
                        Focus
                      </Button>
                      {student.suspiciousActivity > 0 && (
                        <Button size="sm" variant="destructive" className="flex-1">
                          <AlertTriangle className="h-3 w-3 mr-1" />
                          Alert
                        </Button>
                      )}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          <TabsContent value="detailed-view" className="space-y-4">
            {selectedStudent ? (
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {/* Student Camera and Info */}
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center justify-between">
                      <span>Student Monitor</span>
                      <Button variant="outline" size="sm" onClick={() => setSelectedStudent(null)}>
                        Back to Grid
                      </Button>
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    {(() => {
                      const student = mockStudentData.find((s) => s.id === selectedStudent)
                      return student ? (
                        <>
                          <div className="flex items-center gap-4">
                            <Avatar className="h-12 w-12">
                              <AvatarFallback>
                                {student.name
                                  .split(" ")
                                  .map((n) => n[0])
                                  .join("")}
                              </AvatarFallback>
                            </Avatar>
                            <div>
                              <h3 className="font-semibold">{student.name}</h3>
                              <p className="text-sm text-muted-foreground">{student.email}</p>
                              <Badge className={getStatusColor(student.status)} variant="secondary">
                                {student.status}
                              </Badge>
                            </div>
                          </div>

                          <CameraFeed isTeacherView={true} studentId={student.id} />

                          <div className="grid grid-cols-2 gap-4 text-sm">
                            <div>
                              <span className="text-muted-foreground">Time Remaining:</span>
                              <p className="font-medium">{formatTime(student.timeRemaining)}</p>
                            </div>
                            <div>
                              <span className="text-muted-foreground">Question Progress:</span>
                              <p className="font-medium">
                                {student.currentQuestion}/{student.totalQuestions}
                              </p>
                            </div>
                            <div>
                              <span className="text-muted-foreground">Last Activity:</span>
                              <p className="font-medium">{getTimeSinceActivity(student.lastActivity)}</p>
                            </div>
                            <div>
                              <span className="text-muted-foreground">Suspicious Activity:</span>
                              <p className="font-medium">{student.suspiciousActivity} alerts</p>
                            </div>
                          </div>
                        </>
                      ) : null
                    })()}
                  </CardContent>
                </Card>

                {/* AI Processing */}
                <Card>
                  <CardHeader>
                    <CardTitle>AI Stylus Processing</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <StylusAIProcessor
                      studentId={selectedStudent!}
                      onTextProcessed={(text, analysis) => handleTextProcessed(selectedStudent!, text, analysis)}
                      deliveryMethod={deliverySettings.method}
                    />
                  </CardContent>
                </Card>
              </div>
            ) : (
              <Card>
                <CardContent className="py-12">
                  <div className="text-center text-muted-foreground">
                    <Monitor className="w-12 h-12 mx-auto mb-4 opacity-50" />
                    <h3 className="text-lg font-medium mb-2">Select a Student</h3>
                    <p>Choose a student from the grid view to see detailed monitoring information.</p>
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="ai-processing" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Brain className="h-5 w-5" />
                  AI Processing Results
                </CardTitle>
              </CardHeader>
              <CardContent>
                <ScrollArea className="h-96">
                  <div className="space-y-4">
                    {processedTexts.length > 0 ? (
                      processedTexts.map((entry, index) => {
                        const student = mockStudentData.find((s) => s.id === entry.studentId)
                        return (
                          <div key={index} className="p-4 border rounded-lg">
                            <div className="flex items-center justify-between mb-2">
                              <div className="flex items-center gap-2">
                                <Avatar className="h-6 w-6">
                                  <AvatarFallback className="text-xs">
                                    {student?.name
                                      .split(" ")
                                      .map((n) => n[0])
                                      .join("") || "U"}
                                  </AvatarFallback>
                                </Avatar>
                                <span className="font-medium text-sm">{student?.name || "Unknown"}</span>
                              </div>
                              <span className="text-xs text-muted-foreground">
                                {entry.timestamp.toLocaleTimeString()}
                              </span>
                            </div>
                            <p className="text-sm mb-2">{entry.text}</p>
                            <div className="text-xs text-muted-foreground">
                              <span>Confidence: {(entry.analysis.confidence * 100).toFixed(1)}%</span>
                              <span className="ml-4">Words: {entry.analysis.wordCount}</span>
                              <span className="ml-4">Sentiment: {entry.analysis.sentiment}</span>
                            </div>
                          </div>
                        )
                      })
                    ) : (
                      <div className="text-center py-8 text-muted-foreground">
                        <Brain className="w-12 h-12 mx-auto mb-4 opacity-50" />
                        <h3 className="text-lg font-medium mb-2">No Processing Results</h3>
                        <p>AI processing results will appear here as students write with their stylus.</p>
                      </div>
                    )}
                  </div>
                </ScrollArea>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="delivery-settings" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Settings className="h-5 w-5" />
                  Delivery Method Configuration
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                {/* Delivery Method Selection */}
                <div className="space-y-2">
                  <label className="text-sm font-medium">Delivery Method</label>
                  <Select
                    value={deliverySettings.method}
                    onValueChange={(value: "display" | "email" | "webhook") =>
                      setDeliverySettings((prev) => ({ ...prev, method: value }))
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="display">
                        <div className="flex items-center gap-2">
                          <Monitor className="h-4 w-4" />
                          Display in Interface
                        </div>
                      </SelectItem>
                      <SelectItem value="email">
                        <div className="flex items-center gap-2">
                          <Mail className="h-4 w-4" />
                          Email Notifications
                        </div>
                      </SelectItem>
                      <SelectItem value="webhook">
                        <div className="flex items-center gap-2">
                          <Webhook className="h-4 w-4" />
                          Webhook Integration
                        </div>
                      </SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                {/* Email Configuration */}
                {deliverySettings.method === "email" && (
                  <div className="space-y-2">
                    <label className="text-sm font-medium">Email Address</label>
                    <input
                      type="email"
                      className="w-full px-3 py-2 border border-border rounded-md"
                      placeholder="teacher@university.edu"
                      value={deliverySettings.emailAddress || ""}
                      onChange={(e) =>
                        setDeliverySettings((prev) => ({
                          ...prev,
                          emailAddress: e.target.value,
                        }))
                      }
                    />
                  </div>
                )}

                {/* Webhook Configuration */}
                {deliverySettings.method === "webhook" && (
                  <div className="space-y-2">
                    <label className="text-sm font-medium">Webhook URL</label>
                    <input
                      type="url"
                      className="w-full px-3 py-2 border border-border rounded-md"
                      placeholder="https://your-system.com/webhook"
                      value={deliverySettings.webhookUrl || ""}
                      onChange={(e) =>
                        setDeliverySettings((prev) => ({
                          ...prev,
                          webhookUrl: e.target.value,
                        }))
                      }
                    />
                  </div>
                )}

                {/* Additional Settings */}
                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <label className="text-sm font-medium">Real-time Updates</label>
                      <p className="text-xs text-muted-foreground">Send updates immediately as text is processed</p>
                    </div>
                    <Button
                      variant={deliverySettings.realTimeUpdates ? "default" : "outline"}
                      size="sm"
                      onClick={() =>
                        setDeliverySettings((prev) => ({
                          ...prev,
                          realTimeUpdates: !prev.realTimeUpdates,
                        }))
                      }
                    >
                      {deliverySettings.realTimeUpdates ? "Enabled" : "Disabled"}
                    </Button>
                  </div>

                  <div className="space-y-2">
                    <label className="text-sm font-medium">Alert Threshold</label>
                    <p className="text-xs text-muted-foreground">
                      Number of suspicious activities before sending alert
                    </p>
                    <Select
                      value={deliverySettings.alertThreshold.toString()}
                      onValueChange={(value) =>
                        setDeliverySettings((prev) => ({
                          ...prev,
                          alertThreshold: Number.parseInt(value),
                        }))
                      }
                    >
                      <SelectTrigger className="w-32">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="1">1 Alert</SelectItem>
                        <SelectItem value="2">2 Alerts</SelectItem>
                        <SelectItem value="3">3 Alerts</SelectItem>
                        <SelectItem value="5">5 Alerts</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                {/* Test Delivery */}
                <div className="pt-4 border-t">
                  <Button
                    onClick={() => {
                      console.log("[v0] Testing delivery method:", deliverySettings.method)
                      handleTextProcessed("test-student", "This is a test message from the AI processing system.", {
                        confidence: 0.95,
                        wordCount: 12,
                        sentiment: "neutral",
                      })
                    }}
                    className="w-full"
                  >
                    Test Delivery Method
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="enhanced-controls" className="space-y-4">
            <EnhancedButtonControls onAction={handleButtonAction} context="teacher" />
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
