"use client"

import type React from "react"
import { useState, useRef, useEffect, useCallback } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Progress } from "@/components/ui/progress"
import { Badge } from "@/components/ui/badge"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Textarea } from "@/components/ui/textarea"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { CameraFeed } from "@/components/camera-feed"
import { StylusAIProcessor } from "@/components/stylus-ai-processor"
import { ExamAPI } from "@/lib/exam-api"
import {
  Camera,
  Clock,
  User,
  Shield,
  FileText,
  ChevronLeft,
  ChevronRight,
  Save,
  Send,
  Undo,
  Redo,
  PenTool,
  Eraser,
  ZoomIn,
  ZoomOut,
  AlertTriangle,
  CheckCircle,
  Circle,
  Wifi,
  WifiOff,
} from "lucide-react"

interface Question {
  id: number
  text: string
  marks: number
  timeLimit?: number
}

interface ExamData {
  subject: string
  duration: number
  totalMarks: number
  questions: Question[]
}

const LOCAL_FALLBACK_EXAM: ExamData = {
  subject: "Class 12 - History, Geography & Political Science",
  duration: 180,
  totalMarks: 75,
  questions: [
    { id: 1, text: "Analyze the causes and consequences of the Revolt of 1857. How did it impact British colonial policy in India?", marks: 12 },
    { id: 2, text: "Explain the concept of sustainable development with reference to India's environmental challenges. Suggest measures for achieving sustainability.", marks: 10 },
    { id: 3, text: "Evaluate the role of regional parties in Indian democracy and coalition politics. How have they influenced national politics?", marks: 12 },
    { id: 4, text: "Evaluate the role of Mahatma Gandhi in the Indian freedom struggle. Discuss his methods and their effectiveness.", marks: 15 },
    { id: 5, text: "Analyze the distribution pattern of iron ore in India and its impact on the steel industry. Discuss the challenges faced by the industry.", marks: 12 },
    { id: 6, text: "Discuss the challenges to democracy in contemporary India. Suggest measures to strengthen democratic institutions.", marks: 14 },
  ],
}

export function StudentExamInterface() {
  const [currentStep, setCurrentStep] = useState<"auth" | "admit" | "exam">("auth")
  const [currentQuestion, setCurrentQuestion] = useState(0)
  const [examData, setExamData] = useState<ExamData | null>(null)
  const [timeRemaining, setTimeRemaining] = useState(0)
  const [answers, setAnswers] = useState<{ [key: number]: { canvas?: string; text?: string; lastSaved?: Date } }>({})
  const [isDrawing, setIsDrawing] = useState(false)
  const [drawingHistory, setDrawingHistory] = useState<string[]>([])
  const [historyStep, setHistoryStep] = useState(-1)
  const [brushSize, setBrushSize] = useState(2)
  const [brushColor, setBrushColor] = useState("#000000")
  const [isEraser, setIsEraser] = useState(false)
  const [zoom, setZoom] = useState(1)
  const [answerMode, setAnswerMode] = useState<"canvas" | "text">("canvas")
  const [textAnswer, setTextAnswer] = useState("")
  const [isOnline, setIsOnline] = useState(true)
  const [autoSaveStatus, setAutoSaveStatus] = useState<"saved" | "saving" | "error">("saved")
  const [examSubmitted, setExamSubmitted] = useState(false)
  const [showWarning, setShowWarning] = useState(false)

  const canvasRef = useRef<HTMLCanvasElement>(null)
  const [authStep, setAuthStep] = useState<"face" | "id" | "complete">("face")

  useEffect(() => {
    const loadExam = async () => {
      try {
        const data = await ExamAPI.getExamData()
        setExamData(data)
        setTimeRemaining(data.duration * 60)
      } catch (e) {
        console.error("[v0] Failed to load exam data, using fallback", e)
        setExamData(LOCAL_FALLBACK_EXAM)
        setTimeRemaining(LOCAL_FALLBACK_EXAM.duration * 60)
      }
    }
    loadExam()
  }, [])

  useEffect(() => {
    if (currentStep === "exam" && timeRemaining > 0 && !examSubmitted) {
      const timer = setInterval(() => {
        setTimeRemaining((prev) => {
          if (prev <= 1) {
            handleAutoSubmit()
            return 0
          }
          return prev - 1
        })
      }, 1000)
      return () => clearInterval(timer)
    }
  }, [currentStep, timeRemaining, examSubmitted])

  useEffect(() => {
    if (currentStep === "exam") {
      const autoSaveInterval = setInterval(() => {
        handleAutoSave()
      }, 30000) // Auto-save every 30 seconds
      return () => clearInterval(autoSaveInterval)
    }
  }, [currentStep, currentQuestion, textAnswer])

  useEffect(() => {
    const handleOnline = () => setIsOnline(true)
    const handleOffline = () => setIsOnline(false)

    window.addEventListener("online", handleOnline)
    window.addEventListener("offline", handleOffline)

    return () => {
      window.removeEventListener("online", handleOnline)
      window.removeEventListener("offline", handleOffline)
    }
  }, [])

  useEffect(() => {
    if (timeRemaining <= 300 && timeRemaining > 0) {
      // 5 minutes warning
      setShowWarning(true)
    } else {
      setShowWarning(false)
    }
  }, [timeRemaining])

  const handleAutoSubmit = useCallback(() => {
    console.log("[v0] Auto-submitting exam due to time expiry")
    setExamSubmitted(true)
    // In real implementation, this would submit to backend
    alert("Time's up! Your exam has been automatically submitted.")
  }, [])

  const handleAutoSave = useCallback(async () => {
    if (examSubmitted || !examData) return

    setAutoSaveStatus("saving")
    try {
      const qid = examData.questions[currentQuestion].id
      const currentAnswer = {
        canvas: canvasRef.current?.toDataURL(),
        text: textAnswer,
        lastSaved: new Date(),
      }

      setAnswers((prev) => ({
        ...prev,
        [qid]: currentAnswer,
      }))

      await ExamAPI.saveAnswer(qid, currentAnswer)
      setAutoSaveStatus("saved")
      console.log("[v0] Auto-saved answer for question", currentQuestion + 1)
    } catch (error) {
      setAutoSaveStatus("error")
      console.log("[v0] Auto-save failed:", error)
    }
  }, [currentQuestion, textAnswer, examSubmitted, examData])

  const saveCanvasState = () => {
    const canvas = canvasRef.current
    if (!canvas) return

    const dataURL = canvas.toDataURL()
    const newHistory = drawingHistory.slice(0, historyStep + 1)
    newHistory.push(dataURL)
    setDrawingHistory(newHistory)
    setHistoryStep(newHistory.length - 1)
  }

  const handleCanvasMouseDown = (e: React.MouseEvent<HTMLCanvasElement>) => {
    setIsDrawing(true)
    const canvas = canvasRef.current
    if (!canvas) return

    const rect = canvas.getBoundingClientRect()
    const x = (e.clientX - rect.left) / zoom
    const y = (e.clientY - rect.top) / zoom

    const ctx = canvas.getContext("2d")
    if (ctx) {
      ctx.globalCompositeOperation = isEraser ? "destination-out" : "source-over"
      ctx.lineWidth = brushSize
      ctx.strokeStyle = isEraser ? "rgba(0,0,0,1)" : brushColor
      ctx.lineCap = "round"
      ctx.lineJoin = "round"
      ctx.beginPath()
      ctx.moveTo(x, y)
    }
  }

  const handleTouchStart = (e: React.TouchEvent<HTMLCanvasElement>) => {
    e.preventDefault()
    const touch = e.touches[0]
    const mouseEvent = new MouseEvent("mousedown", {
      clientX: touch.clientX,
      clientY: touch.clientY,
    })
    handleCanvasMouseDown(mouseEvent as any)
  }

  const handleTouchMove = (e: React.TouchEvent<HTMLCanvasElement>) => {
    e.preventDefault()
    const touch = e.touches[0]
    const mouseEvent = new MouseEvent("mousemove", {
      clientX: touch.clientX,
      clientY: touch.clientY,
    })
    handleCanvasMouseMove(mouseEvent as any)
  }

  const handleTouchEnd = (e: React.TouchEvent<HTMLCanvasElement>) => {
    e.preventDefault()
    handleCanvasMouseUp()
  }

  const handleCanvasMouseMove = (e: React.MouseEvent<HTMLCanvasElement>) => {
    if (!isDrawing) return

    const canvas = canvasRef.current
    if (!canvas) return

    const rect = canvas.getBoundingClientRect()
    const x = (e.clientX - rect.left) / zoom
    const y = (e.clientY - rect.top) / zoom

    const ctx = canvas.getContext("2d")
    if (ctx) {
      ctx.lineTo(x, y)
      ctx.stroke()
    }
  }

  const handleCanvasMouseUp = () => {
    if (isDrawing) {
      setIsDrawing(false)
      saveCanvasState()
    }
  }

  const handleUndo = () => {
    if (historyStep > 0) {
      setHistoryStep(historyStep - 1)
      restoreCanvasState(drawingHistory[historyStep - 1])
    }
  }

  const handleRedo = () => {
    if (historyStep < drawingHistory.length - 1) {
      setHistoryStep(historyStep + 1)
      restoreCanvasState(drawingHistory[historyStep + 1])
    }
  }

  const restoreCanvasState = (dataURL: string) => {
    const canvas = canvasRef.current
    if (!canvas) return

    const ctx = canvas.getContext("2d")
    if (!ctx) return

    const img = new Image()
    img.onload = () => {
      ctx.clearRect(0, 0, canvas.width, canvas.height)
      ctx.drawImage(img, 0, 0)
    }
    img.src = dataURL
  }

  const clearCanvas = () => {
    const canvas = canvasRef.current
    if (!canvas) return

    const ctx = canvas.getContext("2d")
    if (ctx) {
      ctx.clearRect(0, 0, canvas.width, canvas.height)
      saveCanvasState()
    }
  }

  const saveAnswer = async () => {
    if (!examData) return
    setAutoSaveStatus("saving")
    try {
      const qid = examData.questions[currentQuestion].id
      const currentAnswer = {
        canvas: answerMode === "canvas" ? canvasRef.current?.toDataURL() : undefined,
        text: answerMode === "text" ? textAnswer : undefined,
        lastSaved: new Date(),
      }

      setAnswers((prev) => ({
        ...prev,
        [qid]: currentAnswer,
      }))

      await ExamAPI.saveAnswer(qid, currentAnswer)
      setAutoSaveStatus("saved")
      console.log("[v0] Manually saved answer for question", currentQuestion + 1)
    } catch (error) {
      setAutoSaveStatus("error")
      console.log("[v0] Save failed:", error)
    }
  }

  const navigateToQuestion = (newIndex: number) => {
    if (newIndex === currentQuestion) return

    handleAutoSave()
    setCurrentQuestion(newIndex)

    if (!examData) return
    const existingAnswer = answers[examData.questions[newIndex].id]
    if (existingAnswer?.text) {
      setTextAnswer(existingAnswer.text)
    } else {
      setTextAnswer("")
    }
  }

  const handleSubmitExam = async () => {
    if (!examData) return
    const unansweredQuestions = examData.questions.filter((q) => !answers[q.id])

    if (unansweredQuestions.length > 0) {
      const proceed = confirm(
        `You have ${unansweredQuestions.length} unanswered questions. Are you sure you want to submit?`,
      )
      if (!proceed) return
    }

    try {
      const resp = await ExamAPI.submitExam(answers)
      setExamSubmitted(true)
      console.log("[v0] Exam submitted with answers:", answers)
      alert(`Exam submitted successfully! Grading status: ${resp.data.gradingStatus}. ETA: ${resp.estimatedGradingTime}`)
    } catch (e) {
      console.error("[v0] Submit failed", e)
      alert("Submit failed. Please try again.")
    }
  }

  const formatTime = (seconds: number) => {
    const hours = Math.floor(seconds / 3600)
    const minutes = Math.floor((seconds % 3600) / 60)
    const secs = seconds % 60
    return `${hours.toString().padStart(2, "0")}:${minutes.toString().padStart(2, "0")}:${secs.toString().padStart(2, "0")}`
  }

  if (currentStep === "auth") {
    return (
      <div className="min-h-screen flex items-center justify-center p-4">
        <Card className="w-full max-w-md">
          <CardHeader className="text-center">
            <div className="mx-auto mb-4 w-16 h-16 bg-secondary rounded-full flex items-center justify-center">
              <Shield className="w-8 h-8 text-secondary-foreground" />
            </div>
            <CardTitle className="text-2xl font-bold">Multi-Factor Authentication</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            {authStep === "face" && (
              <>
                <div className="text-center">
                  <Camera className="w-12 h-12 mx-auto mb-4 text-muted-foreground" />
                  <h3 className="text-lg font-semibold mb-2">Face Verification</h3>
                  <p className="text-muted-foreground mb-4">Position your face in the camera frame for verification</p>
                </div>
                <div className="bg-muted rounded-lg h-48 flex items-center justify-center">
                  <p className="text-muted-foreground">Camera feed would appear here</p>
                </div>
                <Button onClick={() => setAuthStep("id")} className="w-full">
                  Verify Face
                </Button>
              </>
            )}

            {authStep === "id" && (
              <>
                <div className="text-center">
                  <User className="w-12 h-12 mx-auto mb-4 text-muted-foreground" />
                  <h3 className="text-lg font-semibold mb-2">Student ID Verification</h3>
                  <p className="text-muted-foreground mb-4">Take a photo of your student ID card</p>
                </div>
                <div className="bg-muted rounded-lg h-48 flex items-center justify-center">
                  <p className="text-muted-foreground">ID capture interface would appear here</p>
                </div>
                <Button onClick={() => setAuthStep("complete")} className="w-full">
                  Capture ID
                </Button>
              </>
            )}

            {authStep === "complete" && (
              <>
                <div className="text-center">
                  <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <Shield className="w-8 h-8 text-green-600" />
                  </div>
                  <h3 className="text-lg font-semibold mb-2 text-green-600">Authentication Successful</h3>
                  <p className="text-muted-foreground mb-4">Identity verified. You may now proceed to your exam.</p>
                </div>
                <Button onClick={() => setCurrentStep("admit")} className="w-full">
                  Continue to Admit Card
                </Button>
              </>
            )}
          </CardContent>
        </Card>
      </div>
    )
  }

  if (currentStep === "admit") {
    return (
      <div className="min-h-screen flex items-center justify-center p-4">
        <Card className="w-full max-w-2xl">
          <CardHeader className="text-center border-b">
            <div className="mx-auto mb-4 w-16 h-16 bg-primary rounded-full flex items-center justify-center">
              <FileText className="w-8 h-8 text-primary-foreground" />
            </div>
            <CardTitle className="text-3xl font-bold">Digital Exam Admit Card</CardTitle>
          </CardHeader>
          <CardContent className="p-8 space-y-6">
            <div className="grid grid-cols-2 gap-6">
              <div>
                <h3 className="font-semibold text-lg mb-4">Exam Details</h3>
                <div className="space-y-3">
                  <div>
                    <span className="text-muted-foreground">Subject:</span>
                    <p className="font-semibold">{examData?.subject}</p>
                  </div>
                  <div>
                    <span className="text-muted-foreground">Duration:</span>
                    <p className="font-semibold">{examData?.duration} minutes</p>
                  </div>
                  <div>
                    <span className="text-muted-foreground">Total Marks:</span>
                    <p className="font-semibold">{examData?.totalMarks}</p>
                  </div>
                  <div>
                    <span className="text-muted-foreground">Questions:</span>
                    <p className="font-semibold">{examData?.questions.length}</p>
                  </div>
                </div>
              </div>

              <div>
                <h3 className="font-semibold text-lg mb-4">Instructions</h3>
                <ul className="space-y-2 text-sm">
                  <li className="flex items-start gap-2">
                    <span className="w-2 h-2 bg-secondary rounded-full mt-2 flex-shrink-0"></span>
                    <span>Use the stylus to write your answers clearly</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="w-2 h-2 bg-secondary rounded-full mt-2 flex-shrink-0"></span>
                    <span>Save your work frequently</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="w-2 h-2 bg-secondary rounded-full mt-2 flex-shrink-0"></span>
                    <span>Questions are presented one at a time</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="w-2 h-2 bg-secondary rounded-full mt-2 flex-shrink-0"></span>
                    <span>Submit before time expires</span>
                  </li>
                </ul>
              </div>
            </div>

            <Alert>
              <Shield className="h-4 w-4" />
              <AlertDescription>
                This is a secure examination environment. Any attempt to access external resources or communicate with
                others will result in disqualification.
              </AlertDescription>
            </Alert>

            <Button onClick={() => setCurrentStep("exam")} className="w-full text-lg py-6" size="lg">
              Start Examination
            </Button>
          </CardContent>
        </Card>
      </div>
    )
  }

  if (!examData) {
    return (
      <div className="min-h-screen flex items-center justify-center p-4">
        <Card className="w-full max-w-md"><CardContent className="py-12 text-center">Loading exam...</CardContent></Card>
      </div>
    )
  }

  const currentQ = examData.questions[currentQuestion]
  const progress = ((currentQuestion + 1) / examData.questions.length) * 100
  const isAnswered = !!answers[currentQ.id]

  return (
    <div className="min-h-screen bg-background">
      {/* Enhanced Header */}
      <div className="border-b bg-card">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <h1 className="text-xl font-bold">{examData.subject}</h1>
              <Badge variant="secondary">
                Question {currentQuestion + 1} of {examData.questions.length}
              </Badge>
              {isAnswered && (
                <Badge variant="default" className="bg-green-600">
                  <CheckCircle className="w-3 h-3 mr-1" />
                  Answered
                </Badge>
              )}
            </div>

            <div className="flex items-center gap-6">
              {/* Connection Status */}
              <div className="flex items-center gap-2">
                {isOnline ? <Wifi className="w-4 h-4 text-green-600" /> : <WifiOff className="w-4 h-4 text-red-600" />}
                <span className="text-sm">{isOnline ? "Online" : "Offline"}</span>
              </div>

              {/* Auto-save Status */}
              <div className="flex items-center gap-2">
                {autoSaveStatus === "saved" && <CheckCircle className="w-4 h-4 text-green-600" />}
                {autoSaveStatus === "saving" && <Circle className="w-4 h-4 text-blue-600 animate-spin" />}
                {autoSaveStatus === "error" && <AlertTriangle className="w-4 h-4 text-red-600" />}
                <span className="text-sm capitalize">{autoSaveStatus}</span>
              </div>

              {/* Timer with warning */}
              <div className={`flex items-center gap-2 ${showWarning ? "text-red-600 animate-pulse" : ""}`}>
                <Clock className="w-4 h-4" />
                <span className="font-mono text-lg font-semibold">{formatTime(timeRemaining)}</span>
              </div>
              <Progress value={progress} className="w-32" />
            </div>

            {/* Warning Alert */}
            {showWarning && (
              <Alert className="mt-4 border-red-200 bg-red-50">
                <AlertTriangle className="h-4 w-4 text-red-600" />
                <AlertDescription className="text-red-800">
                  Warning: Less than 5 minutes remaining! Your exam will auto-submit when time expires.
                </AlertDescription>
              </Alert>
            )}
          </div>
        </div>
      </div>

      <div className="container mx-auto px-4 py-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Enhanced Question Panel */}
          <div className="lg:col-span-1">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <span>Question {currentQ.id}</span>
                  <Badge>{currentQ.marks} marks</Badge>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-balance leading-relaxed">{currentQ.text}</p>
              </CardContent>
            </Card>

            {/* Exam Monitoring */}
            <Card className="mt-4">
              <CardHeader>
                <CardTitle className="text-sm">Exam Monitoring</CardTitle>
              </CardHeader>
              <CardContent>
                <CameraFeed
                  isTeacherView={false}
                  onStreamStart={(stream) => {
                    console.log("[v0] Student camera stream started")
                  }}
                  onStreamEnd={() => {
                    console.log("[v0] Student camera stream ended")
                  }}
                />
              </CardContent>
            </Card>

            {/* Question Navigator */}
            <Card className="mt-4">
              <CardHeader>
                <CardTitle className="text-sm">Question Navigator</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-5 gap-2">
                  {examData.questions.map((q, index) => (
                    <Button
                      key={q.id}
                      variant={index === currentQuestion ? "default" : "outline"}
                      size="sm"
                      className={`relative ${answers[q.id] ? "bg-green-100 border-green-300" : ""}`}
                      onClick={() => navigateToQuestion(index)}
                    >
                      {index + 1}
                      {answers[q.id] && <CheckCircle className="w-3 h-3 absolute -top-1 -right-1 text-green-600" />}
                    </Button>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Navigation */}
            <div className="mt-4 flex gap-2">
              <Button
                variant="outline"
                onClick={() => navigateToQuestion(Math.max(0, currentQuestion - 1))}
                disabled={currentQuestion === 0}
                className="flex-1"
              >
                <ChevronLeft className="w-4 h-4 mr-2" />
                Previous
              </Button>
              <Button
                variant="outline"
                onClick={() => navigateToQuestion(Math.min(examData.questions.length - 1, currentQuestion + 1))}
                disabled={currentQuestion === examData.questions.length - 1}
                className="flex-1"
              >
                Next
                <ChevronRight className="w-4 h-4 ml-2" />
              </Button>
            </div>
          </div>

          {/* Enhanced Answer Interface */}
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle>Answer Sheet</CardTitle>
                  <div className="flex items-center gap-2">
                    <Tabs value={answerMode} onValueChange={(value) => setAnswerMode(value as "canvas" | "text")}>
                      <TabsList>
                        <TabsTrigger value="canvas">Draw</TabsTrigger>
                        <TabsTrigger value="text">Type</TabsTrigger>
                      </TabsList>
                    </Tabs>
                  </div>
                </div>
              </CardHeader>
              <CardContent>
                <Tabs value={answerMode} onValueChange={(value) => setAnswerMode(value as "canvas" | "text")}>
                  <TabsContent value="canvas">
                    {/* Drawing Tools */}
                    <div className="flex items-center gap-2 mb-4 p-2 bg-muted rounded-lg">
                      <Button
                        variant={isEraser ? "default" : "outline"}
                        size="sm"
                        onClick={() => setIsEraser(!isEraser)}
                      >
                        {isEraser ? <Eraser className="w-4 h-4" /> : <PenTool className="w-4 h-4" />}
                      </Button>

                      <div className="flex items-center gap-2">
                        <input
                          type="color"
                          value={brushColor}
                          onChange={(e) => setBrushColor(e.target.value)}
                          className="w-8 h-8 rounded border"
                          disabled={isEraser}
                        />
                        <input
                          type="range"
                          min="1"
                          max="10"
                          value={brushSize}
                          onChange={(e) => setBrushSize(Number(e.target.value))}
                          className="w-20"
                        />
                        <span className="text-sm">{brushSize}px</span>
                      </div>

                      <div className="flex items-center gap-1">
                        <Button variant="outline" size="sm" onClick={handleUndo} disabled={historyStep <= 0}>
                          <Undo className="w-4 h-4" />
                        </Button>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={handleRedo}
                          disabled={historyStep >= drawingHistory.length - 1}
                        >
                          <Redo className="w-4 h-4" />
                        </Button>
                      </div>

                      <div className="flex items-center gap-1">
                        <Button variant="outline" size="sm" onClick={() => setZoom(Math.max(0.5, zoom - 0.1))}>
                          <ZoomOut className="w-4 h-4" />
                        </Button>
                        <span className="text-sm">{Math.round(zoom * 100)}%</span>
                        <Button variant="outline" size="sm" onClick={() => setZoom(Math.min(2, zoom + 0.1))}>
                          <ZoomIn className="w-4 h-4" />
                        </Button>
                      </div>

                      <Button variant="outline" size="sm" onClick={clearCanvas}>
                        Clear
                      </Button>
                      <Button variant="outline" size="sm" onClick={saveAnswer}>
                        <Save className="w-4 h-4 mr-2" />
                        Save
                      </Button>
                    </div>

                    <div className="border-2 border-dashed border-border rounded-lg p-4">
                      <canvas
                        ref={canvasRef}
                        width={800}
                        height={600}
                        className="w-full h-96 border border-border rounded cursor-crosshair bg-white"
                        style={{ transform: `scale(${zoom})`, transformOrigin: "top left" }}
                        onMouseDown={handleCanvasMouseDown}
                        onMouseMove={handleCanvasMouseMove}
                        onMouseUp={handleCanvasMouseUp}
                        onMouseLeave={handleCanvasMouseUp}
                        onTouchStart={handleTouchStart}
                        onTouchMove={handleTouchMove}
                        onTouchEnd={handleTouchEnd}
                      />
                      <p className="text-sm text-muted-foreground mt-2 text-center">
                        Use your stylus, mouse, or finger to write your answer above
                      </p>
                    </div>
                  </TabsContent>

                  <TabsContent value="text">
                    <Textarea
                      value={textAnswer}
                      onChange={(e) => setTextAnswer(e.target.value)}
                      placeholder="Type your answer here..."
                      className="min-h-96 font-mono"
                    />
                    <div className="mt-4 flex justify-end">
                      <Button variant="outline" size="sm" onClick={saveAnswer}>
                        <Save className="w-4 h-4 mr-2" />
                        Save Answer
                      </Button>
                    </div>
                  </TabsContent>
                </Tabs>
              </CardContent>
            </Card>

            {/* AI Stylus Processing */}
            <Card className="mt-4">
              <CardHeader>
                <CardTitle className="text-sm">AI Stylus Processing</CardTitle>
              </CardHeader>
              <CardContent>
                <StylusAIProcessor
                  studentId="student-001"
                  onTextProcessed={(text, analysis) => {
                    console.log("[v0] Student stylus text processed:", text, analysis)
                  }}
                  deliveryMethod="display"
                />
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Enhanced Submit Section */}
        <div className="mt-6 flex justify-center">
          <Button size="lg" className="px-8" onClick={handleSubmitExam} disabled={examSubmitted}>
            <Send className="w-4 h-4 mr-2" />
            {examSubmitted ? "Exam Submitted" : "Submit Examination"}
          </Button>
        </div>
      </div>
    </div>
  )
}
