"use client"

import type React from "react"

import { useState, useRef, useCallback } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Textarea } from "@/components/ui/textarea"
import { Progress } from "@/components/ui/progress"
import { Wand2, Brain, FileText, Loader2, CheckCircle, AlertCircle } from "lucide-react"

interface StylusAIProcessorProps {
  onTextProcessed?: (processedText: string, analysis: any) => void
  deliveryMethod?: "display" | "email" | "webhook"
  studentId?: string
}

export function StylusAIProcessor({ onTextProcessed, deliveryMethod = "display", studentId }: StylusAIProcessorProps) {
  const [isProcessing, setIsProcessing] = useState(false)
  const [progress, setProgress] = useState(0)
  const [handwrittenText, setHandwrittenText] = useState("")
  const [processedResult, setProcessedResult] = useState<any>(null)
  const [error, setError] = useState<string | null>(null)
  const canvasRef = useRef<HTMLCanvasElement>(null)
  const [isDrawing, setIsDrawing] = useState(false)

  const startDrawing = useCallback((e: React.MouseEvent<HTMLCanvasElement>) => {
    setIsDrawing(true)
    const canvas = canvasRef.current
    if (!canvas) return

    const rect = canvas.getBoundingClientRect()
    const ctx = canvas.getContext("2d")
    if (!ctx) return

    ctx.beginPath()
    ctx.moveTo(e.clientX - rect.left, e.clientY - rect.top)
  }, [])

  const draw = useCallback(
    (e: React.MouseEvent<HTMLCanvasElement>) => {
      if (!isDrawing) return

      const canvas = canvasRef.current
      if (!canvas) return

      const rect = canvas.getBoundingClientRect()
      const ctx = canvas.getContext("2d")
      if (!ctx) return

      ctx.lineWidth = 2
      ctx.lineCap = "round"
      ctx.strokeStyle = "#3b82f6"
      ctx.lineTo(e.clientX - rect.left, e.clientY - rect.top)
      ctx.stroke()
    },
    [isDrawing],
  )

  const stopDrawing = useCallback(() => {
    setIsDrawing(false)
  }, [])

  const clearCanvas = () => {
    const canvas = canvasRef.current
    if (!canvas) return

    const ctx = canvas.getContext("2d")
    if (!ctx) return

    ctx.clearRect(0, 0, canvas.width, canvas.height)
    setHandwrittenText("")
    setProcessedResult(null)
    setError(null)
  }

  const processHandwriting = async () => {
    const canvas = canvasRef.current
    if (!canvas) return

    setIsProcessing(true)
    setProgress(0)
    setError(null)

    try {
      console.log("[v0] Starting handwriting recognition...")

      // Simulate AI processing steps
      setProgress(25)
      await new Promise((resolve) => setTimeout(resolve, 1000))

      // Convert canvas to image data
      const imageData = canvas.toDataURL("image/png")
      console.log("[v0] Canvas data extracted")

      setProgress(50)
      await new Promise((resolve) => setTimeout(resolve, 1000))

      // Placeholder OCR: no model connected yet
      const recognizedText = ""
      setHandwrittenText(recognizedText)

      setProgress(75)
      await new Promise((resolve) => setTimeout(resolve, 500))

      // Minimal analysis placeholder
      const analysis = {
        confidence: 0,
        wordCount: 0,
        sentiment: "neutral",
        topics: [],
        suggestions: [],
      }

      setProgress(100)
      const result = {
        text: recognizedText,
        analysis,
        timestamp: new Date().toISOString(),
        deliveryMethod: deliveryMethod,
      }
      setProcessedResult(result)

      onTextProcessed?.(recognizedText, analysis)

      // Publish to SSE for teacher realtime view
      if (studentId) {
        fetch("/api/realtime/publish", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({ studentId, payload: { type: "stylus-text", ...result } }),
        }).catch(() => {})
      }

      console.log("[v0] AI processing completed (placeholder)")
    } catch (err) {
      console.error("[v0] Processing error:", err)
      setError("Failed to process handwriting. Please try again.")
    } finally {
      setIsProcessing(false)
    }
  }

  return (
    <Card className="glass-effect">
      <CardHeader className="pb-4">
        <div className="flex items-center justify-between">
          <CardTitle className="text-lg flex items-center gap-2">
            <Wand2 className="h-5 w-5" />
            AI Stylus Processor
          </CardTitle>
          <Badge variant={processedResult ? "default" : "secondary"} className="flex items-center gap-1">
            {processedResult ? <CheckCircle className="h-3 w-3" /> : <Brain className="h-3 w-3" />}
            {processedResult ? "Processed" : "Ready"}
          </Badge>
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Drawing Canvas */}
        <div className="space-y-2">
          <label className="text-sm font-medium">Handwriting Canvas</label>
          <div className="border border-border rounded-lg p-4 bg-background">
            <canvas
              ref={canvasRef}
              width={400}
              height={200}
              className="w-full border border-dashed border-muted-foreground/30 rounded cursor-crosshair"
              onMouseDown={startDrawing}
              onMouseMove={draw}
              onMouseUp={stopDrawing}
              onMouseLeave={stopDrawing}
            />
            <div className="flex gap-2 mt-2">
              <Button onClick={clearCanvas} variant="outline" size="sm">
                Clear Canvas
              </Button>
              <Button onClick={processHandwriting} disabled={isProcessing} size="sm" className="flex-1">
                {isProcessing ? (
                  <>
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                    Processing...
                  </>
                ) : (
                  <>
                    <Brain className="h-4 w-4 mr-2" />
                    Process Handwriting
                  </>
                )}
              </Button>
            </div>
          </div>
        </div>

        {/* Processing Progress */}
        {isProcessing && (
          <div className="space-y-2">
            <div className="flex items-center justify-between text-sm">
              <span>Processing handwriting...</span>
              <span>{progress}%</span>
            </div>
            <Progress value={progress} className="h-2" />
          </div>
        )}

        {/* Error Display */}
        {error && (
          <div className="p-3 bg-destructive/10 border border-destructive/20 rounded-lg flex items-center gap-2">
            <AlertCircle className="h-4 w-4 text-destructive" />
            <p className="text-sm text-destructive">{error}</p>
          </div>
        )}

        {/* Recognized Text */}
        {handwrittenText && (
          <div className="space-y-2">
            <label className="text-sm font-medium flex items-center gap-2">
              <FileText className="h-4 w-4" />
              Recognized Text
            </label>
            <Textarea
              value={handwrittenText}
              onChange={(e) => setHandwrittenText(e.target.value)}
              placeholder="Recognized text will appear here..."
              className="min-h-[100px]"
            />
          </div>
        )}

        {/* AI Analysis Results */}
        {processedResult && (
          <div className="space-y-3 p-4 bg-muted/50 rounded-lg">
            <h4 className="font-medium flex items-center gap-2">
              <Brain className="h-4 w-4" />
              AI Analysis Results
            </h4>
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div>
                <span className="text-muted-foreground">Confidence:</span>
                <span className="ml-2 font-medium">{(processedResult.analysis.confidence * 100).toFixed(1)}%</span>
              </div>
              <div>
                <span className="text-muted-foreground">Word Count:</span>
                <span className="ml-2 font-medium">{processedResult.analysis.wordCount}</span>
              </div>
              <div>
                <span className="text-muted-foreground">Sentiment:</span>
                <span className="ml-2 font-medium capitalize">{processedResult.analysis.sentiment}</span>
              </div>
              <div>
                <span className="text-muted-foreground">Delivery:</span>
                <span className="ml-2 font-medium capitalize">{deliveryMethod}</span>
              </div>
            </div>

            {processedResult.analysis.suggestions && (
              <div className="space-y-1">
                <span className="text-sm text-muted-foreground">AI Suggestions:</span>
                <ul className="text-sm space-y-1">
                  {processedResult.analysis.suggestions.map((suggestion: string, index: number) => (
                    <li key={index} className="flex items-start gap-2">
                      <CheckCircle className="h-3 w-3 text-primary mt-0.5 flex-shrink-0" />
                      {suggestion}
                    </li>
                  ))}
                </ul>
              </div>
            )}
          </div>
        )}
      </CardContent>
    </Card>
  )
}
