"use client"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Progress } from "@/components/ui/progress"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { BookOpen, MapPin, Vote, Target, TrendingUp, CheckCircle, AlertCircle } from "lucide-react"
import {
  getSubjectEvaluation,
  calculateCriteriaScore,
  getGradeFromPercentage,
  generateSubjectSpecificFeedback,
} from "@/lib/subject-evaluation"

interface SubjectSpecificGradingProps {
  subject: "History" | "Geography" | "Political Science"
  answer: string
  totalMarks: number
  awardedMarks: number
}

export function SubjectSpecificGrading({ subject, answer, totalMarks, awardedMarks }: SubjectSpecificGradingProps) {
  const evaluation = getSubjectEvaluation(subject)
  const percentage = (awardedMarks / totalMarks) * 100
  const grade = getGradeFromPercentage(percentage, evaluation.gradingRubric)
  const feedback = generateSubjectSpecificFeedback(subject, answer, totalMarks, awardedMarks)

  const getSubjectIcon = (subject: string) => {
    switch (subject) {
      case "History":
        return <BookOpen className="w-5 h-5" />
      case "Geography":
        return <MapPin className="w-5 h-5" />
      case "Political Science":
        return <Vote className="w-5 h-5" />
      default:
        return <BookOpen className="w-5 h-5" />
    }
  }

  const getGradeColor = (grade: string) => {
    switch (grade) {
      case "Excellent":
        return "bg-green-100 text-green-800"
      case "Good":
        return "bg-blue-100 text-blue-800"
      case "Satisfactory":
        return "bg-yellow-100 text-yellow-800"
      case "Needs Improvement":
        return "bg-red-100 text-red-800"
      default:
        return "bg-gray-100 text-gray-800"
    }
  }

  return (
    <div className="space-y-6">
      {/* Subject Overview */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <span className="flex items-center gap-2">
              {getSubjectIcon(subject)}
              {subject} - Subject-Specific Evaluation
            </span>
            <Badge className={getGradeColor(grade)}>
              {grade} ({percentage.toFixed(1)}%)
            </Badge>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-primary">{awardedMarks}</div>
              <div className="text-sm text-muted-foreground">Marks Awarded</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold">{totalMarks}</div>
              <div className="text-sm text-muted-foreground">Total Marks</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">{percentage.toFixed(1)}%</div>
              <div className="text-sm text-muted-foreground">Percentage</div>
            </div>
          </div>
        </CardContent>
      </Card>

      <Tabs defaultValue="criteria" className="space-y-4">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="criteria">Evaluation Criteria</TabsTrigger>
          <TabsTrigger value="rubric">Grading Rubric</TabsTrigger>
          <TabsTrigger value="feedback">Detailed Feedback</TabsTrigger>
          <TabsTrigger value="improvement">Improvement Plan</TabsTrigger>
        </TabsList>

        <TabsContent value="criteria" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Target className="w-5 h-5" />
                {subject} Evaluation Criteria
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {evaluation.criteria.map((criterion, index) => {
                  const score = calculateCriteriaScore(answer, criterion)
                  const percentage = (score / criterion.maxPoints) * 100

                  return (
                    <div key={index} className="p-4 border rounded-lg">
                      <div className="flex items-center justify-between mb-2">
                        <h4 className="font-medium">{criterion.name}</h4>
                        <Badge variant="outline">
                          {score.toFixed(1)}/{criterion.maxPoints}
                        </Badge>
                      </div>
                      <p className="text-sm text-muted-foreground mb-3">{criterion.description}</p>
                      <div className="space-y-2">
                        <div className="flex items-center justify-between text-xs">
                          <span>Performance</span>
                          <span>{percentage.toFixed(1)}%</span>
                        </div>
                        <Progress value={percentage} className="h-2" />
                      </div>
                      <div className="mt-2">
                        <p className="text-xs text-muted-foreground">
                          Key areas: {criterion.keywords.slice(0, 3).join(", ")}
                          {criterion.keywords.length > 3 && "..."}
                        </p>
                      </div>
                    </div>
                  )
                })}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="rubric" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Grading Rubric - {subject}</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {Object.entries(evaluation.gradingRubric).map(([level, details]) => (
                  <div
                    key={level}
                    className={`p-4 border rounded-lg ${
                      grade.toLowerCase().replace(" ", "") === level ? "border-primary bg-primary/5" : ""
                    }`}
                  >
                    <div className="flex items-center justify-between mb-2">
                      <h4 className="font-medium capitalize">{level.replace(/([A-Z])/g, " $1").trim()}</h4>
                      <Badge variant="outline">
                        {details.range[0]}% - {details.range[1]}%
                      </Badge>
                    </div>
                    <p className="text-sm text-muted-foreground">{details.description}</p>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="feedback" className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-green-600">
                  <CheckCircle className="w-5 h-5" />
                  Strengths Identified
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  {feedback.strengths.length > 0 ? (
                    feedback.strengths.map((strength, index) => (
                      <div key={index} className="flex items-start gap-2">
                        <CheckCircle className="w-4 h-4 text-green-600 mt-0.5 flex-shrink-0" />
                        <p className="text-sm">{strength}</p>
                      </div>
                    ))
                  ) : (
                    <p className="text-sm text-muted-foreground">
                      Continue working to develop stronger responses in this subject area.
                    </p>
                  )}
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-orange-600">
                  <AlertCircle className="w-5 h-5" />
                  Areas for Improvement
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  {feedback.improvements.map((improvement, index) => (
                    <div key={index} className="flex items-start gap-2">
                      <AlertCircle className="w-4 h-4 text-orange-600 mt-0.5 flex-shrink-0" />
                      <p className="text-sm">{improvement}</p>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="improvement" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <TrendingUp className="w-5 h-5" />
                Subject-Specific Improvement Tips
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {feedback.subjectSpecificTips.map((tip, index) => (
                  <div key={index} className="flex items-start gap-3 p-3 bg-muted rounded-lg">
                    <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-xs font-bold flex-shrink-0">
                      {index + 1}
                    </div>
                    <p className="text-sm">{tip}</p>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Sample Questions for Practice */}
          <Card>
            <CardHeader>
              <CardTitle>Practice Questions - {subject}</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {evaluation.sampleQuestions.map((question, index) => (
                  <div key={index} className="p-4 border rounded-lg">
                    <div className="flex items-center justify-between mb-2">
                      <Badge variant="outline">{question.type}</Badge>
                      <span className="text-sm font-medium">{question.marks} marks</span>
                    </div>
                    <p className="text-sm font-medium mb-2">{question.question}</p>
                    <div className="text-xs text-muted-foreground space-y-1">
                      <p>Expected length: {question.expectedLength}</p>
                      <p>Key points to cover: {question.keyPoints.join(", ")}</p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
}
