"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Progress } from "@/components/ui/progress"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Input } from "@/components/ui/input"
import { ScrollArea } from "@/components/ui/scroll-area"
import {
  Users,
  FileText,
  CheckCircle,
  TrendingUp,
  Eye,
  Settings,
  Brain,
  Star,
  RefreshCw,
  Search,
  Filter,
  BookOpen,
  MapPin,
  Vote,
} from "lucide-react"

// Import new components and utility functions
import { AIGradingInterface } from "@/components/ai-grading-interface"
import { SubjectSpecificGrading } from "@/components/subject-specific-grading"
import { gradeAnswerWithAI } from "@/lib/ai-grading"
import { LiveMonitoringInterface } from "@/components/live-monitoring-interface"

interface StudentAnswer {
  id: string
  questionId: string
  question: string
  studentAnswer: string
  correctAnswer: string
  subject: "History" | "Geography" | "Political Science"
  marks: number
  totalMarks: number
  similarityScore: number
  aiGrade?: AIGradeResult
  submittedAt: string
}

interface AIGradeResult {
  totalMarks: number
  awardedMarks: number
  pointBreakdown: {
    point: string
    awarded: boolean
    reason: string
  }[]
  overallFeedback: string
  suggestions: string[]
  similarityScore?: number // Added similarityScore to AIGradeResult
}

interface Student {
  id: string
  name: string
  email: string
  status: "in-progress" | "completed" | "not-started"
  progress: number
  timeRemaining: number
  score?: number
  answers: StudentAnswer[]
}

interface ExamSession {
  id: string
  subject: "History" | "Geography" | "Political Science"
  title: string
  date: string
  duration: number
  totalStudents: number
  completed: number
  inProgress: number
  notStarted: number
  questions: ExamQuestion[]
}

interface ExamQuestion {
  id: string
  question: string
  correctAnswer: string
  marks: number
  type: "descriptive" | "analytical" | "case-study"
}

const mockClass12Questions: ExamQuestion[] = [
  {
    id: "hist1",
    question:
      "Analyze the causes and consequences of the Revolt of 1857. How did it impact British colonial policy in India?",
    correctAnswer:
      "The Revolt of 1857 had multiple causes including economic exploitation through land revenue policies, social reforms causing unrest, military grievances over cartridge grease, and political annexations using Doctrine of Lapse. Key consequences included end of East India Company rule, introduction of direct Crown rule, policy changes in military recruitment, and increased communal divide. The British became more cautious about social reforms and adopted a policy of divide and rule.",
    marks: 12,
    type: "analytical",
  },
  {
    id: "geo1",
    question:
      "Explain the concept of sustainable development with reference to India's environmental challenges. Discuss any three major environmental issues.",
    correctAnswer:
      "Sustainable development meets present needs without compromising future generations' ability to meet their needs. India faces major environmental challenges: 1) Air pollution in cities like Delhi-NCR due to vehicular emissions and industrial activities, 2) Water scarcity in western regions due to over-extraction and climate change, 3) Deforestation in Northeast and Western Ghats affecting biodiversity. Solutions include renewable energy adoption, water conservation techniques, and afforestation programs.",
    marks: 10,
    type: "descriptive",
  },
  {
    id: "pol1",
    question:
      "Evaluate the role of regional parties in Indian democracy. How have they influenced coalition politics at the center?",
    correctAnswer:
      "Regional parties represent local interests, provide federal balance, and have been crucial in coalition governments since 1990s. They've influenced policy-making on agriculture, language, and regional development. Examples include DMK's influence on Tamil issues, JD(U)'s role in Bihar development, and AIADMK's impact on federal policies. They've strengthened democratic participation and ensured regional representation in national politics, though sometimes leading to policy instability.",
    marks: 12,
    type: "case-study",
  },
  {
    id: "hist2",
    question:
      "Evaluate the role of Mahatma Gandhi in the Indian freedom struggle. Discuss his methods and their effectiveness.",
    correctAnswer:
      "Gandhi's role was pivotal through his philosophy of non-violent resistance (Satyagraha). His methods included mass movements like Non-cooperation (1920-22) and Civil Disobedience (1930-34), with the Salt March being particularly effective in mobilizing masses and gaining international attention. The Quit India Movement (1942) demonstrated the depth of anti-British sentiment. His effectiveness lay in making the freedom struggle a mass movement, gaining international sympathy, and providing moral leadership.",
    marks: 15,
    type: "analytical",
  },
  {
    id: "geo2",
    question:
      "Analyze the distribution pattern of iron ore in India and its impact on the steel industry. Discuss the challenges faced by the industry.",
    correctAnswer:
      "Iron ore is concentrated in Jharkhand (Singhbhum), Odisha (Mayurbhanj, Keonjhar), and Chhattisgarh (Bastar) forming the mineral belt. This has led to steel plants being located in these regions like Jamshedpur (TISCO), Rourkela (RSP), and Bhilai (BSP). Challenges include high transportation costs to coastal areas, environmental degradation from mining, competition from cheaper imports, and need for technological upgradation.",
    marks: 12,
    type: "analytical",
  },
  {
    id: "pol2",
    question:
      "Discuss the challenges to democracy in contemporary India. Suggest measures to strengthen democratic institutions.",
    correctAnswer:
      "Contemporary challenges include corruption in political system, criminalization of politics with candidates having criminal records, excessive money power in elections, communalism and identity politics, media manipulation, and weakening of institutions like Election Commission and judiciary. Measures include electoral reforms (state funding of elections), transparency in political funding, strengthening anti-corruption mechanisms like Lokpal, promoting civic education, and ensuring independence of institutions.",
    marks: 15,
    type: "analytical",
  },
]

const mockStudentAnswers: StudentAnswer[] = [
  {
    id: "ans1",
    questionId: "hist1",
    question:
      "Analyze the causes and consequences of the Revolt of 1857. How did it impact British colonial policy in India?",
    studentAnswer:
      "The Revolt of 1857 was caused by economic problems, religious interference, and military issues. The British were taking over kingdoms using the Doctrine of Lapse. Sepoys were angry about cartridges with cow and pig fat. After the revolt, the British government took direct control from the East India Company. They also changed their policies to be more careful about Indian customs and started divide and rule policy.",
    correctAnswer:
      "The Revolt of 1857 had multiple causes including economic exploitation through land revenue policies, social reforms causing unrest, military grievances over cartridge grease, and political annexations using Doctrine of Lapse. Key consequences included end of East India Company rule, introduction of direct Crown rule, policy changes in military recruitment, and increased communal divide.",
    subject: "History",
    marks: 8,
    totalMarks: 12,
    similarityScore: 75,
    submittedAt: "2024-01-15T10:30:00Z",
  },
  {
    id: "ans2",
    questionId: "geo1",
    question:
      "Explain the concept of sustainable development with reference to India's environmental challenges. Discuss any three major environmental issues.",
    studentAnswer:
      "Sustainable development means development that doesn't harm the environment for future generations. India has pollution problems in cities like Delhi with smog and air quality issues. Water is becoming scarce in many places like Rajasthan and Maharashtra. Forests are being cut down for development projects. We need to use renewable energy like solar and wind power and plant more trees to solve these problems.",
    correctAnswer:
      "Sustainable development meets present needs without compromising future generations. India faces air pollution, water scarcity, and deforestation. Solutions include renewable energy, water conservation, and afforestation programs.",
    subject: "Geography",
    marks: 7,
    totalMarks: 10,
    similarityScore: 78,
    submittedAt: "2024-01-15T11:15:00Z",
  },
  {
    id: "ans3",
    questionId: "pol1",
    question:
      "Evaluate the role of regional parties in Indian democracy. How have they influenced coalition politics at the center?",
    studentAnswer:
      "Regional parties play important role in Indian democracy by representing local people's interests. They have become powerful since 1990s when coalition governments started. Parties like DMK in Tamil Nadu, JD(U) in Bihar, and Trinamool Congress in West Bengal have influenced national policies. They help in making policies for their states and ensure regional development. Coalition politics has made them kingmakers in forming governments at center.",
    correctAnswer:
      "Regional parties represent local interests, provide federal balance, and have been crucial in coalition governments since 1990s. They've influenced policy-making on agriculture, language, and regional development. Examples include DMK's influence on Tamil issues, JD(U)'s role in Bihar development, and AIADMK's impact on federal policies.",
    subject: "Political Science",
    marks: 9,
    totalMarks: 12,
    similarityScore: 82,
    submittedAt: "2024-01-15T12:00:00Z",
  },
]

const mockExamSessions: ExamSession[] = [
  {
    id: "1",
    subject: "History",
    title: "Modern Indian History - Freedom Struggle",
    date: "2024-01-15",
    duration: 180,
    totalStudents: 45,
    completed: 12,
    inProgress: 8,
    notStarted: 25,
    questions: mockClass12Questions.filter((q) => q.id.startsWith("hist")),
  },
  {
    id: "2",
    subject: "Geography",
    title: "Environmental Geography and Sustainability",
    date: "2024-01-14",
    duration: 120,
    totalStudents: 38,
    completed: 38,
    inProgress: 0,
    notStarted: 0,
    questions: mockClass12Questions.filter((q) => q.id.startsWith("geo")),
  },
  {
    id: "3",
    subject: "Political Science",
    title: "Indian Democracy and Governance",
    date: "2024-01-16",
    duration: 150,
    totalStudents: 42,
    completed: 5,
    inProgress: 15,
    notStarted: 22,
    questions: mockClass12Questions.filter((q) => q.id.startsWith("pol")),
  },
]

const mockStudents: Student[] = []

export function TeacherDashboard() {
  const [selectedExam, setSelectedExam] = useState(mockExamSessions[0])
  const [selectedStudent, setSelectedStudent] = useState<Student | null>(null)
  const [selectedAnswer, setSelectedAnswer] = useState<StudentAnswer | null>(null)
  const [isGrading, setIsGrading] = useState(false)
  const [searchTerm, setSearchTerm] = useState("")
  const [subjectFilter, setSubjectFilter] = useState<string>("all")
  const [showDetailedGrading, setShowDetailedGrading] = useState(false)
  const [gradingHistory, setGradingHistory] = useState<Record<string, AIGradeResult>>({})

  const gradeWithAI = async (answer: StudentAnswer) => {
    setIsGrading(true)

    try {
      const result = await gradeAnswerWithAI({
        question: answer.question,
        studentAnswer: answer.studentAnswer,
        correctAnswer: answer.correctAnswer,
        subject: answer.subject,
        totalMarks: answer.totalMarks,
      })

      // Update the answer with AI grade
      if (selectedAnswer && selectedAnswer.id === answer.id) {
        selectedAnswer.aiGrade = result
      }

      // Store in grading history
      setGradingHistory((prev) => ({
        ...prev,
        [answer.id]: result,
      }))
    } catch (error) {
      console.error("Failed to grade answer:", error)
      // Fallback mock result
      const mockAIResult: AIGradeResult = {
        totalMarks: answer.totalMarks,
        awardedMarks: answer.marks,
        pointBreakdown: [
          { point: "Understanding of concept", awarded: true, reason: "Student demonstrates good grasp" },
          { point: "Factual accuracy", awarded: true, reason: "Most facts are correct" },
          { point: "Analysis and evaluation", awarded: false, reason: "Limited critical analysis" },
          { point: "Use of examples", awarded: false, reason: "Could include more examples" },
          { point: "Structure and clarity", awarded: true, reason: "Well-organized response" },
        ],
        overallFeedback: "Good understanding with room for deeper analysis and more specific examples.",
        suggestions: [
          "Include more specific examples",
          "Develop critical analysis further",
          "Use subject-specific terminology",
        ],
        similarityScore: answer.similarityScore,
      }

      if (selectedAnswer && selectedAnswer.id === answer.id) {
        selectedAnswer.aiGrade = mockAIResult
      }

      setGradingHistory((prev) => ({
        ...prev,
        [answer.id]: mockAIResult,
      }))
    } finally {
      setIsGrading(false)
    }
  }

  const handleGradeUpdate = (answerId: string, grade: AIGradeResult) => {
    setGradingHistory((prev) => ({
      ...prev,
      [answerId]: grade,
    }))
  }

  const getStatusColor = (status: Student["status"]) => {
    switch (status) {
      case "completed":
        return "bg-green-100 text-green-800"
      case "in-progress":
        return "bg-blue-100 text-blue-800"
      case "not-started":
        return "bg-gray-100 text-gray-800"
    }
  }

  const getSubjectIcon = (subject: string) => {
    switch (subject) {
      case "History":
        return <BookOpen className="w-4 h-4" />
      case "Geography":
        return <MapPin className="w-4 h-4" />
      case "Political Science":
        return <Vote className="w-4 h-4" />
      default:
        return <FileText className="w-4 h-4" />
    }
  }

  const formatTime = (minutes: number) => {
    const hours = Math.floor(minutes / 60)
    const mins = minutes % 60
    return `${hours}h ${mins}m`
  }

  const filteredStudents = mockStudents.filter((student) => {
    const matchesSearch =
      student.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      student.email.toLowerCase().includes(searchTerm.toLowerCase())
    return matchesSearch
  })

  const filteredExams = mockExamSessions.filter((exam) => {
    return subjectFilter === "all" || exam.subject === subjectFilter
  })

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">Teacher Dashboard</h1>
              <p className="text-muted-foreground">
                AI-Powered Class 12 Examination System - History, Geography & Political Science
              </p>
            </div>
            <div className="flex items-center gap-4">
              <Button variant="outline" size="sm">
                <Settings className="w-4 h-4 mr-2" />
                Settings
              </Button>
              <Avatar>
                <AvatarImage src="/placeholder.svg?height=32&width=32" />
                <AvatarFallback>DR</AvatarFallback>
              </Avatar>
            </div>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-6 py-6">
        <Tabs defaultValue="overview" className="space-y-6">
          <TabsList className="grid w-full grid-cols-6">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="live-monitoring">Live Monitoring</TabsTrigger>
            <TabsTrigger value="submissions">Student Submissions</TabsTrigger>
            <TabsTrigger value="ai-grading">AI Grading</TabsTrigger>
            <TabsTrigger value="analytics">Analytics</TabsTrigger>
            <TabsTrigger value="camera-monitoring">Camera Monitoring</TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="space-y-6">
            {/* Subject Filter */}
            <div className="flex items-center gap-4">
              <div className="flex items-center gap-2">
                <Filter className="w-4 h-4" />
                <span className="text-sm font-medium">Filter by Subject:</span>
              </div>
              <div className="flex gap-2">
                <Button
                  variant={subjectFilter === "all" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setSubjectFilter("all")}
                >
                  All Subjects
                </Button>
                <Button
                  variant={subjectFilter === "History" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setSubjectFilter("History")}
                >
                  <BookOpen className="w-4 h-4 mr-2" />
                  History
                </Button>
                <Button
                  variant={subjectFilter === "Geography" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setSubjectFilter("Geography")}
                >
                  <MapPin className="w-4 h-4 mr-2" />
                  Geography
                </Button>
                <Button
                  variant={subjectFilter === "Political Science" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setSubjectFilter("Political Science")}
                >
                  <Vote className="w-4 h-4 mr-2" />
                  Political Science
                </Button>
              </div>
            </div>

            {/* Stats Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Students</CardTitle>
                  <Users className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{selectedExam.totalStudents}</div>
                  <p className="text-xs text-muted-foreground">Enrolled in current exam</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Completed</CardTitle>
                  <CheckCircle className="h-4 w-4 text-green-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-green-600">{selectedExam.completed}</div>
                  <p className="text-xs text-muted-foreground">
                    {Math.round((selectedExam.completed / selectedExam.totalStudents) * 100)}% completion rate
                  </p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">AI Graded</CardTitle>
                  <Brain className="h-4 w-4 text-purple-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-purple-600">{Object.keys(gradingHistory).length || "—"}</div>
                  <p className="text-xs text-muted-foreground">Automatically evaluated</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Avg Score</CardTitle>
                  <Star className="h-4 w-4 text-yellow-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-yellow-600">{
                    (() => {
                      const vals = Object.values(gradingHistory) as any[]
                      if (!vals.length) return "—"
                      const avg = vals.reduce((s, g) => s + Math.round((g.awardedMarks / g.totalMarks) * 100), 0) / vals.length
                      return `${Math.round(avg)}%`
                    })()
                  }</div>
                  <p className="text-xs text-muted-foreground">Class average</p>
                </CardContent>
              </Card>
            </div>

            {/* Recent Exam Sessions */}
            <Card>
              <CardHeader>
                <CardTitle>Class 12 Exam Sessions</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {filteredExams.map((exam) => (
                    <div key={exam.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <div className="w-12 h-12 bg-secondary rounded-lg flex items-center justify-center">
                          {getSubjectIcon(exam.subject)}
                        </div>
                        <div>
                          <h3 className="font-semibold">{exam.title}</h3>
                          <p className="text-sm text-muted-foreground">
                            {exam.subject} • {exam.date} • {formatTime(exam.duration)}
                          </p>
                        </div>
                      </div>
                      <div className="flex items-center gap-4">
                        <div className="text-right">
                          <p className="text-sm font-medium">{exam.totalStudents} students</p>
                          <p className="text-xs text-muted-foreground">{exam.completed} completed</p>
                        </div>
                        <Button variant="outline" size="sm" onClick={() => setSelectedExam(exam)}>
                          <Eye className="w-4 h-4 mr-2" />
                          View
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="live-monitoring" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <span>Live Student Monitoring - {selectedExam?.subject ?? ""}</span>
                  <Badge variant="secondary">0 Active</Badge>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {mockStudents.map((student) => (
                    <div key={student.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <Avatar>
                          <AvatarImage src={`/generic-placeholder-graphic.png?height=40&width=40`} />
                          <AvatarFallback>
                            {student.name
                              .split(" ")
                              .map((n) => n[0])
                              .join("")}
                          </AvatarFallback>
                        </Avatar>
                        <div>
                          <h3 className="font-semibold">{student.name}</h3>
                          <p className="text-sm text-muted-foreground">{student.email}</p>
                        </div>
                      </div>

                      <div className="flex items-center gap-6">
                        <div className="text-right">
                          <Badge className={getStatusColor(student.status)}>{student.status.replace("-", " ")}</Badge>
                          {student.status === "in-progress" && (
                            <p className="text-xs text-muted-foreground mt-1">
                              {formatTime(student.timeRemaining)} remaining
                            </p>
                          )}
                        </div>

                        <div className="w-32">
                          <div className="flex items-center justify-between text-xs mb-1">
                            <span>Progress</span>
                            <span>{student.progress}%</span>
                          </div>
                          <Progress value={student.progress} className="h-2" />
                        </div>

                        {student.score && (
                          <div className="text-right">
                            <p className="text-lg font-bold">{student.score}%</p>
                            <p className="text-xs text-muted-foreground">Score</p>
                          </div>
                        )}

                        <Button variant="outline" size="sm">
                          <Eye className="w-4 h-4 mr-2" />
                          Monitor
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="submissions" className="space-y-6">
            <div className="flex items-center gap-4 mb-6">
              <div className="relative flex-1 max-w-md">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                <Input
                  placeholder="Search students..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
              <Button
                variant={showDetailedGrading ? "default" : "outline"}
                onClick={() => setShowDetailedGrading(!showDetailedGrading)}
              >
                <Brain className="w-4 h-4 mr-2" />
                {showDetailedGrading ? "Simple View" : "Detailed Grading"}
              </Button>
            </div>

            {showDetailedGrading && selectedAnswer ? (
              <div className="space-y-6">
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center justify-between">
                      <span>Detailed Grading - {selectedStudent?.name}</span>
                      <Button
                        variant="outline"
                        onClick={() => {
                          setShowDetailedGrading(false)
                          setSelectedAnswer(null)
                        }}
                      >
                        Back to List
                      </Button>
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <Tabs defaultValue="ai-grading" className="space-y-4">
                      <TabsList className="grid w-full grid-cols-2">
                        <TabsTrigger value="ai-grading">AI Grading</TabsTrigger>
                        <TabsTrigger value="subject-analysis">Subject Analysis</TabsTrigger>
                      </TabsList>

                      <TabsContent value="ai-grading">
                        <AIGradingInterface answer={selectedAnswer} onGradeUpdate={handleGradeUpdate} />
                      </TabsContent>

                      <TabsContent value="subject-analysis">
                        <SubjectSpecificGrading
                          subject={selectedAnswer.subject}
                          answer={selectedAnswer.studentAnswer}
                          totalMarks={selectedAnswer.totalMarks}
                          awardedMarks={selectedAnswer.aiGrade?.awardedMarks || selectedAnswer.marks}
                        />
                      </TabsContent>
                    </Tabs>
                  </CardContent>
                </Card>
              </div>
            ) : (
              <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                {/* Student List */}
                <Card className="lg:col-span-1">
                  <CardHeader>
                    <CardTitle className="flex items-center justify-between">
                      <span>Students - {selectedExam.title}</span>
                      <Badge variant="secondary">{filteredStudents.length}</Badge>
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <ScrollArea className="h-96">
                      <div className="space-y-2">
                        {filteredStudents.map((student) => (
                          <div
                            key={student.id}
                            className={`p-3 rounded-lg border cursor-pointer transition-colors ${
                              selectedStudent?.id === student.id ? "bg-primary/10 border-primary" : "hover:bg-muted"
                            }`}
                            onClick={() => setSelectedStudent(student)}
                          >
                            <div className="flex items-center gap-3">
                              <Avatar className="w-8 h-8">
                                <AvatarFallback className="text-xs">
                                  {student.name
                                    .split(" ")
                                    .map((n) => n[0])
                                    .join("")}
                                </AvatarFallback>
                              </Avatar>
                              <div className="flex-1 min-w-0">
                                <p className="font-medium text-sm truncate">{student.name}</p>
                                <p className="text-xs text-muted-foreground truncate">{student.email}</p>
                              </div>
                              <div className="flex flex-col items-end gap-1">
                                <Badge className={getStatusColor(student.status)} variant="secondary">
                                  {student.status === "completed"
                                    ? "Done"
                                    : student.status === "in-progress"
                                      ? "Active"
                                      : "Pending"}
                                </Badge>
                                {student.answers.length > 0 && (
                                  <div className="flex items-center gap-1">
                                    <Brain className="w-3 h-3 text-purple-600" />
                                    <span className="text-xs text-purple-600">
                                      {student.answers.filter((a) => gradingHistory[a.id]).length}/
                                      {student.answers.length}
                                    </span>
                                  </div>
                                )}
                              </div>
                            </div>
                          </div>
                        ))}
                      </div>
                    </ScrollArea>
                  </CardContent>
                </Card>

                {/* Student Answers */}
                <Card className="lg:col-span-2">
                  <CardHeader>
                    <CardTitle className="flex items-center justify-between">
                      <span>
                        {selectedStudent
                          ? `${selectedStudent.name}'s Submissions`
                          : "Select a student to view submissions"}
                      </span>
                      {selectedStudent && selectedStudent.answers.length > 0 && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={async () => {
                            for (const answer of selectedStudent.answers) {
                              if (!gradingHistory[answer.id]) {
                                await gradeWithAI(answer)
                              }
                            }
                          }}
                          disabled={isGrading}
                        >
                          {isGrading ? (
                            <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                          ) : (
                            <Brain className="w-4 h-4 mr-2" />
                          )}
                          Grade All
                        </Button>
                      )}
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    {selectedStudent ? (
                      <ScrollArea className="h-96">
                        <div className="space-y-4">
                          {selectedStudent.answers.length > 0 ? (
                            selectedStudent.answers.map((answer) => {
                              const hasAIGrade = gradingHistory[answer.id] || answer.aiGrade

                              return (
                                <div
                                  key={answer.id}
                                  className={`p-4 border rounded-lg cursor-pointer transition-colors ${
                                    selectedAnswer?.id === answer.id ? "border-primary bg-primary/5" : "hover:bg-muted"
                                  }`}
                                  onClick={() => setSelectedAnswer(answer)}
                                >
                                  <div className="flex items-start justify-between mb-2">
                                    <div className="flex items-center gap-2">
                                      {getSubjectIcon(answer.subject)}
                                      <Badge variant="outline">{answer.subject}</Badge>
                                      {hasAIGrade && (
                                        <Badge variant="secondary" className="bg-purple-100 text-purple-800">
                                          <Brain className="w-3 h-3 mr-1" />
                                          AI Graded
                                        </Badge>
                                      )}
                                    </div>
                                    <div className="text-right">
                                      <p className="text-sm font-medium">
                                        {hasAIGrade
                                          ? `${hasAIGrade.awardedMarks}/${answer.totalMarks}`
                                          : `${answer.marks}/${answer.totalMarks}`}
                                      </p>
                                      <p className="text-xs text-muted-foreground">
                                        {hasAIGrade?.similarityScore || answer.similarityScore}% similarity
                                      </p>
                                    </div>
                                  </div>
                                  <p className="text-sm font-medium mb-2 line-clamp-2">{answer.question}</p>
                                  <p className="text-sm text-muted-foreground line-clamp-3">{answer.studentAnswer}</p>

                                  <div className="flex items-center justify-between mt-3 pt-3 border-t">
                                    <div className="flex gap-2">
                                      <Button
                                        size="sm"
                                        variant="outline"
                                        onClick={(e) => {
                                          e.stopPropagation()
                                          setSelectedAnswer(answer)
                                          setShowDetailedGrading(true)
                                        }}
                                      >
                                        <Eye className="w-3 h-3 mr-1" />
                                        Review
                                      </Button>
                                      {!hasAIGrade && (
                                        <Button
                                          size="sm"
                                          variant="outline"
                                          onClick={(e) => {
                                            e.stopPropagation()
                                            gradeWithAI(answer)
                                          }}
                                          disabled={isGrading}
                                        >
                                          <Brain className="w-3 h-3 mr-1" />
                                          Grade
                                        </Button>
                                      )}
                                    </div>
                                    <div className="text-xs text-muted-foreground">
                                      Submitted: {new Date(answer.submittedAt).toLocaleDateString()}
                                    </div>
                                  </div>
                                </div>
                              )
                            })
                          ) : (
                            <div className="text-center py-8 text-muted-foreground">
                              <FileText className="w-12 h-12 mx-auto mb-4 opacity-50" />
                              <h3 className="text-lg font-medium mb-2">No Submissions Yet</h3>
                              <p>This student hasn't submitted any answers for this exam.</p>
                            </div>
                          )}
                        </div>
                      </ScrollArea>
                    ) : (
                      <div className="h-96 flex items-center justify-center text-muted-foreground">
                        <div className="text-center">
                          <Users className="w-12 h-12 mx-auto mb-4 opacity-50" />
                          <h3 className="text-lg font-medium mb-2">Select a Student</h3>
                          <p>Choose a student from the list to view their submissions and start grading.</p>
                        </div>
                      </div>
                    )}
                  </CardContent>
                </Card>
              </div>
            )}
          </TabsContent>

          <TabsContent value="ai-grading" className="space-y-6">
            {selectedAnswer ? (
              <div className="space-y-6">
                <AIGradingInterface answer={selectedAnswer} onGradeUpdate={handleGradeUpdate} />

                <SubjectSpecificGrading
                  subject={selectedAnswer.subject}
                  answer={selectedAnswer.studentAnswer}
                  totalMarks={selectedAnswer.totalMarks}
                  awardedMarks={selectedAnswer.aiGrade?.awardedMarks || selectedAnswer.marks}
                />
              </div>
            ) : (
              <Card>
                <CardContent className="py-12">
                  <div className="text-center text-muted-foreground">
                    <Brain className="w-12 h-12 mx-auto mb-4 opacity-50" />
                    <h3 className="text-lg font-medium mb-2">AI Grading System</h3>
                    <p className="mb-4">
                      Select a student submission from the Submissions tab to begin AI-powered grading
                    </p>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 max-w-2xl mx-auto">
                      <div className="p-4 bg-muted rounded-lg">
                        <BookOpen className="w-8 h-8 mx-auto mb-2 text-blue-600" />
                        <h4 className="font-medium mb-1">History</h4>
                        <p className="text-xs">Analyze historical events, causes, and consequences</p>
                      </div>
                      <div className="p-4 bg-muted rounded-lg">
                        <MapPin className="w-8 h-8 mx-auto mb-2 text-green-600" />
                        <h4 className="font-medium mb-1">Geography</h4>
                        <p className="text-xs">Evaluate geographical concepts and environmental awareness</p>
                      </div>
                      <div className="p-4 bg-muted rounded-lg">
                        <Vote className="w-8 h-8 mx-auto mb-2 text-purple-600" />
                        <h4 className="font-medium mb-1">Political Science</h4>
                        <p className="text-xs">Assess political understanding and critical analysis</p>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="analytics" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="w-5 h-5" />
                    Performance Trends
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="h-64 flex items-center justify-center text-muted-foreground">
                    Performance chart would be rendered here
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Question Analysis</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Question 1 - Differential Equations</span>
                      <div className="flex items-center gap-2">
                        <Progress value={85} className="w-20 h-2" />
                        <span className="text-sm text-muted-foreground">85%</span>
                      </div>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Question 2 - Integration</span>
                      <div className="flex items-center gap-2">
                        <Progress value={72} className="w-20 h-2" />
                        <span className="text-sm text-muted-foreground">72%</span>
                      </div>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Question 3 - Series Convergence</span>
                      <div className="flex items-center gap-2">
                        <Progress value={58} className="w-20 h-2" />
                        <span className="text-sm text-muted-foreground">58%</span>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="camera-monitoring" className="space-y-6">
            <LiveMonitoringInterface />
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
