export interface ExamData {
  subject: string
  duration: number
  totalMarks: number
  questions: Question[]
}

export interface Question {
  id: number
  text: string
  marks: number
  timeLimit?: number
}

export interface Answer {
  canvas?: string
  text?: string
  lastSaved?: Date
}

export interface SubmissionResponse {
  success: boolean
  data: {
    id: string
    status: string
    gradingStatus: string
  }
  message: string
  estimatedGradingTime?: string
}

export interface GradingStatus {
  submissionId: string
  status: string
  progress: number
  stages: {
    handwritingRecognition: "pending" | "processing" | "completed"
    aiGrading: "pending" | "processing" | "completed"
    humanReview: "pending" | "processing" | "completed"
  }
  estimatedCompletion?: string
  results?: {
    totalScore: number
    maxScore: number
    grade: string
    questionScores: Record<
      number,
      {
        score: number
        maxScore: number
        feedback: string
      }
    >
  }
}

export class ExamAPI {
  private static baseUrl = "/api"

  static async getExamData(): Promise<ExamData> {
    const response = await fetch(`${this.baseUrl}/exam`)
    if (!response.ok) {
      throw new Error("Failed to fetch exam data")
    }
    return response.json()
  }

  static async saveAnswer(questionId: number, answer: Answer, studentId?: string, examId?: string): Promise<any> {
    const response = await fetch(`${this.baseUrl}/exam/save-answer`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        questionId,
        answer,
        studentId,
        examId,
      }),
    })

    if (!response.ok) {
      throw new Error("Failed to save answer")
    }

    return response.json()
  }

  static async submitExam(
    answers: Record<number, Answer>,
    studentId?: string,
    examId?: string,
  ): Promise<SubmissionResponse> {
    const response = await fetch(`${this.baseUrl}/exam/submit`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        answers,
        studentId,
        examId,
        submittedAt: new Date().toISOString(),
      }),
    })

    if (!response.ok) {
      throw new Error("Failed to submit exam")
    }

    return response.json()
  }

  static async getGradingStatus(submissionId: string): Promise<GradingStatus> {
    const response = await fetch(`${this.baseUrl}/grading/status?submissionId=${submissionId}`)
    if (!response.ok) {
      throw new Error("Failed to fetch grading status")
    }
    return response.json()
  }

  static async verifyAuthentication(type: "face" | "id", data?: any): Promise<any> {
    const response = await fetch(`${this.baseUrl}/auth/verify`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({ type, data }),
    })

    if (!response.ok) {
      throw new Error("Authentication verification failed")
    }

    return response.json()
  }
}
