export interface SubjectCriteria {
  name: string
  description: string
  maxPoints: number
  keywords: string[]
}

export interface SubjectEvaluation {
  subject: "History" | "Geography" | "Political Science"
  criteria: SubjectCriteria[]
  gradingRubric: GradingRubric
  sampleQuestions: SampleQuestion[]
}

export interface GradingRubric {
  excellent: { range: [number, number]; description: string }
  good: { range: [number, number]; description: string }
  satisfactory: { range: [number, number]; description: string }
  needsImprovement: { range: [number, number]; description: string }
}

export interface SampleQuestion {
  id: string
  question: string
  type: "analytical" | "descriptive" | "case-study" | "comparative"
  marks: number
  expectedLength: string
  keyPoints: string[]
}

export const historyEvaluation: SubjectEvaluation = {
  subject: "History",
  criteria: [
    {
      name: "Historical Knowledge",
      description: "Demonstrates accurate knowledge of historical facts, dates, and events",
      maxPoints: 25,
      keywords: ["chronology", "dates", "events", "facts", "historical accuracy"],
    },
    {
      name: "Cause and Effect Analysis",
      description: "Shows understanding of historical causation and consequences",
      maxPoints: 25,
      keywords: ["causes", "consequences", "impact", "resulted in", "led to", "because"],
    },
    {
      name: "Historical Interpretation",
      description: "Provides critical analysis and interpretation of historical events",
      maxPoints: 20,
      keywords: ["analysis", "interpretation", "significance", "evaluate", "assess"],
    },
    {
      name: "Use of Evidence",
      description: "Supports arguments with relevant historical evidence and examples",
      maxPoints: 20,
      keywords: ["evidence", "examples", "sources", "documents", "primary", "secondary"],
    },
    {
      name: "Historical Context",
      description: "Places events within appropriate historical context",
      maxPoints: 10,
      keywords: ["context", "period", "era", "background", "circumstances"],
    },
  ],
  gradingRubric: {
    excellent: {
      range: [90, 100],
      description: "Comprehensive understanding with excellent analysis and use of evidence",
    },
    good: { range: [70, 89], description: "Good understanding with adequate analysis and some evidence" },
    satisfactory: { range: [50, 69], description: "Basic understanding with limited analysis" },
    needsImprovement: { range: [0, 49], description: "Insufficient understanding and analysis" },
  },
  sampleQuestions: [
    {
      id: "hist_1",
      question: "Analyze the causes and consequences of the Revolt of 1857. How did it impact British colonial policy?",
      type: "analytical",
      marks: 12,
      expectedLength: "300-400 words",
      keyPoints: [
        "Multiple causes",
        "Economic exploitation",
        "Social reforms",
        "Military grievances",
        "Policy changes",
        "Crown rule",
      ],
    },
  ],
}

export const geographyEvaluation: SubjectEvaluation = {
  subject: "Geography",
  criteria: [
    {
      name: "Geographical Concepts",
      description: "Understanding of fundamental geographical concepts and processes",
      maxPoints: 25,
      keywords: ["concepts", "processes", "patterns", "distribution", "spatial"],
    },
    {
      name: "Environmental Awareness",
      description: "Knowledge of environmental issues and sustainability",
      maxPoints: 25,
      keywords: ["environment", "sustainability", "conservation", "pollution", "climate"],
    },
    {
      name: "Case Studies",
      description: "Effective use of relevant case studies and examples",
      maxPoints: 20,
      keywords: ["case study", "example", "region", "area", "location"],
    },
    {
      name: "Geographical Skills",
      description: "Application of geographical skills and techniques",
      maxPoints: 20,
      keywords: ["maps", "data", "statistics", "graphs", "analysis"],
    },
    {
      name: "Human-Environment Interaction",
      description: "Understanding of relationships between humans and environment",
      maxPoints: 10,
      keywords: ["interaction", "relationship", "impact", "adaptation", "modification"],
    },
  ],
  gradingRubric: {
    excellent: { range: [90, 100], description: "Excellent geographical understanding with strong use of examples" },
    good: { range: [70, 89], description: "Good geographical knowledge with adequate examples" },
    satisfactory: { range: [50, 69], description: "Basic geographical understanding" },
    needsImprovement: { range: [0, 49], description: "Limited geographical knowledge" },
  },
  sampleQuestions: [
    {
      id: "geo_1",
      question: "Explain sustainable development with reference to India's environmental challenges",
      type: "descriptive",
      marks: 10,
      expectedLength: "250-300 words",
      keyPoints: [
        "Sustainable development definition",
        "Environmental challenges",
        "Air pollution",
        "Water scarcity",
        "Solutions",
      ],
    },
  ],
}

export const politicalScienceEvaluation: SubjectEvaluation = {
  subject: "Political Science",
  criteria: [
    {
      name: "Political Concepts",
      description: "Understanding of fundamental political concepts and theories",
      maxPoints: 25,
      keywords: ["democracy", "constitution", "rights", "governance", "political theory"],
    },
    {
      name: "Indian Political System",
      description: "Knowledge of Indian political institutions and processes",
      maxPoints: 25,
      keywords: ["parliament", "judiciary", "federalism", "elections", "political parties"],
    },
    {
      name: "Critical Analysis",
      description: "Critical evaluation of political issues and developments",
      maxPoints: 20,
      keywords: ["analysis", "evaluate", "critique", "assess", "examine"],
    },
    {
      name: "Contemporary Issues",
      description: "Awareness of current political developments and challenges",
      maxPoints: 20,
      keywords: ["contemporary", "current", "recent", "modern", "today"],
    },
    {
      name: "Political Terminology",
      description: "Appropriate use of political science terminology",
      maxPoints: 10,
      keywords: ["terminology", "concepts", "definitions", "political vocabulary"],
    },
  ],
  gradingRubric: {
    excellent: { range: [90, 100], description: "Excellent political understanding with strong critical analysis" },
    good: { range: [70, 89], description: "Good political knowledge with adequate analysis" },
    satisfactory: { range: [50, 69], description: "Basic political understanding" },
    needsImprovement: { range: [0, 49], description: "Limited political knowledge" },
  },
  sampleQuestions: [
    {
      id: "pol_1",
      question: "Evaluate the role of regional parties in Indian democracy and coalition politics",
      type: "analytical",
      marks: 12,
      expectedLength: "350-400 words",
      keyPoints: [
        "Regional parties role",
        "Coalition politics",
        "Federal balance",
        "Local representation",
        "Policy influence",
      ],
    },
  ],
}

export const subjectEvaluations = {
  History: historyEvaluation,
  Geography: geographyEvaluation,
  "Political Science": politicalScienceEvaluation,
}

export function getSubjectEvaluation(subject: "History" | "Geography" | "Political Science"): SubjectEvaluation {
  return subjectEvaluations[subject]
}

export function calculateCriteriaScore(answer: string, criteria: SubjectCriteria): number {
  const answerLower = answer.toLowerCase()
  let score = 0
  let keywordMatches = 0

  // Check for keyword matches
  criteria.keywords.forEach((keyword) => {
    if (answerLower.includes(keyword.toLowerCase())) {
      keywordMatches++
    }
  })

  // Calculate score based on keyword density and answer length
  const keywordDensity = keywordMatches / criteria.keywords.length
  const answerLength = answer.split(" ").length
  const expectedLength = answerLength > 50 ? 1 : answerLength / 50

  score = (keywordDensity * 0.7 + expectedLength * 0.3) * criteria.maxPoints

  return Math.min(score, criteria.maxPoints)
}

export function getGradeFromPercentage(percentage: number, rubric: GradingRubric): string {
  if (percentage >= rubric.excellent.range[0]) return "Excellent"
  if (percentage >= rubric.good.range[0]) return "Good"
  if (percentage >= rubric.satisfactory.range[0]) return "Satisfactory"
  return "Needs Improvement"
}

export function generateSubjectSpecificFeedback(
  subject: "History" | "Geography" | "Political Science",
  answer: string,
  totalMarks: number,
  awardedMarks: number,
): {
  strengths: string[]
  improvements: string[]
  subjectSpecificTips: string[]
} {
  const evaluation = getSubjectEvaluation(subject)
  const percentage = (awardedMarks / totalMarks) * 100

  const feedback = {
    strengths: [] as string[],
    improvements: [] as string[],
    subjectSpecificTips: [] as string[],
  }

  // Subject-specific feedback based on performance
  switch (subject) {
    case "History":
      if (percentage >= 80) {
        feedback.strengths.push("Strong historical knowledge demonstrated")
        feedback.strengths.push("Good understanding of cause-effect relationships")
      }
      if (percentage < 70) {
        feedback.improvements.push("Include more specific historical dates and events")
        feedback.improvements.push("Develop stronger cause-effect analysis")
      }
      feedback.subjectSpecificTips = [
        "Use primary source references where possible",
        "Connect events to broader historical themes",
        "Practice chronological thinking",
      ]
      break

    case "Geography":
      if (percentage >= 80) {
        feedback.strengths.push("Good understanding of geographical concepts")
        feedback.strengths.push("Effective use of examples")
      }
      if (percentage < 70) {
        feedback.improvements.push("Include more case studies and examples")
        feedback.improvements.push("Strengthen environmental awareness")
      }
      feedback.subjectSpecificTips = [
        "Use maps and diagrams to support answers",
        "Connect local examples to global patterns",
        "Focus on human-environment interactions",
      ]
      break

    case "Political Science":
      if (percentage >= 80) {
        feedback.strengths.push("Strong grasp of political concepts")
        feedback.strengths.push("Good critical analysis")
      }
      if (percentage < 70) {
        feedback.improvements.push("Develop stronger critical analysis")
        feedback.improvements.push("Use more political science terminology")
      }
      feedback.subjectSpecificTips = [
        "Connect theory to contemporary examples",
        "Analyze multiple perspectives on issues",
        "Use appropriate political terminology",
      ]
      break
  }

  return feedback
}
