import os
import argparse
import torch
from diffusers import StableDiffusion3Pipeline, BitsAndBytesConfig, SD3Transformer2DModel

def get_styles():
    return ['Pixel art', 'Oil painting', 'Watercolor sketch', 'Charcoal drawing', 'Art Nouveau', 'Art Deco', 'Impressionism', 'Expressionism', 'Surrealism', 'Cubism', 'Pop Art', 'Steampunk', 'Cyberpunk', 'Gothic art', 'Renaissance painting', 'Baroque style', 'Rococo', 'Minimalism', 'Abstract', 'Geometric', 'Photorealism', 'Ukiyo-e woodblock', 'Anime style', 'Manga screentone', 'Chibi', 'Cartoon', 'Comic book ink', 'Graphic novel panel', 'Stained glass', 'Mosaic', 'Collage', 'Technical blueprint', 'Architectural sketch', 'Botanical illustration', 'Medical illustration', 'Infographic', 'Vintage advertisement', 'Psychedelic', 'Grunge texture', 'Film noir', 'Sepia tone', 'Black and white', 'Double exposure', 'Long exposure', 'Pinhole photography', 'Macro photography', 'Infrared photography', 'Glitch art', '3D render', 'Voxel art', 'Vector illustration', 'Low poly', 'High poly', 'Matte painting', 'Concept art', 'Storyboard sketch', 'Claymation', 'Stop motion', 'Linocut print', 'Woodcut print', 'Etching', 'Lithograph', 'Screenprint', 'Cyanotype', 'Daguerreotype', 'Calligraphy', 'Graffiti tag', 'Stencil art', 'Airbrush', 'Body paint', 'Tattoo flash', 'Henna design', 'Sand art', 'Ice sculpture', 'Topiary', 'Origami', 'Kirigami', 'Paper quilling', 'Embroidery', 'Cross-stitch pattern', 'Tapestry', 'Batik', 'Tie-dye', 'Shibori dye', 'Celtic knotwork', 'Viking rune stone', 'Illuminated manuscript', 'Egyptian hieroglyphs', 'Greek black-figure pottery', 'Roman fresco', 'Byzantine icon', 'Russian folk art', 'Islamic geometric patterns', 'Persian miniature', 'Chinese ink wash', 'Tibetan thangka', 'Aboriginal dot painting', 'Haida formline art', 'Aztec codex', 'Mayan glyphs', 'Folk art', 'Naive art', 'Pointillism', 'Fauvism', 'Dadaism', 'Suprematism', 'Constructivism', 'Bauhaus design', 'Abstract Expressionism', 'Color Field painting', 'Op Art', 'Kinetic art', 'Land art', 'Street art', 'Lowbrow art', 'Fantasy map', 'Character design sheet', 'Creature concept', 'Mecha design', 'Spaceship concept', 'Alien landscape', 'Eldritch horror', 'High fantasy', 'Grimdark', 'Dieselpunk', 'Atompunk', 'Biopunk', 'Solarpunk', 'Afrofuturism', 'Vaporwave aesthetic', 'Synthwave', 'Lo-fi aesthetic', 'Cottagecore', 'Goblincore', 'Dark academia', 'Light academia', 'Fairy kei', 'Whimsical', 'Film grain', 'Anamorphic lens flare', 'Fisheye lens', 'Tilt-shift', 'Bokeh', 'High-speed photography', 'Schlieren photography', 'Kirlian photography', 'X-ray', 'Thermal imaging', 'Microscope slide', 'Satellite imagery', 'Sonogram', 'Schematic diagram', 'Exploded-view diagram', 'Cutaway illustration', 'Patent drawing', 'Chalk drawing', 'Pastel drawing', 'Crayon drawing', 'Marker art', 'Ballpoint pen sketch', 'Fountain pen ink', 'Ink wash', 'Stippling', 'Cross-hatching', 'Sgraffito', 'Impasto', 'Fresco', 'Gouache painting', 'Encaustic painting', 'Tempera on wood', 'Glass painting', 'Silhouette', 'Cameo carving', 'Scrimshaw', 'Pyrography (wood burning)', 'Leather tooling', 'Metal engraving', 'Jewelry design', 'Fashion illustration', 'Costume design sketch', 'Set design model', 'Architectural model', 'Diorama', 'Terrarium', 'Bonsai', 'Ikebana arrangement', 'Food plating', 'Latte art', 'Cake decorating', 'Fruit carving', 'LEGO bricks', 'Sandcastle', 'Light painting', 'Shadow puppet', 'Hologram', 'Anaglyph 3D', 'Lenticular print', 'Risograph print', 'Mimeograph', 'Dot matrix print', 'ASCII art', 'ANSI art', 'Glitch text', 'VHS tape effect', 'Super 8 film', 'Silent film title card', 'German Expressionist film', 'Soviet propaganda poster', 'WPA poster style', 'Swiss Style typography', 'Brutalism', 'Deconstructivism', 'Googie architecture', 'Mid-century modern', 'Victorian style', 'Colonial art', 'Americana', 'Western art', 'Tribal patterns', 'Cave painting', 'Petroglyph', 'Geoglyph', 'Nazca lines', 'Crop circle', 'Labyrinth design', 'Mandala', 'Yantra diagram', 'Zentangle', 'Doodle', 'Caricature', 'Political cartoon', 'Sunday funnies', 'Webcomic style', 'Visual novel sprite', 'Dating sim interface', 'JRPG pixel art', 'Fighting game character', 'Isometric sprite art', 'Platformer game background', 'Racing game screenshot', 'Flight simulator view', 'MMORPG screenshot', 'RTS game perspective', 'Text adventure screen', 'MUD interface', 'Desktop screenshot', 'UI/UX wireframe', 'Website mockup', 'App icon', 'Emoji set', 'Sticker design', 'Enamel pin design', 'Patch design', 'Logo design', 'Brand mascot', 'Album cover art', 'Book cover design', 'Movie poster', 'Gig poster', 'Festival flyer', 'Zine', 'Scrapbook page', 'Found photo', 'Polaroid photo', 'Photobooth strip', 'Disposable camera photo', 'CCTV footage', 'Body cam footage', 'Drone footage', 'Satellite map', 'Topographic map', 'Nautical chart', 'Celestial map', 'Weather map', 'Heat map', 'Flowchart', 'Mind map', 'Venn diagram', 'Family tree', 'Organizational chart', 'Sheet music', 'Tablature', 'Player piano roll', 'Music box mechanism', 'Circuit board', 'Engineering schematic', 'Particle accelerator data', 'DNA sequencing gel', 'Petri dish culture', 'Crystal structure', 'Fractal art', 'Generative art', 'Neural network visualization', 'Data visualization', 'Soundwave form', 'Spectrogram', 'Rorschach inkblot', 'Hypnotic spiral', 'Impossible object', 'Optical illusion', "Trompe-l'œil", 'Anamorphosis', 'Hidden image puzzle', '"Where\'s Waldo?" style', 'Coloring book page', 'Paint-by-numbers', 'Connect-the-dots', 'Paper doll', 'Board game design', 'Playing card design', 'Tarot card', 'Postage stamp', 'Banknote design', 'Passport page', 'Wax seal', 'Heraldic coat of arms', 'Company flag design', 'Sports team logo', 'University crest', 'Military insignia', 'Medal design', 'Coin (obverse/reverse)', 'Manhole cover design', 'Tile pattern', 'Wallpaper pattern', 'Textile pattern', 'Upholstery fabric', 'Rug design', 'Quilt pattern', 'Knitting pattern', 'Crochet pattern', 'Macrame', 'Weaving', 'Lacework', 'Beading pattern', 'Decoupage', 'Marquetry', 'Inlay', 'Filigree', 'Chasing and repoussé', 'Glassblowing', 'Lampworking (glass beads)', 'Kintsugi', 'Raku ware pottery', 'Delftware', 'Wedgwood jasperware', 'Majolica pottery', 'Terracotta army style', 'Netsuke carving', 'Scrimshaw on bone', 'Fabergé egg', 'Automaton', 'Diorama in a bottle', 'Model railroad', 'War game miniature', 'Dollhouse', 'Marionette', 'Bunraku puppet', 'Shadow play', 'Wayang kulit', 'Kabuki makeup', 'Noh mask', 'Venetian carnival mask', 'Mardi Gras float', 'Day of the Dead', 'Sugar skull', 'Alebrije', 'Huichol bead art', 'Molas textile art', 'Shipibo-Conibo pattern', 'Kufic script', 'Thuluth calligraphy', 'Cuneiform tablet', 'Ogham stone', 'Linear B script', 'Voynich manuscript style', 'Codex Seraphinianus style', 'Asemic writing', 'Concrete poetry', 'Blackout poetry', 'Found object sculpture', 'Assemblage art', 'Readymade', 'Installation art', 'Performance art documentation', 'Earthwork', 'Environmental art', 'BioArt', 'Bubble-gram', 'Light installation', 'Neon sign', 'Laser show', 'Projection mapping', 'VR painting', 'Augmented reality art', 'Photogrammetry scan', 'LIDAR scan', '3D point cloud', 'Wireframe model', 'Procedural generation', 'Cellular automata', 'Reaction-diffusion pattern', 'Fluid dynamics simulation', 'Cloth simulation', 'Particle system', 'Demolition simulation', 'Photobashing', 'Digital collage', 'Speed painting', 'Greyscale value painting', 'Tonalism', 'Luminism', 'Pre-Raphaelite Brotherhood', 'Arts and Crafts', 'Wiener Werkstätte', 'The Blue Rider', 'Die Brücke', 'Pittura Metafisica', 'Futurism', 'Vorticism', 'Orphism', 'Synchromism', 'De Stijl', 'Magic Realism', 'Social Realism', 'Ashcan School', 'Regionalism (American)', 'Precisionism', 'Cobra movement', 'Art Brut', 'Tachisme', 'Gutai group', 'Fluxus', 'Nouveau Réalisme', 'Arte Povera', 'Hard-edge painting', 'Shaped canvas', 'Post-painterly Abstraction', 'Feminist art', 'Video art', 'Neo-Expressionism', 'Transavantgarde', 'Neo-Geo', 'Stuckism', 'Toyism', 'Superflat', 'Massurrealism', 'Post-internet art', 'Vaporwave', 'Seapunk', 'Health Goth', 'Normcore', 'Kidcore', 'Dreamcore', 'Weirdcore', 'Traumacore', 'Gloomy art', 'Macabre', 'Grotesque', 'Ethereal', 'Sublime', 'Pastoral', 'Idyllic', 'Romanticism', 'Neoclassicism', 'Academic art', 'History painting', 'Genre painting', 'Still life', 'Vanitas', 'Portraiture', 'Landscape painting', 'Seascape', 'Cityscape', 'Cloudscape', 'Nocturne painting', 'Tenebrism', 'Chiaroscuro', 'Sfumato', 'Grisaille', 'Verdaille', 'Underpainting', 'Alla prima', 'Dry brush', 'Glazing', 'Scumbling', 'Monochromatic', 'Polychromatic', 'Achromatic', 'Complementary colors', 'Analogous colors', 'Triadic color scheme', 'Tonal harmony', 'Isometric perspective', 'One-point perspective', 'Two-point perspective', 'Three-point perspective', 'Forced perspective', 'Aerial perspective', 'Reverse perspective', 'Fisheye projection', 'Stereographic projection', 'Mercator projection', 'Dymaxion map', 'Holographic foil', 'Iridescent sheen', 'Pearlescent finish', 'High gloss', 'Matte finish', 'Textured paper', 'Vellum', 'Papyrus']

def main():
    parser = argparse.ArgumentParser(description="Generate diffusion images from a list of styles.")
    parser.add_argument('--model_id', type=str, default='stabilityai/stable-diffusion-3-medium-nf4', help='The model ID for Stable Diffusion.')
    parser.add_argument('--output_dir', type=str, default='./new_diffusion_images', help='Directory to save the generated images.')
    parser.add_argument('--images_per_style', type=int, default=8, help='Number of images to generate for each style.')
    parser.add_argument('--batch_size', type=int, default=1, help='Number of images to generate per prompt.')
    parser.add_argument('--seed', type=int, default=0, help='Random seed for generation.')
    args = parser.parse_args()

    device = "cuda" if torch.cuda.is_available() else "cpu"
    
    nf4_config = BitsAndBytesConfig(
       load_in_4bit=True,
       bnb_4bit_quant_type="nf4",
       bnb_4bit_compute_dtype=torch.bfloat16
    )
    model_nf4 = SD3Transformer2DModel.from_pretrained(
        args.model_id,
        subfolder="transformer",
        quantization_config=nf4_config,
        torch_dtype=torch.bfloat16
    )

    pipeline = StableDiffusion3Pipeline.from_pretrained(
        "stabilityai/stable-diffusion-3-medium", 
        transformer=model_nf4,
        torch_dtype=torch.bfloat16
    )
    
    pipeline.safety_checker = None
    pipeline = pipeline.to(device)

    generator = torch.Generator(device="cpu").manual_seed(args.seed)
    os.makedirs(args.output_dir, exist_ok=True)
    
    styles = get_styles()

    with torch.amp.autocast('cuda', torch.bfloat16):
        with torch.no_grad():
            for prompt in styles:
                domain_folder = prompt.replace('.', '').replace(' ', '_').lower()
                style_path = os.path.join(args.output_dir, domain_folder)
                
                if not os.path.exists(style_path):
                    os.makedirs(style_path)
                    print(f"Generating images for style: {prompt}")
                    for i in range(args.images_per_style):
                        image = pipeline(
                            prompt=f"{prompt}",
                            num_inference_steps=40,
                            num_images_per_prompt=args.batch_size,
                            generator=generator
                        ).images[0]
                        image.save(os.path.join(style_path, f'{i}.jpg'))
                else:
                    print(f"Skipping style {prompt}, directory already exists.")

if __name__ == '__main__':
    main()
