
# %% Imports

import pickle
from time import time
import matplotlib.pyplot as plt

import tensorflow as tf
from tensorflow_probability import distributions as tfd

from pavi.dual.models import (
    UIVFamily
)
from pavi.utils.callbacks import (TimerCallback,)
from generative_hbms.HV import (
    seed,
    N1_full,
    full_hbm,
    link_functions,
    val_data,
    na_val_idx,
)

# %% UIVFamily kwargs

nf_chain_kwargs = dict(
    nf_type_kwargs_per_bijector=[
        (
            "affine",
            dict(
                scale_type="diag",
                rff_kwargs=dict(
                    units_per_layers=[64, 64]
                )
            )
        )
    ],
    with_permute=False,
    with_batch_norm=False
)

uivi_kwargs = dict(
    generative_hbm=full_hbm,
    link_functions=link_functions,
    observed_rv="c",
    conditional_nf_chain_kwargs=nf_chain_kwargs,
    embedding_RV_size=6
)


# %% We build our architecture
ui_family = UIVFamily(
    **uivi_kwargs
)

# %%

observed_values = {
    "c": tf.expand_dims(val_data["c"][na_val_idx], axis=0)
}

# %% We fit the training data
time_1 = time()

# %% We select the loss used for training

ui_family.compile(
    n_theta_draws=8,
    t_mcmc_burn_in=5,
    n_mcmc_samples=5,
    optimizer=tf.keras.optimizers.Adam(
        learning_rate=tf.keras.optimizers.schedules.ExponentialDecay(
            initial_learning_rate=1e-2,
            decay_steps=300,
            decay_rate=0.9
        )
    )
)

# %% We fit the training data
hist_2 = ui_family.fit(
    observed_values,
    batch_size=1,
    epochs=3_000,
    verbose=0,
    callbacks=[
        TimerCallback()
    ]
)
time_2 = time()

# %% We compute the ELBO
sample_size = 256
repeated_c = tf.repeat(
    val_data["c"][na_val_idx:na_val_idx + 1],
    repeats=(sample_size,),
    axis=0
)
z, epsilon = ui_family.sample((sample_size,), return_epsilon=True)
q = ui_family.q_z(z, epsilon)
p = ui_family.generative_hbm.log_prob(
    **z,
    c=repeated_c
)
ELBO = tf.reduce_mean(
    p - q
).numpy()

print(f"""
    Idx:  {na_val_idx}
    Time: {time_2 - time_1}
    Loss: {- ELBO}
""")

# %%
base_name = (
    "../data/"
    f"HV_UIVI_idx{na_val_idx}_seed{seed}_"
)

pickle.dump(
    hist_2.history,
    open(
        base_name + "history.p",
        "wb"
    )
)
pickle.dump(
    - ELBO,
    open(
        base_name + "loss.p",
        "wb"
    )
)