
# %% Imports

import pickle
import os
from time import time
import matplotlib.pyplot as plt

import tensorflow as tf
import numpy as np
from tensorflow_probability import distributions as tfd

from pavi.dual.models import (
    PAVFFamily
)
from pavi.utils.callbacks import (TimerCallback, ELBOCallback)
from generative_hbms.GM import (
    seed,
    G, N,
    pavi_kwargs,
    val_data,
    na_val_idx,
)

# %% FRMFamily kwargs

d = 16

rff_kwargs = dict(
    units_per_layers=[32, 32]
)

conditional_nf_chain_kwargs = dict(
    nf_type_kwargs_per_bijector=[
        (
            "MAF",
            dict(
                hidden_units=[32],
                kernel_initializer=tf.keras.initializers.RandomNormal(
                    mean=0.0,
                    stddev=1e-4
                ),
                bias_initializer="zeros"
            )
        ),
        (
            "affine",
            dict(
                scale_type="diag",
                rff_kwargs=rff_kwargs
            )
        )
    ],
    with_permute=False,
    with_batch_norm=False
)

flow_posterior_scheme_kwargs = (
    "flow",
    dict(
        conditional_nf_chain_kwargs=conditional_nf_chain_kwargs
    )
)

posterior_schemes_kwargs = {
    "mu": flow_posterior_scheme_kwargs,
    "mu_g": flow_posterior_scheme_kwargs,
    "probs": flow_posterior_scheme_kwargs,
    "x": (
        "observed",
        dict()
    )
}

pavi_family_kwargs = dict(
    posterior_schemes_kwargs=posterior_schemes_kwargs,
    encoding_sizes={
        ('P',): d,
        ('P', 'G'): d,
    },
    **pavi_kwargs
)


# %% We build our architecture
frm_family = PAVFFamily(
    **pavi_family_kwargs
)

# %%

observed_values = {
    "x": tf.expand_dims(val_data["x"][na_val_idx], axis=0)
}

# %% We fit the training data
time_1 = time()

# %% We select the loss used for training

frm_family.compile(
    train_method="reverse_KL",
    n_theta_draws=8,
    optimizer="adam"
)

# %% We fit the training data
hist_2 = frm_family.fit(
    observed_values,
    batch_size=1,
    epochs=5_000,
    verbose=2,
    callbacks=[
        TimerCallback(),
        ELBOCallback(
            elbo_epochs=50,
            p_model=pavi_kwargs["full_hbm"],
            observed_values=observed_values
        )
    ]
)
time_2 = time()

# %% We compute the ELBO
sample_size = 256
q_sample = frm_family.sample(
    sample_shape=(sample_size,),
    observed_values=observed_values,
    return_observed_values=True
)
q = frm_family.log_prob(
    q_sample
)
p = pavi_kwargs["full_hbm"].log_prob(
    q_sample
)
ELBO = tf.reduce_mean(p - q).numpy()

print(f"""
    Idx:  {na_val_idx}
    Time: {time_2 - time_1}
    Loss: {- ELBO}
""")

# %%

fig, axs = plt.subplots(
    nrows=1,
    ncols=2,
    figsize=(20, 10)
)
for g in range(G):
    axs[0].scatter(
        val_data["x"][na_val_idx, g, :, 0],
        val_data["x"][na_val_idx, g, :, 1],
        color=f"C{g}",
        alpha=0.5
    )

axs[0].axis("equal")
axs[0].set_ylabel(
    f"{na_val_idx}",
    fontsize=30,
    rotation=0
)
plt.draw()
x_lim = axs[0].get_xlim()
y_lim = axs[0].get_ylim()

axs[1].scatter(
    q_sample["mu"][:, 0, :, 0],
    q_sample["mu"][:, 0, :, 1],
    color="black",
    s=20,
    alpha=0.05
)
for g in range(G):
    axs[1].scatter(
        q_sample["mu_g"][:, 0, g, :, 0],
        q_sample["mu_g"][:, 0, g, :, 1],
        color=f"C{g}",
        s=20,
        alpha=0.05
    )
axs[1].set_xlim(x_lim)
axs[1].set_ylim(y_lim)

axs[0].set_title(
    "Data",
    fontsize=30
)
axs[1].set_title(
    "Posterior samples",
    fontsize=30
)
plt.show()

# %%
base_name = (
    "../data/"
    f"GM_PAVI-F_idx{na_val_idx}_seed{seed}_"
)

pickle.dump(
    hist_2.history,
    open(
        base_name + "history.p",
        "wb"
    )
)
pickle.dump(
    - ELBO,
    open(
        base_name + "loss.p",
        "wb"
    )
)