import numpy as np
import pandas as pd
import scipy.linalg as la
import os

# ─── PARAMETERS ───────────────────────────────────────────────────────────────
base_dir    = "/Users/home/Documents/naz/research_codes/uncert_prop/realworld_exp/hai_down1"
output_dir  = os.path.join(base_dir, "centralized_lti_system")
os.makedirs(output_dir, exist_ok=True)
U_file      = os.path.join(base_dir, "U_complete.csv")
Y_file      = os.path.join(base_dir, "Y_complete.csv")
i           = 10   # Hankel depth, tune as needed
# ───────────────────────────────────────────────────────────────────────────────

# 1. Load U and Y
U_df = pd.read_csv(U_file)
Y_df = pd.read_csv(Y_file)

# 1a. Normalize U and Y (z-score normalization)
U_mean = U_df.mean()
U_std = U_df.std()
U_norm_df = (U_df - U_mean) / U_std

Y_mean = Y_df.mean()
Y_std = Y_df.std()
Y_norm_df = (Y_df - Y_mean) / Y_std

# Save normalization parameters for later use
U_mean.to_csv(os.path.join(output_dir, "U_mean.csv"))
U_std.to_csv(os.path.join(output_dir, "U_std.csv"))
Y_mean.to_csv(os.path.join(output_dir, "Y_mean.csv"))
Y_std.to_csv(os.path.join(output_dir, "Y_std.csv"))

# Use normalized data for identification
U = U_norm_df.values.T   # shape (m, T)
Y = Y_norm_df.values.T   # shape (p, T)
p, T = Y.shape
m, _ = U.shape

# 2. Build block-Hankel matrices
def block_hankel(data, depth):
    d, N = data.shape
    width = N - 2*depth + 1
    return np.vstack([data[:, k:k+width] for k in range(depth)])

Up = block_hankel(U, i)
Yp = block_hankel(Y, i)
Uf = block_hankel(U, i)
Yf = block_hankel(Y, i)
# 3. Oblique projection & SVD
W = np.vstack((Up, Yp))
Proj = Yf @ W.T @ la.inv(W @ W.T)
U2, S2, Vt2 = la.svd(Proj)
# pick model order n based on S2 decay
n = 15
Un = U2[:, :n]
Sn = np.diag(S2[:n])
Vn = Vt2[:n, :]

# 4. Initial C_raw and A0
O = Un @ np.sqrt(Sn)
C_raw = O[:p, :]                                     # (p×n)
O_up = O[:-p, :]
O_down = O[p:, :]
A0 = la.lstsq(O_up, O_down)[0]                       # (n×n)

# 5. Initial B0 via state sequence
X = np.sqrt(Sn) @ Vn                                  # (n×T-2i+1)
print('X.shape:', X.shape)
X1 = X[:, :-1]
X2 = X[:, 1:]
U1 = U[:, i:i + X1.shape[1]]

# 6. Define process output partitions
cols = list(Y_df.columns)
process_rows = {
    'P1': [i for i, c in enumerate(cols) if c.startswith('P1_')],
    'P2': [i for i, c in enumerate(cols) if c.startswith('P2_')],
    'P3': [i for i, c in enumerate(cols) if c.startswith('P3_')],
}

# Identify input columns for each process
input_cols = {
    'P1': [i for i, c in enumerate(U_df.columns) if c.startswith('P1_')],
    'P2': [i for i, c in enumerate(U_df.columns) if c.startswith('P2_')],
    'P3': [i for i, c in enumerate(U_df.columns) if c.startswith('P3_')],
}

# 7. Assign states → processes by C_raw participation
assignment = {}
for j in range(n):
    col = C_raw[:, j]
    scores = {p: np.linalg.norm(col[rows], 2) for p, rows in process_rows.items()}
    assignment[j] = max(scores, key=scores.get)

process_states = {p: [j for j, p_ass in assignment.items() if p_ass == p]
                  for p in process_rows}

# 8. Build block-diagonal C_block
C_block = np.zeros_like(C_raw)
for p, rows in process_rows.items():
    cols_idx = process_states[p]
    Yp = Y[rows, i:i + X.shape[1]]  # align time indices!
    Xp = X[cols_idx, :]
    Cp = Yp @ la.pinv(Xp)
    C_block[np.ix_(rows, cols_idx)] = Cp

# --- Reorder outputs, states, and inputs for true block-diagonal structure ---
ordered_output_indices = []
for p in ['P1', 'P2', 'P3']:
    ordered_output_indices.extend(process_rows[p])
ordered_state_indices = []
for p in ['P1', 'P2', 'P3']:
    ordered_state_indices.extend(process_states[p])
ordered_input_indices = []
for p in ['P1', 'P2', 'P3']:
    ordered_input_indices.extend(input_cols[p])

C_block_ordered = C_block[np.ix_(ordered_output_indices, ordered_state_indices)]

# Optionally, set column and row names for clarity
ordered_output_names = [cols[i] for i in ordered_output_indices]
ordered_state_names = []
for p in ['P1', 'P2', 'P3']:
    for idx in range(len(process_states[p])):
        ordered_state_names.append(f"{p}_{idx+1}")
ordered_input_names = [U_df.columns[i] for i in ordered_input_indices]

C_block_df = pd.DataFrame(C_block_ordered, index=ordered_output_names, columns=ordered_state_names)

# 9. Build block-diagonal B
B_block = np.zeros((n, m))
for pi, p in enumerate(['P1', 'P2', 'P3']):
    state_idx = process_states[p]
    input_idx = input_cols[p]
    U1_p = U[input_idx, i:i + X1.shape[1]]
    X1_p = X1[state_idx, :]
    X2_p = X2[state_idx, :]
    # Least-squares fit for this block
    if len(input_idx) > 0 and len(state_idx) > 0:
        # Use current A0 for initial fit
        Bp = (X2_p - A0[np.ix_(state_idx, state_idx)] @ X1_p) @ la.pinv(U1_p)
        B_block[np.ix_(state_idx, input_idx)] = Bp

# Reorder B to match state and input order
B_block_ordered = B_block[np.ix_(ordered_state_indices, ordered_input_indices)]
B_block_df = pd.DataFrame(B_block_ordered, index=ordered_state_names, columns=ordered_input_names)

# 10. Re-estimate A with block-diagonal B
A_block = np.zeros((n, n))
X1_ord = X1[ordered_state_indices, :]
X2_ord = X2[ordered_state_indices, :]
U1_ord = U1[ordered_input_indices, :]
A_block = (X2_ord - B_block_ordered @ U1_ord) @ la.pinv(X1_ord)

# 11. Save results in centralized_lti_system folder
pd.DataFrame(A_block, index=ordered_state_names, columns=ordered_state_names).to_csv(os.path.join(output_dir, "A_complete.csv"))
B_block_df.to_csv(os.path.join(output_dir, "B_complete.csv"))
C_block_df.to_csv(os.path.join(output_dir, "C_complete.csv"))
U_norm_df.to_csv(os.path.join(output_dir, "U_complete.csv"), index=False)
Y_norm_df.to_csv(os.path.join(output_dir, "Y_complete.csv"), index=False)

# 12. Save states with process-based column names, ordered as in C_block
state_names = ordered_state_names
X_ordered = X[ordered_state_indices, :]
states_df = pd.DataFrame(X_ordered.T, columns=state_names)
states_df.to_csv(os.path.join(output_dir, "states_complete.csv"), index=False)

# 13. Estimate block-diagonal process and observation noise covariance matrices

# Compute residuals using reordered matrices
W_res = X2_ord - A_block @ X1_ord - B_block_ordered @ U1_ord  # (n × (T-1))
V_res = Y[ordered_output_indices, i+1:i+1+X1_ord.shape[1]] - (C_block_ordered @ X_ordered[:, 1:])  # (p × (T-1))

# Initialize block-diagonal covariance matrices
Q_block = np.zeros((len(ordered_state_indices), len(ordered_state_indices)))
R_block = np.zeros((len(ordered_output_indices), len(ordered_output_indices)))

# Estimate process-noise Q per state-block
for p in ['P1', 'P2', 'P3']:
    state_idx = process_states[p]
    block_idx = [ordered_state_indices.index(j) for j in state_idx]
    if len(block_idx) == 0:
        continue
    Wp = W_res[block_idx, :]
    Qp = np.cov(Wp, bias=False)
    idx = np.ix_(block_idx, block_idx)
    Q_block[idx] = Qp

# Estimate observation-noise R per output-block
for p in ['P1', 'P2', 'P3']:
    row_idx = process_rows[p]
    block_idx = [ordered_output_indices.index(j) for j in row_idx]
    if len(block_idx) == 0:
        continue
    Vp = V_res[block_idx, :]
    Rp = np.cov(Vp, bias=False)
    idx = np.ix_(block_idx, block_idx)
    R_block[idx] = Rp

# Save Q and R as Q_complete.csv and R_complete.csv
pd.DataFrame(Q_block, index=ordered_state_names, columns=ordered_state_names).to_csv(os.path.join(output_dir, "Q_complete.csv"))
pd.DataFrame(R_block, index=ordered_output_names, columns=ordered_output_names).to_csv(os.path.join(output_dir, "R_complete.csv"))

print("Fitted LTI model with block-diagonal B and C, and re-estimated A. Files saved to:", output_dir)
print("Block-diagonal Q and R estimated and saved as Q_complete.csv and R_complete.csv.")

print("Fitted LTI model with block-diagonal B and C, and re-estimated A. Files saved to:", output_dir)