import numpy as np
import pandas as pd
import os
from scipy.linalg import block_diag

def create_block_diagonal_matrix(parent_folder, num_components, matrix_type):
    """
    Creates a block diagonal matrix from component matrices
    matrix_type: str, one of 'C', 'Q', or 'R'
    """
    component_matrices = []
    
    # Read each component matrix
    for i in range(1, num_components + 1):
        folder_name = f'C{i}'
        file_path = os.path.join(parent_folder, folder_name, f'{matrix_type}.csv')
        
        try:
            matrix = pd.read_csv(file_path, header=None).values
            component_matrices.append(matrix)
        except FileNotFoundError:
            print(f"Error: Could not find {file_path}")
            return None
    
    # Create block diagonal matrix
    complete_matrix = block_diag(*component_matrices)
    
    # Save the complete matrix
    output_path = os.path.join(parent_folder, f'{matrix_type}_complete.csv')
    pd.DataFrame(complete_matrix).to_csv(output_path, header=False, index=False)
    
    print(f"Block diagonal matrix for {matrix_type} saved to {output_path}")
    return complete_matrix

def process_all_matrices(parent_folder, num_components):
    """
    Process C, Q, and R matrices for all components
    """
    matrix_types = ['C', 'Q', 'R']
    results = {}
    
    for matrix_type in matrix_types:
        print(f"\nProcessing {matrix_type} matrices...")
        result = create_block_diagonal_matrix(parent_folder, num_components, matrix_type)
        if result is None:
            print(f"Failed to create complete {matrix_type} matrix")
        else:
            results[matrix_type] = result
    
    return results

def process_initial_states(parent_folder, num_components):
    """
    Extract second row from X.csv files and create concatenated initial state vector
    """
    initial_states = []
    
    # Process each component folder
    for i in range(1, num_components + 1):
        folder_name = f'C{i}'
        x_file_path = os.path.join(parent_folder, folder_name, 'X.csv')
        
        try:
            # Read X.csv and extract second row
            X_matrix = pd.read_csv(x_file_path, header=None).values
            x0_row = X_matrix[1:2, :]  # Extract second row (index 1)
            
            # Save individual x0 row to component folder
            x0_file_path = os.path.join(parent_folder, folder_name, 'x0.csv')
            pd.DataFrame(x0_row).to_csv(x0_file_path, header=False, index=False)
            print(f"Saved x0 for component {i} to {x0_file_path}")
            
            # Store for concatenation
            initial_states.append(x0_row)
            
        except FileNotFoundError:
            print(f"Error: Could not find {x_file_path}")
            return None
    
    # Concatenate all initial states horizontally
    x0_complete = np.hstack(initial_states)
    
    # Save concatenated initial state vector
    output_path = os.path.join(parent_folder, 'x0_complete.csv')
    pd.DataFrame(x0_complete).to_csv(output_path, header=False, index=False)
    print(f"\nSaved complete initial state vector to {output_path}")
    
    return x0_complete

def process_zero_initial_states(parent_folder, num_components):
    """
    Create zero initial state vectors based on A matrix dimensions
    """
    initial_states = []
    
    # Process each component folder
    for i in range(1, num_components + 1):
        folder_name = f'C{i}'
        a_file_path = os.path.join(parent_folder, folder_name, 'A.csv')
        
        try:
            # Read A matrix to get size
            A_matrix = pd.read_csv(a_file_path, header=None).values
            p = A_matrix.shape[0]  # A should be p x p
            
            if A_matrix.shape[0] != A_matrix.shape[1]:
                raise ValueError(f"A matrix in {folder_name} is not square")
            
            # Create zero initial state
            x0_row = np.zeros((1, p))  # 1 x p row vector
            
            # Save individual x0 row to component folder
            x0_file_path = os.path.join(parent_folder, folder_name, 'x0.csv')
            pd.DataFrame(x0_row).to_csv(x0_file_path, header=False, index=False)
            print(f"Saved zero x0 for component {i} to {x0_file_path}")
            
            # Store for concatenation
            initial_states.append(x0_row)
            
        except FileNotFoundError:
            print(f"Error: Could not find {a_file_path}")
            return None
    
    # Concatenate all initial states horizontally
    x0_complete = np.hstack(initial_states)
    
    # Save concatenated initial state vector
    output_path = os.path.join(parent_folder, 'x0_complete.csv')
    pd.DataFrame(x0_complete).to_csv(output_path, header=False, index=False)
    print(f"\nSaved complete initial state vector to {output_path}")
    
    return x0_complete

def create_control_matrices(parent_folder, num_components):
    """
    Create B matrices based on A matrix sizes
    """
    component_matrices = []
    
    # Process each component folder
    for i in range(1, num_components + 1):
        folder_name = f'C{i}'
        a_file_path = os.path.join(parent_folder, folder_name, 'A.csv')
        b_file_path = os.path.join(parent_folder, folder_name, 'B.csv')
        
        try:
            # Read A matrix to get size
            A_matrix = pd.read_csv(a_file_path, header=None).values
            rows, cols = A_matrix.shape
            
            # Create B matrix of zeros with same size
            B_matrix = np.zeros((rows, cols))
            
            # Save B matrix
            pd.DataFrame(B_matrix).to_csv(b_file_path, header=False, index=False)
            print(f"Created B matrix for component {i}")
            
            component_matrices.append(B_matrix)
            
        except FileNotFoundError:
            print(f"Error: Could not find {a_file_path}")
            return None
    
    # Create complete B matrix
    B_complete = block_diag(*component_matrices)
    
    # Save complete B matrix
    output_path = os.path.join(parent_folder, 'B_complete.csv')
    pd.DataFrame(B_complete).to_csv(output_path, header=False, index=False)
    print(f"\nSaved complete B matrix to {output_path}")
    
    return B_complete


def add_measurement_noise(parent_folder, num_components, sigma_r, seed):
    """
    Add noise to measurement data and measurement noise covariance
    """
    # Initialize random number generator
    rng = np.random.RandomState(seed)
    
    # First handle the complete R matrix in parent folder
    r_complete_path = os.path.join(parent_folder, 'R_complete.csv')
    try:
        # Read and update R_complete
        R_complete = pd.read_csv(r_complete_path, header=None).values
        D = R_complete.shape[0]
        
        if R_complete.shape[0] != R_complete.shape[1]:
            raise ValueError("R_complete matrix is not square")
            
        # Add noise to complete R matrix
        R_complete_add = np.eye(D) * sigma_r
        R_complete_new = R_complete + R_complete_add
        
        # Save updated R_complete
        pd.DataFrame(R_complete_new).to_csv(r_complete_path, header=False, index=False)
        print(f"Updated R_complete matrix in parent folder")
    
    except FileNotFoundError as e:
        print(f"Error: Could not find R_complete.csv in parent folder")
        raise e
    
    # Handle component folders
    for i in range(1, num_components + 1):
        folder_name = f'C{i}'
        r_file_path = os.path.join(parent_folder, folder_name, 'R.csv')
        y_file_path = os.path.join(parent_folder, folder_name, 'Y.csv')
        
        try:
            # Process R matrix
            R = pd.read_csv(r_file_path, header=None).values
            d = R.shape[0]
            
            if R.shape[0] != R.shape[1]:
                raise ValueError(f"R matrix in {folder_name} is not square")
            
            # Create additional noise covariance
            R_add = np.eye(d) * sigma_r
            R_new = R + R_add
            
            # Save updated R matrix
            pd.DataFrame(R_new).to_csv(r_file_path, header=False, index=False)
            print(f"Updated R matrix for component {i}")
            
            # Process Y matrix
            Y = pd.read_csv(y_file_path, header=None).values
            num_rows = Y.shape[0]
            
            if Y.shape[1] != d:
                raise ValueError(f"Y matrix columns in {folder_name} don't match R dimensions")
            
            # Generate and add noise to each row
            for j in range(num_rows):
                noise = rng.multivariate_normal(np.zeros(d), R_add)
                Y[j, :] += noise
            
            # Save updated Y matrix
            pd.DataFrame(Y).to_csv(y_file_path, header=False, index=False)
            print(f"Added noise to Y matrix for component {i}")
            
        except FileNotFoundError as e:
            print(f"Error: Could not find required files in {folder_name}")
            raise e
        except Exception as e:
            print(f"Error processing component {i}: {str(e)}")
            raise e


if __name__ == "__main__":
    # Define the inputs directly or use proper input prompts
    parent_folder = "/Users/home/Documents/naz/research_codes/uncert_prop/realworld_exp/SWaT/Training/P2"
    num_components = 5
    
    # Create and save all block diagonal matrices
    results = process_all_matrices(parent_folder, num_components)

    #  # Process initial states
    # x0_complete = process_initial_states(parent_folder, num_components)

    # Process zero initial states
    x0_complete = process_zero_initial_states(parent_folder, num_components)

    # Create and save control matrices
    B_complete = create_control_matrices(parent_folder, num_components)


    # # Parameters
    # parent_folder = "/Users/home/Documents/naz/research_codes/uncert_prop/realworld_exp/HAI/sigma_y_set1/sigma_y_10e1/P2"
    # num_components = 4
    # sigma_r = 10
    # seed = 42
    
    # # Add noise to measurements and update covariance matrices
    # add_measurement_noise(parent_folder, num_components, sigma_r, seed)

