"""
Configuration file for Video-R1 Dual Reasoning Training

This file contains all configurable parameters for the dual reasoning algorithm.
Modify these settings according to your environment and requirements.
"""

import os
from pathlib import Path

# Project paths
PROJECT_ROOT = Path(__file__).parent.parent.absolute()
DATA_DIR = PROJECT_ROOT / "data"
MODELS_DIR = PROJECT_ROOT / "models"
LOG_DIR = PROJECT_ROOT / "logs"

# Model configuration
DEFAULT_MODEL_NAME = "Qwen2.5-VL-7B-COT-SFT"
MODEL_PATH = MODELS_DIR / DEFAULT_MODEL_NAME

# Dataset configuration
DEFAULT_DATASET_NAME = "dual_reasoning_dataset.json"
DATASET_PATH = DATA_DIR / DEFAULT_DATASET_NAME

# Training hyperparameters
TRAINING_CONFIG = {
    # Basic training settings
    "max_prompt_length": 16384,
    "max_completion_length": 768,
    "per_device_train_batch_size": 1,
    "gradient_accumulation_steps": 1,
    "learning_rate": 1e-6,
    "lr_scheduler_type": "cosine",
    "weight_decay": 0.01,
    "num_train_epochs": 1,
    "logging_steps": 1,
    "save_steps": 200,

    # Model specific settings
    "bf16": True,
    "gradient_checkpointing": True,
    "attn_implementation": "flash_attention_2",
    "max_pixels": 401408,
    "save_only_model": True,

    # GRPO specific settings
    "beta": 0.04,
    "max_grad_norm": 5,
    "num_generations": 8,

    # Video-R1 specific settings
    "temporal": False,  # Set to True for T-GRPO
    "len_control": True,

    # Dual reasoning settings
    "dual_reasoning": True,
    "dual_reasoning_reward_list": [1.0, 0.7, -0.5, -1.0],
    "use_kl_check": False,
    "kl_lambda": 0.3,
    "reward_funcs": ["dual_reasoning", "format"]
}

# GPU configuration
GPU_CONFIG = {
    "num_gpus": int(os.environ.get("NUM_GPUS", "8")),
    "cuda_devices": os.environ.get("CUDA_VISIBLE_DEVICES", "0,1,2,3,4,5,6,7"),
    "master_addr": "127.0.0.1",
    "master_port": "23456"
}

# DeepSpeed configuration
DEEPSPEED_CONFIG_PATH = PROJECT_ROOT / "src/r1-v/local_scripts/zero3.json"

# Debug settings
DEBUG_CONFIG = {
    "debug_mode": os.environ.get("DEBUG_MODE", "false").lower() == "true",
    "log_path": LOG_DIR / "debug_dual_reasoning.txt"
}

def get_output_dir(model_name: str, dataset_name: str) -> Path:
    """Generate output directory name based on model and dataset."""
    model_abbr = Path(model_name).name
    dataset_abbr = Path(dataset_name).name
    return LOG_DIR / f"{model_abbr}-DualReasoning_{dataset_abbr}_bf16_frame_16"

def validate_paths():
    """Validate that required paths exist."""
    if not MODEL_PATH.exists():
        raise FileNotFoundError(f"Model path does not exist: {MODEL_PATH}")

    if not DATASET_PATH.exists():
        raise FileNotFoundError(f"Dataset path does not exist: {DATASET_PATH}")

    if not DEEPSPEED_CONFIG_PATH.exists():
        raise FileNotFoundError(f"DeepSpeed config does not exist: {DEEPSPEED_CONFIG_PATH}")

def get_run_name(model_name: str, dataset_name: str) -> str:
    """Generate run name for logging."""
    model_abbr = Path(model_name).name
    dataset_abbr = Path(dataset_name).name
    return f"{model_abbr}-DualReasoning_{dataset_abbr}_bf16_frame_16"