#!/bin/bash

# Video-R1 Dual Reasoning Training Script
# This script implements the dual reasoning algorithm with shuffled options
# to improve model robustness and reasoning consistency

cd src/r1-v

# Configuration
export DEBUG_MODE="true"  # Enable debug to see model rollout during training
export LOG_PATH="./debug_log_dual_reasoning.txt"

# Get the project root directory dynamically
PROJECT_ROOT="$(cd ../.. && pwd)"
echo "Project root: $PROJECT_ROOT"

# Model and dataset paths (configure these according to your setup)
MODEL_PATH="${MODEL_PATH:-${PROJECT_ROOT}/models/Qwen2.5-VL-7B-COT-SFT}"
DATASET_PATH="${DATASET_PATH:-${PROJECT_ROOT}/data/LLaVA-Video-large-swift-origin_new_videor1_format_updated_fixed_first_4600.json}"

# Validate paths
if [ ! -d "$MODEL_PATH" ]; then
    echo "Error: Model path does not exist: $MODEL_PATH"
    echo "Please set MODEL_PATH environment variable or place model in: ${PROJECT_ROOT}/models/Qwen2.5-VL-7B-COT-SFT"
    exit 1
fi

if [ ! -f "$DATASET_PATH" ]; then
    echo "Error: Dataset file does not exist: $DATASET_PATH"
    echo "Please set DATASET_PATH environment variable or place dataset in: ${PROJECT_ROOT}/data/"
    exit 1
fi

# Extract abbreviated names for logging
MODEL_NAME_ABBR="$(basename "$MODEL_PATH")"
DATASET_NAME_ABBR="$(basename "$DATASET_PATH")"

# GPU configuration (adjust based on your setup)
NUM_GPUS="${NUM_GPUS:-8}"
CUDA_DEVICES="${CUDA_VISIBLE_DEVICES:-0,1,2,3,4,5,6,7}"

echo "Starting dual reasoning training with:"
echo "  Model: $MODEL_PATH"
echo "  Dataset: $DATASET_PATH"
echo "  GPUs: $CUDA_DEVICES"
echo "  Output directory: ./log/${MODEL_NAME_ABBR}-DualReasoning_${DATASET_NAME_ABBR}_bf16_frame_16"

# Run the training
CUDA_VISIBLE_DEVICES=$CUDA_DEVICES torchrun --nproc_per_node="$NUM_GPUS" \
    --nnodes="1" \
    --node_rank="0" \
    --master_addr="127.0.0.1" \
    --master_port="23456" \
    src/open_r1/grpo.py \
    --output_dir "./log/${MODEL_NAME_ABBR}-DualReasoning_${DATASET_NAME_ABBR}_bf16_frame_16" \
    --model_name_or_path "$MODEL_PATH" \
    --dataset_name "$DATASET_PATH" \
    --deepspeed "$(pwd)/local_scripts/zero3.json" \
    --max_prompt_length 16384 \
    --max_completion_length 768 \
    --per_device_train_batch_size 1 \
    --gradient_accumulation_steps 1 \
    --learning_rate 1e-6 \
    --lr_scheduler_type "cosine" \
    --weight_decay 0.01 \
    --logging_steps 1 \
    --gradient_checkpointing true \
    --temporal false \
    --bf16 true \
    --len_control true \
    --attn_implementation flash_attention_2 \
    --max_pixels 401408 \
    --num_train_epochs 1 \
    --run_name "${MODEL_NAME_ABBR}-DualReasoning_${DATASET_NAME_ABBR}_bf16_frame_16" \
    --save_steps 200 \
    --beta 0.04 \
    --max_grad_norm 5 \
    --save_only_model true \
    --num_generations 8 \
    --dual_reasoning true \
    --dual_reasoning_reward_list 1.0 0.7 -0.5 -1.0 \
    --use_kl_check false \
    --kl_lambda 0.3 \
    --reward_funcs dual_reasoning format

echo "Training completed. Check logs at: ./log/${MODEL_NAME_ABBR}-DualReasoning_${DATASET_NAME_ABBR}_bf16_frame_16"