#!/usr/bin/env python3
"""
Validation script for Video-R1 Dual Reasoning setup
This script checks if all necessary files and configurations are in place.
"""

import os
import sys
import json
from pathlib import Path
from typing import List, Tuple

def check_file_exists(file_path: Path, description: str) -> bool:
    """Check if a file exists and report status."""
    if file_path.exists():
        print(f"✅ {description}: {file_path}")
        return True
    else:
        print(f"❌ {description}: {file_path} (NOT FOUND)")
        return False

def check_directory_exists(dir_path: Path, description: str) -> bool:
    """Check if a directory exists and report status."""
    if dir_path.is_dir():
        print(f"✅ {description}: {dir_path}")
        return True
    else:
        print(f"❌ {description}: {dir_path} (NOT FOUND)")
        return False

def check_python_imports() -> List[Tuple[str, bool]]:
    """Check if required Python packages can be imported."""
    required_packages = [
        "torch",
        "transformers",
        "datasets",
        "trl",
        "deepspeed",
        "nltk",
        "rouge_score"
    ]

    results = []
    for package in required_packages:
        try:
            __import__(package)
            print(f"✅ Python package: {package}")
            results.append((package, True))
        except ImportError:
            print(f"❌ Python package: {package} (NOT INSTALLED)")
            results.append((package, False))

    return results

def validate_dual_reasoning_setup():
    """Main validation function."""
    print("🔍 Validating Video-R1 Dual Reasoning Setup...")
    print("=" * 50)

    # Get project root
    project_root = Path(__file__).parent.absolute()
    print(f"Project root: {project_root}")
    print()

    # Check essential directories
    print("📁 Checking directories...")
    essential_dirs = [
        (project_root / "src" / "r1-v" / "src" / "open_r1", "Main source code"),
        (project_root / "src" / "qwen-vl-utils", "Qwen VL utilities"),
        (project_root / "scripts", "Training scripts"),
        (project_root / "config", "Configuration files"),
        (project_root / "data", "Data directory"),
        (project_root / "models", "Models directory"),
    ]

    dir_success = all(check_directory_exists(path, desc) for path, desc in essential_dirs)
    print()

    # Check essential files
    print("📄 Checking essential files...")
    essential_files = [
        (project_root / "scripts" / "run_dual_reasoning.sh", "Dual reasoning training script"),
        (project_root / "config" / "dual_reasoning_config.py", "Configuration file"),
        (project_root / "src" / "r1-v" / "src" / "open_r1" / "grpo.py", "Main GRPO training script"),
        (project_root / "src" / "r1-v" / "src" / "open_r1" / "trainer" / "grpo_trainer.py", "GRPO trainer"),
        (project_root / "setup.sh", "Environment setup script"),
        (project_root / "README.md", "Documentation"),
    ]

    file_success = all(check_file_exists(path, desc) for path, desc in essential_files)
    print()

    # Check Python imports
    print("🐍 Checking Python packages...")
    import_results = check_python_imports()
    import_success = all(success for _, success in import_results)
    print()

    # Check dual reasoning implementation
    print("🧠 Checking dual reasoning implementation...")
    grpo_file = project_root / "src" / "r1-v" / "src" / "open_r1" / "grpo.py"
    if grpo_file.exists():
        with open(grpo_file, 'r') as f:
            content = f.read()
            if "dual_reasoning_reward" in content:
                print("✅ Dual reasoning reward function found")
                dual_impl_success = True
            else:
                print("❌ Dual reasoning reward function not found")
                dual_impl_success = False
    else:
        dual_impl_success = False
    print()

    # Environment variables check
    print("🌍 Checking environment setup...")
    env_vars = ["MODEL_PATH", "DATASET_PATH", "CUDA_VISIBLE_DEVICES"]
    for var in env_vars:
        if var in os.environ:
            print(f"✅ Environment variable {var}: {os.environ[var]}")
        else:
            print(f"⚠️  Environment variable {var}: Not set (optional)")
    print()

    # Summary
    print("📊 Validation Summary")
    print("=" * 30)
    all_checks = [
        ("Directories", dir_success),
        ("Essential files", file_success),
        ("Python packages", import_success),
        ("Dual reasoning implementation", dual_impl_success)
    ]

    for check_name, success in all_checks:
        status = "✅ PASS" if success else "❌ FAIL"
        print(f"{check_name}: {status}")

    overall_success = all(success for _, success in all_checks)

    print()
    if overall_success:
        print("🎉 Setup validation PASSED! You can proceed with dual reasoning training.")
        print("📋 Next steps:")
        print("1. Place your model in: ./models/Qwen2.5-VL-7B-COT-SFT/")
        print("2. Place your dataset in: ./data/")
        print("3. Run: bash scripts/run_dual_reasoning.sh")
    else:
        print("⚠️  Setup validation FAILED. Please address the issues above.")
        print("📋 Common solutions:")
        print("1. Install missing Python packages: pip install -r requirements.txt")
        print("2. Run setup script: bash setup.sh")
        print("3. Check file paths and permissions")

    return overall_success

if __name__ == "__main__":
    success = validate_dual_reasoning_setup()
    sys.exit(0 if success else 1)