import math

class Furniture:
    def __init__(self, label, bbox):
        self.label = label
        self.bbox = bbox  # Bounding box as a list of two points
        self.x_min = min(bbox[0][0], bbox[1][0])
        self.x_max = max(bbox[0][0], bbox[1][0])
        self.y_min = min(bbox[0][1], bbox[1][1])
        self.y_max = max(bbox[0][1], bbox[1][1])
        self.z_min = min(bbox[0][2], bbox[1][2])
        self.z_max = max(bbox[0][2], bbox[1][2])
        self.x_center = (self.x_min + self.x_max) / 2
        self.y_center = (self.y_min + self.y_max) / 2
        self.z_center = (self.z_min + self.z_max) / 2
        self.x_size = self.x_max - self.x_min
        self.y_size = self.y_max - self.y_min
        self.z_size = self.z_max - self.z_min

def check_overlap(box1, box2):
    x_overlap = box1.x_min <= box2.x_max and box1.x_max >= box2.x_min
    y_overlap = box1.y_min <= box2.y_max and box1.y_max >= box2.y_min
    z_overlap = box1.z_min <= box2.z_max and box1.z_max >= box2.z_min
    return x_overlap and y_overlap and z_overlap

def compute_distance(box1, box2):
    dx = max(0, max(box2.x_min - box1.x_max, box1.x_min - box2.x_max))
    dy = max(0, max(box2.y_min - box1.y_max, box1.y_min - box2.y_max))
    dz = max(0, max(box2.z_min - box1.z_max, box1.z_min - box2.z_max))
    distance = math.sqrt(dx*dx + dy*dy + dz*dz)
    return distance

def fuzzy_relationship(value, size, axis):
    """
    Determine the linguistic term based on the normalized distance and axis.
    """
    ratio = value / size
    if axis == 'x':
        # Left / Right
        if ratio < -0.6:
            return "far left of"
        elif -0.6 <= ratio < -0.2:
            return "left of"
        elif -0.2 <= ratio < -0.05:
            return "slightly left of"
        elif -0.05 <= ratio <= 0.05:
            return "aligned along x-axis with"
        elif 0.05 < ratio <= 0.2:
            return "slightly right of"
        elif 0.2 < ratio <= 0.6:
            return "right of"
        else:
            return "far right of"
    elif axis == 'y':
        # Below / Above
        if ratio < -0.6:
            return "far below"
        elif -0.6 <= ratio < -0.2:
            return "below"
        elif -0.2 <= ratio < -0.05:
            return "slightly below"
        elif -0.05 <= ratio <= 0.05:
            return "aligned along y-axis with"
        elif 0.05 < ratio <= 0.2:
            return "slightly above"
        elif 0.2 < ratio <= 0.6:
            return "above"
        else:
            return "far above"
    elif axis == 'z':
        # Behind / In front of
        if ratio < -0.6:
            return "far behind"
        elif -0.6 <= ratio < -0.2:
            return "behind"
        elif -0.2 <= ratio < -0.05:
            return "slightly behind"
        elif -0.05 <= ratio <= 0.05:
            return "aligned along z-axis with"
        elif 0.05 < ratio <= 0.2:
            return "slightly in front of"
        elif 0.2 < ratio <= 0.6:
            return "in front of"
        else:
            return "far in front of"
    else:
        return "unknown"

def get_relative_position(box1, box2):
    positions = []
    # Calculate the normalized distance along each axis
    x_diff = box1.x_center - box2.x_center
    y_diff = box1.y_center - box2.y_center
    z_diff = box1.z_center - box2.z_center

    avg_x_size = (box1.x_size + box2.x_size) / 2
    avg_y_size = (box1.y_size + box2.y_size) / 2
    avg_z_size = (box1.z_size + box2.z_size) / 2

    # Along x-axis
    x_relation = fuzzy_relationship(x_diff, avg_x_size, 'x')
    positions.append(x_relation.replace("far ", ""))  # Simplify for natural language

    # Along y-axis
    y_relation = fuzzy_relationship(y_diff, avg_y_size, 'y')
    positions.append(y_relation.replace("far ", ""))

    # Along z-axis
    z_relation = fuzzy_relationship(z_diff, avg_z_size, 'z')
    positions.append(z_relation.replace("far ", ""))

    return positions


def number_to_text(n):
    ordinals = [
        "the first", "the second", "the third", "the fourth",
        "the fifth", "the sixth", "the seventh", "the eighth",
        "the ninth", "the tenth"
    ]
    if 1 <= n <= len(ordinals):
        return ordinals[n - 1]
    else:
        return f"the {n}th"

def get_furni_rel(data, if_print=False):
    # Create Furniture objects from the JSON data
    furnitures = []
    string_list = ""
    enumer_history = []
    for item in data['furniture']:
        label = item['label']
        if label in enumer_history:
            new_label = number_to_text(enumer_history.count(label) + 1) + " " + label
            print("We find a duplicate!")
        else:
            new_label = "the " + label
        enumer_history.append(label)
        bbox = item['bbox']
        furniture = Furniture(new_label, bbox)
        furnitures.append(furniture)

    # Analyze relationships among all furniture items
    for i in range(len(furnitures)):
        for j in range(i+1, len(furnitures)):
            box1 = furnitures[i]
            box2 = furnitures[j]
            distance = compute_distance(box1, box2)
            if distance < 0.5:
                # overlap = check_overlap(box1, box2)
                positions = get_relative_position(box1, box2)
                # new_string = ""
                # new_string += f"{box1.label} and {box2.label}:\n"
                # # new_string += f"  Very Close To: {'Yes' if overlap else 'No'}\n"
                # new_string += f"  Minimum Distance: {distance:.2f} units\n"
                # new_string += f"  Relative Positions: {box1.label} is {positions[0]} {box2.label},\n"
                # # new_string += f"                     {positions[1]} {box2.label},\n"
                # new_string += f"                     and {positions[2]} {box2.label}.\n\n"


                string_list += get_one_string(box1.label, box2.label, positions)
                # if if_print:
                #     print(new_string)
    
                # print(f"{box1.label} and {box2.label}:")
                # print(f"  Very Close To: {'Yes' if overlap else 'No'}")
                # print(f"  Minimum Distance: {distance:.2f} units")
                # print(f"  Relative Positions: {box1.label} is {positions[0]} {box2.label},")
                # # print(f"                     {positions[1]} {box2.label},")
                # print(f"                     and {positions[2]} {box2.label}.")
                # print()
    # if empty string, return None
    if string_list == "":
        return None
    if if_print:
        print(string_list)
    return string_list

def get_one_string(box1_label, box2_label, positions):
    if ("aligned" in positions[0]) and ("aligned" in positions[2]):
        return ""  # or handle that scenario in whatever way is needed
    elif "aligned" in positions[0]:
        return f"{box1_label} is {positions[2]} {box2_label}. "
    elif "aligned" in positions[2]:
        return f"{box1_label} is {positions[0]} {box2_label}. "
    else:
        return f"{box1_label} is {positions[0]} and {positions[2]} {box2_label}. "
