import math
import warnings
from typing import List, Optional, Tuple, Union

import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch import nn
from torch.nn import BCEWithLogitsLoss, CrossEntropyLoss, MSELoss


import math
import warnings
from typing import List, Optional, Tuple, Union

import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch import nn
from torch.nn import BCEWithLogitsLoss, CrossEntropyLoss, MSELoss

from transformers.activations import ACT2FN

# ----------------------------------------------------------#
# from transformers.cache_utils import Cache, DynamicCache
from .cache_utils import Cache, DynamicCachePlus

# ----------------------------------------------------------#
from transformers.modeling_attn_mask_utils import (
    AttentionMaskConverter,
    _prepare_4d_attention_mask,
    _prepare_4d_causal_attention_mask,
    _prepare_4d_causal_attention_mask_for_sdpa,
)
from transformers.modeling_outputs import (
    BaseModelOutputWithPast,
    CausalLMOutputWithPast,
    SequenceClassifierOutputWithPast,
)
from transformers.modeling_utils import PreTrainedModel
from transformers.pytorch_utils import (
    ALL_LAYERNORM_LAYERS,
    is_torch_greater_or_equal_than_1_13,
)
from transformers.utils import (
    add_start_docstrings,
    add_start_docstrings_to_model_forward,
    is_flash_attn_2_available,
    is_flash_attn_greater_or_equal_2_10,
    logging,
    replace_return_docstrings,
)
from transformers.utils.import_utils import is_torch_fx_available
from transformers.models.llama.configuration_llama import LlamaConfig

# ----------------------------------------------------------#
from dataclasses import dataclass
from transformers.utils import ModelOutput

from .custom_transformer_layer import (
    SelfTransformerEncoderBlock,
    CrossTransformerEncoderBlock,
)

special_text = {"ASSISTANT:": [319, 1799, 9047, 13566, 29901], "USER:": [11889, 29901]}
# ----------------------------------------------------------#

if is_flash_attn_2_available():
    from flash_attn import flash_attn_func, flash_attn_varlen_func
    from flash_attn.bert_padding import index_first_axis, pad_input, unpad_input  # noqa


# This makes `_prepare_4d_causal_attention_mask` a leaf function in the FX graph.
# It means that the function will not be traced through and simply appear as a node in the graph.
if is_torch_fx_available():
    if not is_torch_greater_or_equal_than_1_13:
        import torch.fx

    _prepare_4d_causal_attention_mask = torch.fx.wrap(_prepare_4d_causal_attention_mask)


logger = logging.get_logger(__name__)

_CONFIG_FOR_DOC = "LlamaConfig"


def _get_unpad_data(attention_mask):
    seqlens_in_batch = attention_mask.sum(dim=-1, dtype=torch.int32)
    indices = torch.nonzero(attention_mask.flatten(), as_tuple=False).flatten()
    max_seqlen_in_batch = seqlens_in_batch.max().item()
    cu_seqlens = F.pad(
        torch.cumsum(seqlens_in_batch, dim=0, dtype=torch.torch.int32), (1, 0)
    )
    return (
        indices,
        cu_seqlens,
        max_seqlen_in_batch,
    )


def _expand_mask(mask: torch.Tensor, dtype: torch.dtype, tgt_len: Optional[int] = None):
    warnings.warn(
        "Calling `transformers.models.llama.modeling_llama._prepare_4d_attention_mask` is deprecated and will be removed in v4.37. Use `transformers.modeling_attn_mask_utils._prepare_4d_attention_mask"
    )
    return _prepare_4d_attention_mask(mask=mask, dtype=dtype, tgt_len=tgt_len)


def _make_causal_mask(
    input_ids_shape: torch.Size,
    dtype: torch.dtype,
    device: torch.device,
    past_key_values_length: int = 0,
):
    warnings.warn(
        "Calling `transformers.models.llama.modeling_llama._make_causal_mask` is deprecated and will be removed in v4.37. Use `transformers.models.llama.modeling_llama.AttentionMaskConverter._make_causal_mask"
    )
    return AttentionMaskConverter._make_causal_mask(
        input_ids_shape=input_ids_shape,
        dtype=dtype,
        device=device,
        past_key_values_length=past_key_values_length,
    )


class LlamaRMSNorm(nn.Module):
    def __init__(self, hidden_size, eps=1e-6):
        """
        LlamaRMSNorm is equivalent to T5LayerNorm
        """
        super().__init__()
        self.weight = nn.Parameter(torch.ones(hidden_size))
        self.variance_epsilon = eps

    def forward(self, hidden_states):
        input_dtype = hidden_states.dtype
        hidden_states = hidden_states.to(torch.float32)
        variance = hidden_states.pow(2).mean(-1, keepdim=True)
        hidden_states = hidden_states * torch.rsqrt(variance + self.variance_epsilon)
        return self.weight * hidden_states.to(input_dtype)


ALL_LAYERNORM_LAYERS.append(LlamaRMSNorm)


class LlamaRotaryEmbedding(nn.Module):
    def __init__(self, dim, max_position_embeddings=2048, base=10000, device=None):
        super().__init__()

        self.dim = dim
        self.max_position_embeddings = max_position_embeddings
        self.base = base
        inv_freq = 1.0 / (
            self.base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim)
        )
        self.register_buffer("inv_freq", inv_freq, persistent=False)

        # Build here to make `torch.jit.trace` work.
        self._set_cos_sin_cache(
            seq_len=max_position_embeddings,
            device=self.inv_freq.device,
            dtype=torch.get_default_dtype(),
        )

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len
        t = torch.arange(
            self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype
        )

        freqs = torch.outer(t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer("cos_cached", emb.cos().to(dtype), persistent=False)
        self.register_buffer("sin_cached", emb.sin().to(dtype), persistent=False)

    def forward(self, x, seq_len=None):
        # x: [bs, num_attention_heads, seq_len, head_size]
        if seq_len > self.max_seq_len_cached:
            self._set_cos_sin_cache(seq_len=seq_len, device=x.device, dtype=x.dtype)

        return (
            self.cos_cached[:seq_len].to(dtype=x.dtype),
            self.sin_cached[:seq_len].to(dtype=x.dtype),
        )


class LlamaLinearScalingRotaryEmbedding(LlamaRotaryEmbedding):
    """LlamaRotaryEmbedding extended with linear scaling. Credits to the Reddit user /u/kaiokendev"""

    def __init__(
        self,
        dim,
        max_position_embeddings=2048,
        base=10000,
        device=None,
        scaling_factor=1.0,
    ):
        self.scaling_factor = scaling_factor
        super().__init__(dim, max_position_embeddings, base, device)

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len
        t = torch.arange(
            self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype
        )
        t = t / self.scaling_factor

        freqs = torch.outer(t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer("cos_cached", emb.cos().to(dtype), persistent=False)
        self.register_buffer("sin_cached", emb.sin().to(dtype), persistent=False)


class LlamaDynamicNTKScalingRotaryEmbedding(LlamaRotaryEmbedding):
    """LlamaRotaryEmbedding extended with Dynamic NTK scaling. Credits to the Reddit users /u/bloc97 and /u/emozilla"""

    def __init__(
        self,
        dim,
        max_position_embeddings=2048,
        base=10000,
        device=None,
        scaling_factor=1.0,
    ):
        self.scaling_factor = scaling_factor
        super().__init__(dim, max_position_embeddings, base, device)

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len

        if seq_len > self.max_position_embeddings:
            base = self.base * (
                (self.scaling_factor * seq_len / self.max_position_embeddings)
                - (self.scaling_factor - 1)
            ) ** (self.dim / (self.dim - 2))
            inv_freq = 1.0 / (
                base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim)
            )
            self.register_buffer("inv_freq", inv_freq, persistent=False)

        t = torch.arange(
            self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype
        )

        freqs = torch.outer(t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer("cos_cached", emb.cos().to(dtype), persistent=False)
        self.register_buffer("sin_cached", emb.sin().to(dtype), persistent=False)


def rotate_half(x):
    """Rotates half the hidden dims of the input."""
    x1 = x[..., : x.shape[-1] // 2]
    x2 = x[..., x.shape[-1] // 2 :]
    return torch.cat((-x2, x1), dim=-1)


def apply_rotary_pos_emb(q, k, cos, sin, position_ids, unsqueeze_dim=1):
    """Applies Rotary Position Embedding to the query and key tensors.

    Args:
        q (`torch.Tensor`): The query tensor.
        k (`torch.Tensor`): The key tensor.
        cos (`torch.Tensor`): The cosine part of the rotary embedding.
        sin (`torch.Tensor`): The sine part of the rotary embedding.
        position_ids (`torch.Tensor`):
            The position indices of the tokens corresponding to the query and key tensors. For example, this can be
            used to pass offsetted position ids when working with a KV-cache.
        unsqueeze_dim (`int`, *optional*, defaults to 1):
            The 'unsqueeze_dim' argument specifies the dimension along which to unsqueeze cos[position_ids] and
            sin[position_ids] so that they can be properly broadcasted to the dimensions of q and k. For example, note
            that cos[position_ids] and sin[position_ids] have the shape [batch_size, seq_len, head_dim]. Then, if q and
            k have the shape [batch_size, heads, seq_len, head_dim], then setting unsqueeze_dim=1 makes
            cos[position_ids] and sin[position_ids] broadcastable to the shapes of q and k. Similarly, if q and k have
            the shape [batch_size, seq_len, heads, head_dim], then set unsqueeze_dim=2.
    Returns:
        `tuple(torch.Tensor)` comprising of the query and key tensors rotated using the Rotary Position Embedding.
    """
    cos = cos[position_ids].unsqueeze(unsqueeze_dim)
    sin = sin[position_ids].unsqueeze(unsqueeze_dim)
    q_embed = (q * cos) + (rotate_half(q) * sin)
    k_embed = (k * cos) + (rotate_half(k) * sin)
    return q_embed, k_embed


class LlamaMLP(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.config = config
        self.hidden_size = config.hidden_size
        self.intermediate_size = config.intermediate_size
        self.gate_proj = nn.Linear(self.hidden_size, self.intermediate_size, bias=False)
        self.up_proj = nn.Linear(self.hidden_size, self.intermediate_size, bias=False)
        self.down_proj = nn.Linear(self.intermediate_size, self.hidden_size, bias=False)
        self.act_fn = ACT2FN[config.hidden_act]

    def forward(self, x):
        if self.config.pretraining_tp > 1:
            slice = self.intermediate_size // self.config.pretraining_tp
            gate_proj_slices = self.gate_proj.weight.split(slice, dim=0)
            up_proj_slices = self.up_proj.weight.split(slice, dim=0)
            down_proj_slices = self.down_proj.weight.split(slice, dim=1)

            gate_proj = torch.cat(
                [
                    F.linear(x, gate_proj_slices[i])
                    for i in range(self.config.pretraining_tp)
                ],
                dim=-1,
            )
            up_proj = torch.cat(
                [
                    F.linear(x, up_proj_slices[i])
                    for i in range(self.config.pretraining_tp)
                ],
                dim=-1,
            )

            intermediate_states = (self.act_fn(gate_proj) * up_proj).split(slice, dim=2)
            down_proj = [
                F.linear(intermediate_states[i], down_proj_slices[i])
                for i in range(self.config.pretraining_tp)
            ]
            down_proj = sum(down_proj)
        else:
            down_proj = self.down_proj(self.act_fn(self.gate_proj(x)) * self.up_proj(x))

        return down_proj


def repeat_kv(hidden_states: torch.Tensor, n_rep: int) -> torch.Tensor:
    """
    This is the equivalent of torch.repeat_interleave(x, dim=1, repeats=n_rep). The hidden states go from (batch,
    num_key_value_heads, seqlen, head_dim) to (batch, num_attention_heads, seqlen, head_dim)
    """
    batch, num_key_value_heads, slen, head_dim = hidden_states.shape
    if n_rep == 1:
        return hidden_states
    hidden_states = hidden_states[:, :, None, :, :].expand(
        batch, num_key_value_heads, n_rep, slen, head_dim
    )
    return hidden_states.reshape(batch, num_key_value_heads * n_rep, slen, head_dim)


class LlamaAttention(nn.Module):
    """Multi-headed attention from 'Attention Is All You Need' paper"""

    def __init__(self, config: LlamaConfig, layer_idx: Optional[int] = None):
        super().__init__()
        self.config = config
        self.layer_idx = layer_idx
        if layer_idx is None:
            logger.warning_once(
                f"Instantiating {self.__class__.__name__} without passing a `layer_idx` is not recommended and will "
                "lead to errors during the forward call if caching is used. Please make sure to provide a `layer_idx` "
                "when creating this class."
            )

        self.attention_dropout = config.attention_dropout
        self.hidden_size = config.hidden_size
        self.num_heads = config.num_attention_heads
        self.head_dim = self.hidden_size // self.num_heads
        self.num_key_value_heads = config.num_key_value_heads
        self.num_key_value_groups = self.num_heads // self.num_key_value_heads
        self.max_position_embeddings = config.max_position_embeddings
        self.rope_theta = config.rope_theta
        self.is_causal = True

        if (self.head_dim * self.num_heads) != self.hidden_size:
            raise ValueError(
                f"hidden_size must be divisible by num_heads (got `hidden_size`: {self.hidden_size}"
                f" and `num_heads`: {self.num_heads})."
            )

        self.q_proj = nn.Linear(
            self.hidden_size, self.num_heads * self.head_dim, bias=config.attention_bias
        )
        self.k_proj = nn.Linear(
            self.hidden_size,
            self.num_key_value_heads * self.head_dim,
            bias=config.attention_bias,
        )
        self.v_proj = nn.Linear(
            self.hidden_size,
            self.num_key_value_heads * self.head_dim,
            bias=config.attention_bias,
        )
        self.o_proj = nn.Linear(
            self.num_heads * self.head_dim, self.hidden_size, bias=config.attention_bias
        )
        self._init_rope()

    def _init_rope(self):
        if self.config.rope_scaling is None:
            self.rotary_emb = LlamaRotaryEmbedding(
                self.head_dim,
                max_position_embeddings=self.max_position_embeddings,
                base=self.rope_theta,
            )
        else:
            scaling_type = self.config.rope_scaling["type"]
            scaling_factor = self.config.rope_scaling["factor"]
            if scaling_type == "linear":
                self.rotary_emb = LlamaLinearScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                    base=self.rope_theta,
                )
            elif scaling_type == "dynamic":
                self.rotary_emb = LlamaDynamicNTKScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                    base=self.rope_theta,
                )
            else:
                raise ValueError(f"Unknown RoPE scaling type {scaling_type}")

    def _shape(self, tensor: torch.Tensor, seq_len: int, bsz: int):
        return (
            tensor.view(bsz, seq_len, self.num_heads, self.head_dim)
            .transpose(1, 2)
            .contiguous()
        )

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Cache] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

        bsz, q_len, _ = hidden_states.size()

        if self.config.pretraining_tp > 1:
            key_value_slicing = (
                self.num_key_value_heads * self.head_dim
            ) // self.config.pretraining_tp
            query_slices = self.q_proj.weight.split(
                (self.num_heads * self.head_dim) // self.config.pretraining_tp, dim=0
            )
            key_slices = self.k_proj.weight.split(key_value_slicing, dim=0)
            value_slices = self.v_proj.weight.split(key_value_slicing, dim=0)

            query_states = [
                F.linear(hidden_states, query_slices[i])
                for i in range(self.config.pretraining_tp)
            ]
            query_states = torch.cat(query_states, dim=-1)

            key_states = [
                F.linear(hidden_states, key_slices[i])
                for i in range(self.config.pretraining_tp)
            ]
            key_states = torch.cat(key_states, dim=-1)

            value_states = [
                F.linear(hidden_states, value_slices[i])
                for i in range(self.config.pretraining_tp)
            ]
            value_states = torch.cat(value_states, dim=-1)

        else:
            query_states = self.q_proj(hidden_states)
            key_states = self.k_proj(hidden_states)
            value_states = self.v_proj(hidden_states)

        query_states = query_states.view(
            bsz, q_len, self.num_heads, self.head_dim
        ).transpose(1, 2)
        key_states = key_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)
        value_states = value_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)

        kv_seq_len = key_states.shape[-2]
        if past_key_value is not None:
            if self.layer_idx is None:
                raise ValueError(
                    f"The cache structure has changed since version v4.36. If you are using {self.__class__.__name__} "
                    "for auto-regressive decoding with k/v caching, please make sure to initialize the attention class "
                    "with a layer index."
                )
            kv_seq_len += past_key_value.get_usable_length(kv_seq_len, self.layer_idx)
        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)
        query_states, key_states = apply_rotary_pos_emb(
            query_states, key_states, cos, sin, position_ids
        )

        if past_key_value is not None:
            cache_kwargs = {"sin": sin, "cos": cos}  # Specific to RoPE models
            key_states, value_states = past_key_value.update(
                key_states, value_states, self.layer_idx, cache_kwargs
            )

        key_states = repeat_kv(key_states, self.num_key_value_groups)
        value_states = repeat_kv(value_states, self.num_key_value_groups)

        attn_weights = torch.matmul(
            query_states, key_states.transpose(2, 3)
        ) / math.sqrt(self.head_dim)

        if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len):
            raise ValueError(
                f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                f" {attn_weights.size()}"
            )

        if attention_mask is not None:
            if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
                raise ValueError(
                    f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
                )
            attn_weights = attn_weights + attention_mask

        # upcast attention to fp32
        attn_weights = nn.functional.softmax(
            attn_weights, dim=-1, dtype=torch.float32
        ).to(query_states.dtype)
        attn_weights = nn.functional.dropout(
            attn_weights, p=self.attention_dropout, training=self.training
        )
        attn_output = torch.matmul(attn_weights, value_states)

        if attn_output.size() != (bsz, self.num_heads, q_len, self.head_dim):
            raise ValueError(
                f"`attn_output` should be of size {(bsz, self.num_heads, q_len, self.head_dim)}, but is"
                f" {attn_output.size()}"
            )

        attn_output = attn_output.transpose(1, 2).contiguous()

        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size)

        if self.config.pretraining_tp > 1:
            attn_output = attn_output.split(
                self.hidden_size // self.config.pretraining_tp, dim=2
            )
            o_proj_slices = self.o_proj.weight.split(
                self.hidden_size // self.config.pretraining_tp, dim=1
            )
            attn_output = sum(
                [
                    F.linear(attn_output[i], o_proj_slices[i])
                    for i in range(self.config.pretraining_tp)
                ]
            )
        else:
            attn_output = self.o_proj(attn_output)

        if not output_attentions:
            attn_weights = None

        return attn_output, attn_weights, past_key_value


class LlamaFlashAttention2(LlamaAttention):
    """
    Llama flash attention module. This module inherits from `LlamaAttention` as the weights of the module stays
    untouched. The only required change would be on the forward pass where it needs to correctly call the public API of
    flash attention and deal with padding tokens in case the input contains any of them.
    """

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        # TODO: Should be removed once Flash Attention for RoCm is bumped to 2.1.
        # flash_attn<2.1 generates top-left aligned causal mask, while what is needed here is bottom-right alignement, that was made default for flash_attn>=2.1. This attribute is used to handle this difference. Reference: https://github.com/Dao-AILab/flash-attention/releases/tag/v2.1.0.
        # Beware that with flash_attn<2.1, using q_seqlen != k_seqlen (except for the case q_seqlen == 1) produces a wrong mask (top-left).
        self._flash_attn_uses_top_left_mask = not is_flash_attn_greater_or_equal_2_10()

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.LongTensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Cache] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        # LlamaFlashAttention2 attention does not support output_attentions
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

            # overwrite attention_mask with padding_mask
            attention_mask = kwargs.pop("padding_mask")

        output_attentions = False

        bsz, q_len, _ = hidden_states.size()

        query_states = self.q_proj(hidden_states)
        key_states = self.k_proj(hidden_states)
        value_states = self.v_proj(hidden_states)

        # Flash attention requires the input to have the shape
        # batch_size x seq_length x head_dim x hidden_dim
        # therefore we just need to keep the original shape
        query_states = query_states.view(
            bsz, q_len, self.num_heads, self.head_dim
        ).transpose(1, 2)
        key_states = key_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)
        value_states = value_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)

        kv_seq_len = key_states.shape[-2]
        if past_key_value is not None:
            kv_seq_len += past_key_value.get_usable_length(kv_seq_len, self.layer_idx)
        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)
        query_states, key_states = apply_rotary_pos_emb(
            query_states, key_states, cos, sin, position_ids
        )

        if past_key_value is not None:
            cache_kwargs = {"sin": sin, "cos": cos}  # Specific to RoPE models
            key_states, value_states = past_key_value.update(
                key_states, value_states, self.layer_idx, cache_kwargs
            )

        # TODO: These transpose are quite inefficient but Flash Attention requires the layout [batch_size, sequence_length, num_heads, head_dim]. We would need to refactor the KV cache
        # to be able to avoid many of these transpose/reshape/view.
        query_states = query_states.transpose(1, 2)
        key_states = key_states.transpose(1, 2)
        value_states = value_states.transpose(1, 2)

        dropout_rate = self.attention_dropout if self.training else 0.0

        # In PEFT, usually we cast the layer norms in float32 for training stability reasons
        # therefore the input hidden states gets silently casted in float32. Hence, we need
        # cast them back in the correct dtype just to be sure everything works as expected.
        # This might slowdown training & inference so it is recommended to not cast the LayerNorms
        # in fp32. (LlamaRMSNorm handles it correctly)

        input_dtype = query_states.dtype
        if input_dtype == torch.float32:
            if torch.is_autocast_enabled():
                target_dtype = torch.get_autocast_gpu_dtype()
            # Handle the case where the model is quantized
            elif hasattr(self.config, "_pre_quantization_dtype"):
                target_dtype = self.config._pre_quantization_dtype
            else:
                target_dtype = self.q_proj.weight.dtype

            logger.warning_once(
                f"The input hidden states seems to be silently casted in float32, this might be related to"
                f" the fact you have upcasted embedding or layer norm layers in float32. We will cast back the input in"
                f" {target_dtype}."
            )

            query_states = query_states.to(target_dtype)
            key_states = key_states.to(target_dtype)
            value_states = value_states.to(target_dtype)

        attn_output = self._flash_attention_forward(
            query_states,
            key_states,
            value_states,
            attention_mask,
            q_len,
            dropout=dropout_rate,
        )

        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size).contiguous()
        attn_output = self.o_proj(attn_output)

        if not output_attentions:
            attn_weights = None

        return attn_output, attn_weights, past_key_value

    def _flash_attention_forward(
        self,
        query_states,
        key_states,
        value_states,
        attention_mask,
        query_length,
        dropout=0.0,
        softmax_scale=None,
    ):
        """
        Calls the forward method of Flash Attention - if the input hidden states contain at least one padding token
        first unpad the input, then computes the attention scores and pad the final attention scores.

        Args:
            query_states (`torch.Tensor`):
                Input query states to be passed to Flash Attention API
            key_states (`torch.Tensor`):
                Input key states to be passed to Flash Attention API
            value_states (`torch.Tensor`):
                Input value states to be passed to Flash Attention API
            attention_mask (`torch.Tensor`):
                The padding mask - corresponds to a tensor of size `(batch_size, seq_len)` where 0 stands for the
                position of padding tokens and 1 for the position of non-padding tokens.
            dropout (`int`, *optional*):
                Attention dropout
            softmax_scale (`float`, *optional*):
                The scaling of QK^T before applying softmax. Default to 1 / sqrt(head_dim)
        """
        if not self._flash_attn_uses_top_left_mask:
            causal = self.is_causal
        else:
            # TODO: Remove the `query_length != 1` check once Flash Attention for RoCm is bumped to 2.1. For details, please see the comment in LlamaFlashAttention2 __init__.
            causal = self.is_causal and query_length != 1

        # Contains at least one padding token in the sequence
        if attention_mask is not None:
            batch_size = query_states.shape[0]
            (
                query_states,
                key_states,
                value_states,
                indices_q,
                cu_seq_lens,
                max_seq_lens,
            ) = self._upad_input(
                query_states, key_states, value_states, attention_mask, query_length
            )

            cu_seqlens_q, cu_seqlens_k = cu_seq_lens
            max_seqlen_in_batch_q, max_seqlen_in_batch_k = max_seq_lens

            attn_output_unpad = flash_attn_varlen_func(
                query_states,
                key_states,
                value_states,
                cu_seqlens_q=cu_seqlens_q,
                cu_seqlens_k=cu_seqlens_k,
                max_seqlen_q=max_seqlen_in_batch_q,
                max_seqlen_k=max_seqlen_in_batch_k,
                dropout_p=dropout,
                softmax_scale=softmax_scale,
                causal=causal,
            )

            attn_output = pad_input(
                attn_output_unpad, indices_q, batch_size, query_length
            )
        else:
            attn_output = flash_attn_func(
                query_states,
                key_states,
                value_states,
                dropout,
                softmax_scale=softmax_scale,
                causal=causal,
            )

        return attn_output

    def _upad_input(
        self, query_layer, key_layer, value_layer, attention_mask, query_length
    ):
        indices_k, cu_seqlens_k, max_seqlen_in_batch_k = _get_unpad_data(attention_mask)
        batch_size, kv_seq_len, num_key_value_heads, head_dim = key_layer.shape

        key_layer = index_first_axis(
            key_layer.reshape(batch_size * kv_seq_len, num_key_value_heads, head_dim),
            indices_k,
        )
        value_layer = index_first_axis(
            value_layer.reshape(batch_size * kv_seq_len, num_key_value_heads, head_dim),
            indices_k,
        )
        if query_length == kv_seq_len:
            query_layer = index_first_axis(
                query_layer.reshape(batch_size * kv_seq_len, self.num_heads, head_dim),
                indices_k,
            )
            cu_seqlens_q = cu_seqlens_k
            max_seqlen_in_batch_q = max_seqlen_in_batch_k
            indices_q = indices_k
        elif query_length == 1:
            max_seqlen_in_batch_q = 1
            cu_seqlens_q = torch.arange(
                batch_size + 1, dtype=torch.int32, device=query_layer.device
            )  # There is a memcpy here, that is very bad.
            indices_q = cu_seqlens_q[:-1]
            query_layer = query_layer.squeeze(1)
        else:
            # The -q_len: slice assumes left padding.
            attention_mask = attention_mask[:, -query_length:]
            query_layer, indices_q, cu_seqlens_q, max_seqlen_in_batch_q = unpad_input(
                query_layer, attention_mask
            )

        return (
            query_layer,
            key_layer,
            value_layer,
            indices_q,
            (cu_seqlens_q, cu_seqlens_k),
            (max_seqlen_in_batch_q, max_seqlen_in_batch_k),
        )


# class LlamaSdpaAttention(LlamaAttention):
#     """
#     Llama attention module using torch.nn.functional.scaled_dot_product_attention. This module inherits from
#     `LlamaAttention` as the weights of the module stays untouched. The only changes are on the forward pass to adapt to
#     SDPA API.
#     """

#     # Adapted from LlamaAttention.forward
#     def forward(
#         self,
#         hidden_states: torch.Tensor,
#         attention_mask: Optional[torch.Tensor] = None,
#         position_ids: Optional[torch.LongTensor] = None,
#         past_key_value: Optional[Cache] = None,
#         output_attentions: bool = False,
#         use_cache: bool = False,
#     ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
#         if output_attentions:
#             # TODO: Improve this warning with e.g. `model.config.attn_implementation = "manual"` once this is implemented.
#             logger.warning_once(
#                 "LlamaModel is using LlamaSdpaAttention, but `torch.nn.functional.scaled_dot_product_attention` does not support `output_attentions=True`. Falling back to the manual attention implementation, "
#                 'but specifying the manual implementation will be required from Transformers version v5.0.0 onwards. This warning can be removed using the argument `attn_implementation="eager"` when loading the model.'
#             )
#             return super().forward(
#                 hidden_states=hidden_states,
#                 attention_mask=attention_mask,
#                 position_ids=position_ids,
#                 past_key_value=past_key_value,
#                 output_attentions=output_attentions,
#                 use_cache=use_cache,
#             )

#         bsz, q_len, _ = hidden_states.size()

#         query_states = self.q_proj(hidden_states)
#         key_states = self.k_proj(hidden_states)
#         value_states = self.v_proj(hidden_states)

#         query_states = query_states.view(
#             bsz, q_len, self.num_heads, self.head_dim
#         ).transpose(1, 2)
#         key_states = key_states.view(
#             bsz, q_len, self.num_key_value_heads, self.head_dim
#         ).transpose(1, 2)
#         value_states = value_states.view(
#             bsz, q_len, self.num_key_value_heads, self.head_dim
#         ).transpose(1, 2)

#         kv_seq_len = key_states.shape[-2]
#         if past_key_value is not None:
#             kv_seq_len += past_key_value.get_usable_length(kv_seq_len, self.layer_idx)
#         cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)

#         query_states, key_states = apply_rotary_pos_emb(
#             query_states, key_states, cos, sin, position_ids
#         )

#         if past_key_value is not None:
#             cache_kwargs = {"sin": sin, "cos": cos}  # Specific to RoPE models
#             key_states, value_states = past_key_value.update(
#                 key_states, value_states, self.layer_idx, cache_kwargs
#             )

#         key_states = repeat_kv(key_states, self.num_key_value_groups)
#         value_states = repeat_kv(value_states, self.num_key_value_groups)

#         if attention_mask is not None:
#             if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
#                 raise ValueError(
#                     f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
#                 )

#         # SDPA with memory-efficient backend is currently (torch==2.1.2) bugged with non-contiguous inputs with custom attn_mask,
#         # Reference: https://github.com/pytorch/pytorch/issues/112577.
#         if query_states.device.type == "cuda" and attention_mask is not None:
#             query_states = query_states.contiguous()
#             key_states = key_states.contiguous()
#             value_states = value_states.contiguous()

#         attn_output = torch.nn.functional.scaled_dot_product_attention(
#             query_states,
#             key_states,
#             value_states,
#             attn_mask=attention_mask,
#             dropout_p=self.attention_dropout if self.training else 0.0,
#             # The q_len > 1 is necessary to match with AttentionMaskConverter.to_causal_4d that does not create a causal mask in case q_len == 1.
#             is_causal=self.is_causal and attention_mask is None and q_len > 1,
#         )

#         attn_output = attn_output.transpose(1, 2).contiguous()
#         attn_output = attn_output.reshape(bsz, q_len, self.hidden_size)

#         attn_output = self.o_proj(attn_output)

#         return attn_output, None, past_key_value


def softmax_with_policy(attn, policy, eps=1e-6):
    B, N, _ = policy.size()
    B, H, N, N = attn.size()
    attn_policy = policy.reshape(B, 1, 1, N)  # * policy.reshape(B, 1, N, 1)
    eye = torch.eye(N, dtype=attn_policy.dtype, device=attn_policy.device).view(
        1, 1, N, N
    )
    attn_policy = attn_policy + (1.0 - attn_policy) * eye
    max_att = torch.max(attn, dim=-1, keepdim=True)[0]
    attn = attn - max_att
    # attn = attn.exp_() * attn_policy
    # return attn / attn.sum(dim=-1, keepdim=True)

    # for stable training
    attn = attn.to(torch.float32).exp_() * attn_policy.to(torch.float32)
    attn = (attn + eps / N) / (attn.sum(dim=-1, keepdim=True) + eps)
    return attn.type_as(max_att)


# Efficient implementation equivalent to the following:
def scaled_dot_product_attention_with_policy(
    query,
    key,
    value,
    attn_mask=None,
    dropout_p=0.0,
    is_causal=False,
    scale=None,
    policy=None,
) -> torch.Tensor:
    B = query.size(0)
    L, S = query.size(-2), key.size(-2)
    scale_factor = 1 / math.sqrt(query.size(-1)) if scale is None else scale
    if attn_mask is not None:
        attn_bias = torch.zeros_like(attn_mask, dtype=query.dtype, device=query.device)
    else:
        attn_bias = torch.zeros(B, 1, L, S, dtype=query.dtype, device=query.device)
    if is_causal:
        assert attn_mask is None
        temp_mask = torch.ones(B, 1, L, S, dtype=torch.bool, device=query.device).tril(
            diagonal=0
        )
        attn_bias.masked_fill_(temp_mask.logical_not(), float("-inf"))
        attn_bias.to(query.dtype)

    if attn_mask is not None:
        if attn_mask.dtype == torch.bool:
            attn_bias.masked_fill_(attn_mask.logical_not(), float("-inf"))
        else:
            attn_bias += attn_mask
    attn_weight = query @ key.transpose(-2, -1) * scale_factor
    attn_weight += attn_bias
    if policy is not None:
        attn_weight = softmax_with_policy(attn_weight, policy=policy)
    else:
        attn_weight = torch.softmax(attn_weight, dim=-1)
    attn_weight = torch.dropout(attn_weight, dropout_p, train=True)
    return attn_weight @ value


class DynamicLlamaSdpaAttention(LlamaAttention):
    """
    Llama attention module using torch.nn.functional.scaled_dot_product_attention. This module inherits from
    `LlamaAttention` as the weights of the module stays untouched. The only changes are on the forward pass to adapt to
    SDPA API.
    """

    # Adapted from LlamaAttention.forward
    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Cache] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        policy: Optional[torch.Tensor] = None,
        kv_seq_len_for_position: Optional[int] = None,
        sparse_layer: Optional[int] = None,
        text_decision: Optional[List[bool]] = None,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        if output_attentions:
            # TODO: Improve this warning with e.g. `model.config.attn_implementation = "manual"` once this is implemented.
            logger.warning_once(
                "LlamaModel is using LlamaSdpaAttention, but `torch.nn.functional.scaled_dot_product_attention` does not support `output_attentions=True`. Falling back to the manual attention implementation, "
                'but specifying the manual implementation will be required from Transformers version v5.0.0 onwards. This warning can be removed using the argument `attn_implementation="eager"` when loading the model.'
            )
            return super().forward(
                hidden_states=hidden_states,
                attention_mask=attention_mask,
                position_ids=position_ids,
                past_key_value=past_key_value,
                output_attentions=output_attentions,
                use_cache=use_cache,
            )

        bsz, q_len, _ = hidden_states.size()

        query_states = self.q_proj(hidden_states)
        key_states = self.k_proj(hidden_states)
        value_states = self.v_proj(hidden_states)

        query_states = query_states.view(
            bsz, q_len, self.num_heads, self.head_dim
        ).transpose(1, 2)
        key_states = key_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)
        value_states = value_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)

        # ----------------------------------------------------------#
        kv_seq_len = key_states.shape[-2]
        if kv_seq_len_for_position is None:
            kv_seq_len_for_position = key_states.shape[-2]
        if past_key_value is not None:
            kv_seq_len += past_key_value.get_usable_length(kv_seq_len, self.layer_idx)
            if sparse_layer is not None and self.layer_idx >= sparse_layer:
                kv_seq_len_for_position += (
                    past_key_value.get_usable_length(
                        kv_seq_len_for_position, sparse_layer - 1
                    )
                    - kv_seq_len_for_position
                )
            else:
                kv_seq_len_for_position += past_key_value.get_usable_length(
                    kv_seq_len_for_position, self.layer_idx
                )
        # ----------------------------------------------------------#
        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len_for_position)

        query_states, key_states = apply_rotary_pos_emb(
            query_states, key_states, cos, sin, position_ids
        )

        if past_key_value is not None:
            cache_kwargs = {"sin": sin, "cos": cos}  # Specific to RoPE models

            # ----------------------------------------------------------#
            # key_states, value_states = past_key_value.update(
            #     key_states,
            #     value_states,
            #     self.layer_idx,
            #     cache_kwargs,
            #     text_decision,
            # )

            if text_decision is not None:
                # Fix one bug
                temp_key_states, temp_value_states = key_states, value_states
                key_states, value_states = past_key_value.get_cache(
                    key_states,
                    value_states,
                    self.layer_idx,
                    cache_kwargs,
                )
                _, _ = past_key_value.update(
                    temp_key_states,
                    temp_value_states,
                    self.layer_idx,
                    cache_kwargs,
                    text_decision,
                )
            else:
                key_states, value_states = past_key_value.update(
                    key_states,
                    value_states,
                    self.layer_idx,
                    cache_kwargs,
                )
            # ----------------------------------------------------------#

        key_states = repeat_kv(key_states, self.num_key_value_groups)
        value_states = repeat_kv(value_states, self.num_key_value_groups)

        if attention_mask is not None:
            # if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
            #     raise ValueError(
            #         f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
            #     )
            if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
                raise ValueError(
                    f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
                )

        # SDPA with memory-efficient backend is currently (torch==2.1.2) bugged with non-contiguous inputs with custom attn_mask,
        # Reference: https://github.com/pytorch/pytorch/issues/112577.
        if query_states.device.type == "cuda" and attention_mask is not None:
            query_states = query_states.contiguous()
            key_states = key_states.contiguous()
            value_states = value_states.contiguous()

        if policy is not None:
            attn_output = scaled_dot_product_attention_with_policy(
                query_states,
                key_states,
                value_states,
                attn_mask=attention_mask,
                dropout_p=self.attention_dropout if self.training else 0.0,
                # The q_len > 1 is necessary to match with AttentionMaskConverter.to_causal_4d that does not create a causal mask in case q_len == 1.
                is_causal=self.is_causal and attention_mask is None and q_len > 1,
                policy=policy,
            )
        else:
            attn_output = torch.nn.functional.scaled_dot_product_attention(
                query_states,
                key_states,
                value_states,
                attn_mask=attention_mask,
                dropout_p=self.attention_dropout if self.training else 0.0,
                # The q_len > 1 is necessary to match with AttentionMaskConverter.to_causal_4d that does not create a causal mask in case q_len == 1.
                is_causal=self.is_causal and attention_mask is None and q_len > 1,
            )

        attn_output = attn_output.transpose(1, 2).contiguous()
        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size)

        attn_output = self.o_proj(attn_output)

        return attn_output, None, past_key_value


LLAMA_ATTENTION_CLASSES = {
    "eager": LlamaAttention,
    "flash_attention_2": LlamaFlashAttention2,
    # "sdpa": LlamaSdpaAttention,
    "sdpa": DynamicLlamaSdpaAttention,
}


# class LlamaDecoderLayer(nn.Module):
#     def __init__(self, config: LlamaConfig, layer_idx: int):
#         super().__init__()
#         self.hidden_size = config.hidden_size

#         self.self_attn = LLAMA_ATTENTION_CLASSES[config._attn_implementation](
#             config=config, layer_idx=layer_idx
#         )

#         self.mlp = LlamaMLP(config)
#         self.input_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
#         self.post_attention_layernorm = LlamaRMSNorm(
#             config.hidden_size, eps=config.rms_norm_eps
#         )

#     def forward(
#         self,
#         hidden_states: torch.Tensor,
#         attention_mask: Optional[torch.Tensor] = None,
#         position_ids: Optional[torch.LongTensor] = None,
#         past_key_value: Optional[Tuple[torch.Tensor]] = None,
#         output_attentions: Optional[bool] = False,
#         use_cache: Optional[bool] = False,
#         **kwargs,
#     ) -> Tuple[
#         torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]
#     ]:
#         """
#         Args:
#             hidden_states (`torch.FloatTensor`): input to the layer of shape `(batch, seq_len, embed_dim)`
#             attention_mask (`torch.FloatTensor`, *optional*):
#                 attention mask of size `(batch_size, sequence_length)` if flash attention is used or `(batch_size, 1,
#                 query_sequence_length, key_sequence_length)` if default attention is used.
#             output_attentions (`bool`, *optional*):
#                 Whether or not to return the attentions tensors of all attention layers. See `attentions` under
#                 returned tensors for more detail.
#             use_cache (`bool`, *optional*):
#                 If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding
#                 (see `past_key_values`).
#             past_key_value (`Tuple(torch.FloatTensor)`, *optional*): cached past key and value projection states
#         """
#         if "padding_mask" in kwargs:
#             warnings.warn(
#                 "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
#             )

#         residual = hidden_states

#         hidden_states = self.input_layernorm(hidden_states)

#         # Self Attention
#         hidden_states, self_attn_weights, present_key_value = self.self_attn(
#             hidden_states=hidden_states,
#             attention_mask=attention_mask,
#             position_ids=position_ids,
#             past_key_value=past_key_value,
#             output_attentions=output_attentions,
#             use_cache=use_cache,
#             **kwargs,
#         )
#         hidden_states = residual + hidden_states

#         # Fully Connected
#         residual = hidden_states
#         hidden_states = self.post_attention_layernorm(hidden_states)
#         hidden_states = self.mlp(hidden_states)
#         hidden_states = residual + hidden_states

#         outputs = (hidden_states,)

#         if output_attentions:
#             outputs += (self_attn_weights,)

#         if use_cache:
#             outputs += (present_key_value,)

#         return outputs


class DynamicLlamaDecoderLayer(nn.Module):
    def __init__(self, config: LlamaConfig, layer_idx: int):
        super().__init__()
        self.hidden_size = config.hidden_size

        self.self_attn = LLAMA_ATTENTION_CLASSES[config._attn_implementation](
            config=config, layer_idx=layer_idx
        )

        self.mlp = LlamaMLP(config)
        self.input_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
        self.post_attention_layernorm = LlamaRMSNorm(
            config.hidden_size, eps=config.rms_norm_eps
        )

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        policy: Optional[torch.Tensor] = None,
        kv_seq_len_for_position: Optional[int] = None,
        sparse_layer: Optional[int] = None,
        text_decision: Optional[List[bool]] = None,
        **kwargs,
    ) -> Tuple[
        torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]
    ]:
        """
        Args:
            hidden_states (`torch.FloatTensor`): input to the layer of shape `(batch, seq_len, embed_dim)`
            attention_mask (`torch.FloatTensor`, *optional*):
                attention mask of size `(batch_size, sequence_length)` if flash attention is used or `(batch_size, 1,
                query_sequence_length, key_sequence_length)` if default attention is used.
            output_attentions (`bool`, *optional*):
                Whether or not to return the attentions tensors of all attention layers. See `attentions` under
                returned tensors for more detail.
            use_cache (`bool`, *optional*):
                If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding
                (see `past_key_values`).
            past_key_value (`Tuple(torch.FloatTensor)`, *optional*): cached past key and value projection states
        """
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

        residual = hidden_states

        hidden_states = self.input_layernorm(hidden_states)

        # Self Attention
        hidden_states, self_attn_weights, present_key_value = self.self_attn(
            hidden_states=hidden_states,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_value=past_key_value,
            output_attentions=output_attentions,
            use_cache=use_cache,
            policy=policy,
            kv_seq_len_for_position=kv_seq_len_for_position,
            sparse_layer=sparse_layer,
            text_decision=text_decision,
            **kwargs,
        )
        hidden_states = residual + hidden_states

        # Fully Connected
        residual = hidden_states
        hidden_states = self.post_attention_layernorm(hidden_states)
        hidden_states = self.mlp(hidden_states)
        hidden_states = residual + hidden_states

        outputs = (hidden_states,)

        if output_attentions:
            outputs += (self_attn_weights,)

        if use_cache:
            outputs += (present_key_value,)

        return outputs


class VisionPredictor(nn.Module):
    def __init__(
        self,
        input_dim=4096,
        d_model=512,
        nhead=8,
        dim_feedforward=2048,
        num_layers=2,
    ):
        super().__init__()
        self.input_dim = input_dim
        self.d_model = d_model
        self.nhead = nhead
        self.dim_feedforward = dim_feedforward
        self.num_layers = num_layers

        self.down_mlp = nn.Sequential(
            nn.LayerNorm(self.input_dim),
            nn.Linear(self.input_dim, self.d_model),
            nn.GELU(),
        )
        self.transformer = nn.Sequential(
            *[
                SelfTransformerEncoderBlock(
                    dim=self.d_model,
                    num_heads=self.nhead,
                    mlp_ratio=self.dim_feedforward / self.d_model,
                )
                for _ in range(self.num_layers)
            ]
        )
        # self.transformer = nn.Sequential(
        #     *[
        #         CrossTransformerEncoderBlock(
        #             dim=self.d_model,
        #             num_heads=self.nhead,
        #             mlp_ratio=self.dim_feedforward / self.d_model,
        #         )
        #         for _ in range(self.num_layers)
        #     ]
        # )
        # encoder_layer = nn.TransformerEncoderLayer(
        #     d_model=self.d_model,
        #     nhead=self.nhead,
        #     dim_feedforward=self.dim_feedforward,
        # )
        # self.transformer = nn.TransformerEncoder(encoder_layer, self.num_layers)
        self.output_mlp = nn.Sequential(
            nn.Linear(self.d_model, self.d_model // 2),
            nn.GELU(),
            nn.Linear(self.d_model // 2, self.d_model // 4),
            nn.GELU(),
            nn.Linear(self.d_model // 4, 2),
            # nn.LogSoftmax(dim=-1),
        )

    def forward(self, x, image_policy) -> torch.FloatTensor:
        predict = []
        # image = []
        # # instruct = []
        # image_len = (
        #     input_embeds_indices[0]["image"][1] - input_embeds_indices[0]["image"][0]
        # )
        # for index in range(x.shape[0]):
        #     cur_input_embeds_indices = input_embeds_indices[index]
        #     cur_image = x[
        #         index,
        #         cur_input_embeds_indices["image"][0] : cur_input_embeds_indices[
        #             "image"
        #         ][1],
        #         :,
        #     ]
        #     # cur_instruct = x[
        #     #     index,
        #     #     cur_input_embeds_indices["instruct"][0] : cur_input_embeds_indices[
        #     #         "instruct"
        #     #     ][1],
        #     #     :,
        #     # ]
        #     image.append(cur_image)
        #     # instruct.append(cur_instruct)

        # new_image_input = []
        # max_image_len = max(cur_image.shape[0] for cur_image in image)
        # for cur_image in image:
        #     cur_len = cur_image.shape[0]
        #     new_image_input.append(
        #         torch.cat(
        #             [
        #                 cur_image,
        #                 torch.zeros(
        #                     (max_image_len - cur_len, cur_image.shape[1]),
        #                     dtype=cur_image.dtype,
        #                     device=cur_image.device,
        #                 ),
        #             ],
        #             dim=0,
        #         )
        #     )

        # new_image_x = torch.stack(new_image_input, dim=0)
        # new_image_x = self.down_mlp(new_image_x)

        # new_instruct_input = []
        # max_instruct_len = max(cur_instruct.shape[0] for cur_instruct in instruct)
        # for cur_instruct in instruct:
        #     cur_len = cur_instruct.shape[0]
        #     new_instruct_input.append(
        #         torch.cat(
        #             [
        #                 cur_instruct,
        #                 torch.zeros(
        #                     (max_instruct_len - cur_len, cur_instruct.shape[1]),
        #                     dtype=cur_instruct.dtype,
        #                     device=cur_instruct.device,
        #                 ),
        #             ],
        #             dim=0,
        #         )
        #     )

        # new_instruct_x = torch.stack(new_instruct_input, dim=0)
        # new_instruct_x = self.down_mlp(new_instruct_x)

        # # q: image, kv: instruct
        # new_x = self.transformer(
        #     (new_image_x * image_policy, new_instruct_x, new_instruct_x)
        # )[0]

        new_image_x = self.down_mlp(x)
        new_x = self.transformer(new_image_x * image_policy)
        B, N, C = new_x.size()
        local_x = new_x[:, :, : C // 2]
        global_x = (new_x[:, :, C // 2 :] * image_policy).sum(
            dim=1, keepdim=True
        ) / torch.sum(image_policy, dim=1, keepdim=True)
        new_x = torch.cat([local_x, global_x.expand(B, N, C // 2)], dim=-1)
        predict = self.output_mlp(new_x)
        return predict


# class OutputTextPredictor(nn.Module):
#     def __init__(
#         self,
#         input_dim=4096,
#         d_model=512,
#         nhead=8,
#         dim_feedforward=2048,
#         num_layers=2,
#     ):
#         super().__init__()
#         self.input_dim = input_dim
#         self.d_model = d_model
#         self.nhead = nhead
#         self.dim_feedforward = dim_feedforward
#         self.num_layers = num_layers

#         self.down_mlp = nn.Sequential(
#             nn.LayerNorm(self.input_dim),
#             nn.Linear(self.input_dim, self.d_model),
#             nn.GELU(),
#         )
#         self.transformer = nn.Sequential(
#             *[
#                 CrossTransformerEncoderBlock(
#                     dim=self.d_model,
#                     num_heads=self.nhead,
#                     mlp_ratio=self.dim_feedforward / self.d_model,
#                 )
#                 for _ in range(self.num_layers)
#             ]
#         )
#         self.output_mlp = nn.Sequential(
#             nn.Linear(self.d_model, self.d_model // 2),
#             nn.GELU(),
#             nn.Linear(self.d_model // 2, self.d_model // 4),
#             nn.GELU(),
#             nn.Linear(self.d_model // 4, 2),
#         )

#     def forward(self, x, pre_x) -> torch.FloatTensor:
#         # q: x, kv: pre_x
#         new_x = self.down_mlp(x)
#         new_pre_x = self.down_mlp(pre_x)
#         new_x = self.transformer((new_x, new_pre_x, new_pre_x))[0]
#         predict = self.output_mlp(new_x)
#         return predict


class TextPredictor(nn.Module):
    def __init__(
        self,
        input_dim=4096,
        d_model=512,
        nhead=8,
        dim_feedforward=2048,
        num_layers=2,
    ):
        super().__init__()
        self.input_dim = input_dim
        self.d_model = d_model
        # self.nhead = nhead
        # self.dim_feedforward = dim_feedforward
        # self.num_layers = num_layers

        # self.down_mlp = nn.Sequential(
        #     nn.LayerNorm(self.input_dim),
        #     nn.Linear(self.input_dim, self.d_model),
        #     nn.GELU(),
        # )
        # self.transformer = nn.Sequential(
        #     *[
        #         CrossTransformerEncoderBlock(
        #             dim=self.d_model,
        #             num_heads=self.nhead,
        #             mlp_ratio=self.dim_feedforward / self.d_model,
        #         )
        #         for _ in range(self.num_layers)
        #     ]
        # )
        self.output_mlp = nn.Sequential(
            nn.LayerNorm(self.input_dim),
            nn.Linear(self.input_dim, self.d_model),
            nn.GELU(),
            nn.Linear(self.d_model, self.d_model // 2),
            nn.GELU(),
            nn.Linear(self.d_model // 2, self.d_model // 4),
            nn.GELU(),
            nn.Linear(self.d_model // 4, 2),
        )

    def forward(self, x) -> torch.FloatTensor:
        # q: x, kv: pre_x
        # new_x = self.down_mlp(x)
        # new_pre_x = self.down_mlp(pre_x)
        # new_x = self.transformer((new_x, new_pre_x, new_pre_x))[0]
        predict = self.output_mlp(x)
        return predict


def weight_merging(select_token, unselect_token):
    pass


def ste_argmax(logits: torch.Tensor, dim: int = -1):
    y_soft = logits
    index = y_soft.max(dim, keepdim=True)[1]
    y_hard = torch.zeros_like(
        logits, memory_format=torch.legacy_contiguous_format
    ).scatter_(dim, index, 1.0)
    ret = y_hard - y_soft.detach() + y_soft
    return ret


def ste_topk(logits: torch.Tensor, k: int, dim: int = -1):
    y_soft = logits
    index = y_soft.topk(k=k, dim=dim, largest=True, sorted=False)[1]
    y_hard = torch.zeros_like(
        logits, memory_format=torch.legacy_contiguous_format
    ).scatter_(dim, index, 1.0)
    ret = y_hard - y_soft.detach() + y_soft
    return ret


# def batch_index_select(x, idx):
#     if len(x.size()) == 3:
#         B, N, C = x.size()
#         N_new = idx.size(1)
#         offset = torch.arange(B, dtype=torch.long, device=x.device).view(B, 1) * N
#         idx = idx + offset
#         out = x.reshape(B * N, C)[idx.reshape(-1)].reshape(B, N_new, C)
#         return out
#     elif len(x.size()) == 2:
#         B, N = x.size()
#         N_new = idx.size(1)
#         offset = torch.arange(B, dtype=torch.long, device=x.device).view(B, 1) * N
#         idx = idx + offset
#         out = x.reshape(B * N)[idx.reshape(-1)].reshape(B, N_new)
#         return out
#     else:
#         raise NotImplementedError


@dataclass
class DynamicModelOutputWithPast(ModelOutput):
    """
    Base class for model's outputs that may also contain a past key/values (to speed up sequential decoding).

    Args:
        last_hidden_state (`torch.FloatTensor` of shape `(batch_size, sequence_length, hidden_size)`):
            Sequence of hidden-states at the output of the last layer of the model.

            If `past_key_values` is used only the last hidden-state of the sequences of shape `(batch_size, 1,
            hidden_size)` is output.
        past_key_values (`tuple(tuple(torch.FloatTensor))`, *optional*, returned when `use_cache=True` is passed or when `config.use_cache=True`):
            Tuple of `tuple(torch.FloatTensor)` of length `config.n_layers`, with each tuple having 2 tensors of shape
            `(batch_size, num_heads, sequence_length, embed_size_per_head)`) and optionally if
            `config.is_encoder_decoder=True` 2 additional tensors of shape `(batch_size, num_heads,
            encoder_sequence_length, embed_size_per_head)`.

            Contains pre-computed hidden-states (key and values in the self-attention blocks and optionally if
            `config.is_encoder_decoder=True` in the cross-attention blocks) that can be used (see `past_key_values`
            input) to speed up sequential decoding.
        hidden_states (`tuple(torch.FloatTensor)`, *optional*, returned when `output_hidden_states=True` is passed or when `config.output_hidden_states=True`):
            Tuple of `torch.FloatTensor` (one for the output of the embeddings, if the model has an embedding layer, +
            one for the output of each layer) of shape `(batch_size, sequence_length, hidden_size)`.

            Hidden-states of the model at the output of each layer plus the optional initial embedding outputs.
        attentions (`tuple(torch.FloatTensor)`, *optional*, returned when `output_attentions=True` is passed or when `config.output_attentions=True`):
            Tuple of `torch.FloatTensor` (one for each layer) of shape `(batch_size, num_heads, sequence_length,
            sequence_length)`.

            Attentions weights after the attention softmax, used to compute the weighted average in the self-attention
            heads.
    """

    last_hidden_state: torch.FloatTensor = None
    past_key_values: Optional[
        Tuple[Tuple[Tuple[torch.FloatTensor]], torch.IntTensor]
    ] = None
    # true_cache_length: Optional[List[torch.FloatTensor]] = None
    hidden_states: Optional[Tuple[torch.FloatTensor]] = None
    attentions: Optional[Tuple[torch.FloatTensor]] = None
    image_masks: Optional[List[torch.FloatTensor]] = None
    output_text_masks_batch_list: Optional[List[List[torch.FloatTensor]]] = None
    instruct_masks_batch_list: Optional[List[List[torch.FloatTensor]]] = None
    image_score_predictor_logits: Optional[List[torch.FloatTensor]] = None


LLAMA_START_DOCSTRING = r"""
    This model inherits from [`PreTrainedModel`]. Check the superclass documentation for the generic methods the
    library implements for all its model (such as downloading or saving, resizing the input embeddings, pruning heads
    etc.)

    This model is also a PyTorch [torch.nn.Module](https://pytorch.org/docs/stable/nn.html#torch.nn.Module) subclass.
    Use it as a regular PyTorch Module and refer to the PyTorch documentation for all matter related to general usage
    and behavior.

    Parameters:
        config ([`LlamaConfig`]):
            Model configuration class with all the parameters of the model. Initializing with a config file does not
            load the weights associated with the model, only the configuration. Check out the
            [`~PreTrainedModel.from_pretrained`] method to load the model weights.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class DynamicLlamaPreTrainedModel(PreTrainedModel):
    config_class = LlamaConfig
    base_model_prefix = "model"
    supports_gradient_checkpointing = True
    # _no_split_modules = ["LlamaDecoderLayer"]
    _no_split_modules = ["DynamicLlamaDecoderLayer"]
    _skip_keys_device_placement = "past_key_values"
    _supports_flash_attn_2 = True
    _supports_sdpa = True
    _supports_cache_class = True

    def _init_weights(self, module):
        std = self.config.initializer_range
        if isinstance(module, nn.Linear):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.bias is not None:
                module.bias.data.zero_()
        elif isinstance(module, nn.Embedding):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.padding_idx is not None:
                module.weight.data[module.padding_idx].zero_()


LLAMA_INPUTS_DOCSTRING = r"""
    Args:
        input_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`):
            Indices of input sequence tokens in the vocabulary. Padding will be ignored by default should you provide
            it.

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            [What are input IDs?](../glossary#input-ids)
        attention_mask (`torch.Tensor` of shape `(batch_size, sequence_length)`, *optional*):
            Mask to avoid performing attention on padding token indices. Mask values selected in `[0, 1]`:

            - 1 for tokens that are **not masked**,
            - 0 for tokens that are **masked**.

            [What are attention masks?](../glossary#attention-mask)

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            If `past_key_values` is used, optionally only the last `input_ids` have to be input (see
            `past_key_values`).

            If you want to change padding behavior, you should read [`modeling_opt._prepare_decoder_attention_mask`]
            and modify to your needs. See diagram 1 in [the paper](https://arxiv.org/abs/1910.13461) for more
            information on the default strategy.

            - 1 indicates the head is **not masked**,
            - 0 indicates the head is **masked**.
        position_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
            Indices of positions of each input sequence tokens in the position embeddings. Selected in the range `[0,
            config.n_positions - 1]`.

            [What are position IDs?](../glossary#position-ids)
        past_key_values (`Cache` or `tuple(tuple(torch.FloatTensor))`, *optional*):
            Pre-computed hidden-states (key and values in the self-attention blocks and in the cross-attention
            blocks) that can be used to speed up sequential decoding. This typically consists in the `past_key_values`
            returned by the model at a previous stage of decoding, when `use_cache=True` or `config.use_cache=True`.

            Two formats are allowed:
            - a [`~cache_utils.Cache`] instance;
            - Tuple of `tuple(torch.FloatTensor)` of length `config.n_layers`, with each tuple having 2 tensors of
            shape `(batch_size, num_heads, sequence_length, embed_size_per_head)`). This is also known as the legacy
            cache format.

            The model will output the same cache format that is fed as input. If no `past_key_values` are passed, the
            legacy cache format will be returned.

            If `past_key_values` are used, the user can optionally input only the last `input_ids` (those that don't
            have their past key value states given to this model) of shape `(batch_size, 1)` instead of all `input_ids`
            of shape `(batch_size, sequence_length)`.
        inputs_embeds (`torch.FloatTensor` of shape `(batch_size, sequence_length, hidden_size)`, *optional*):
            Optionally, instead of passing `input_ids` you can choose to directly pass an embedded representation. This
            is useful if you want more control over how to convert `input_ids` indices into associated vectors than the
            model's internal embedding lookup matrix.
        use_cache (`bool`, *optional*):
            If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding (see
            `past_key_values`).
        output_attentions (`bool`, *optional*):
            Whether or not to return the attentions tensors of all attention layers. See `attentions` under returned
            tensors for more detail.
        output_hidden_states (`bool`, *optional*):
            Whether or not to return the hidden states of all layers. See `hidden_states` under returned tensors for
            more detail.
        return_dict (`bool`, *optional*):
            Whether or not to return a [`~utils.ModelOutput`] instead of a plain tuple.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class DynamicLlamaModel(DynamicLlamaPreTrainedModel):
    """
    Transformer decoder consisting of *config.num_hidden_layers* layers. Each layer is a [`LlamaDecoderLayer`]

    Args:
        config: LlamaConfig
    """

    def __init__(self, config: LlamaConfig):
        super().__init__(config)
        self.padding_idx = config.pad_token_id
        self.vocab_size = config.vocab_size

        self.embed_tokens = nn.Embedding(
            config.vocab_size, config.hidden_size, self.padding_idx
        )
        self.layers = nn.ModuleList(
            [
                DynamicLlamaDecoderLayer(config, layer_idx)
                for layer_idx in range(config.num_hidden_layers)
            ]
        )
        self._use_sdpa = config._attn_implementation == "sdpa"
        self._use_flash_attention_2 = config._attn_implementation == "flash_attention_2"
        self.norm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

        self.gradient_checkpointing = False

        # self.sparse_config = config.sparse_config
        if config.sparse_config["use_vision_predictor"]:
            self.image_score_predictor = VisionPredictor(
                input_dim=config.hidden_size,
                d_model=config.sparse_config["d_model"],
                nhead=config.sparse_config["nhead"],
                dim_feedforward=config.sparse_config["dim_feedforward"],
                num_layers=config.sparse_config["num_layers"],
            )
        if config.sparse_config["use_text_predictor"]:
            if config.sparse_config["use_output_text_predictor"]:
                self.output_text_score_predictor = TextPredictor(
                    input_dim=config.hidden_size,
                    d_model=config.sparse_config["d_model"],
                    nhead=config.sparse_config["nhead"],
                    dim_feedforward=config.sparse_config["dim_feedforward"],
                    num_layers=config.sparse_config["num_layers"],
                )
            if config.sparse_config["use_instruct_predictor"]:
                self.instruct_score_predictor = TextPredictor(
                    input_dim=config.hidden_size,
                    d_model=config.sparse_config["d_model"],
                    nhead=config.sparse_config["nhead"],
                    dim_feedforward=config.sparse_config["dim_feedforward"],
                    num_layers=config.sparse_config["num_layers"],
                )
            # self.text_score_predictor = TextPredictor(
            #     input_dim=config.hidden_size,
            #     d_model=config.sparse_config["d_model"],
            #     nhead=config.sparse_config["nhead"],
            #     dim_feedforward=config.sparse_config["dim_feedforward"],
            #     num_layers=config.sparse_config["num_layers"],
            # )

        self.gumbel_tau = 1.0
        # self.pre_answer = None

        # for no kv cache
        self.answer_indice = None

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.embed_tokens

    def set_input_embeddings(self, value):
        self.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        input_embeds_indices: Optional[List[dict]] = None,
    ) -> Union[Tuple, BaseModelOutputWithPast]:
        output_attentions = (
            output_attentions
            if output_attentions is not None
            else self.config.output_attentions
        )
        output_hidden_states = (
            output_hidden_states
            if output_hidden_states is not None
            else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache

        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        # retrieve input_ids and inputs_embeds
        if input_ids is not None and inputs_embeds is not None:
            raise ValueError(
                "You cannot specify both input_ids and inputs_embeds at the same time"
            )
        elif input_ids is not None:
            batch_size, seq_length = input_ids.shape[:2]
        elif inputs_embeds is not None:
            batch_size, seq_length = inputs_embeds.shape[:2]
        else:
            raise ValueError("You have to specify either input_ids or inputs_embeds")

        if self.gradient_checkpointing and self.training:
            if use_cache:
                logger.warning_once(
                    "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`..."
                )
                use_cache = False

        past_key_values_length = 0
        if use_cache:
            use_legacy_cache = not isinstance(past_key_values, Cache)
            if use_legacy_cache:
                # ----------------------------------------------------------#
                # past_key_values = DynamicCache.from_legacy_cache(past_key_values)
                past_key_values = DynamicCachePlus.from_legacy_cache(past_key_values)
                # ----------------------------------------------------------#
            past_key_values_length = past_key_values.get_usable_length(seq_length)

        if position_ids is None:
            device = input_ids.device if input_ids is not None else inputs_embeds.device
            position_ids = torch.arange(
                past_key_values_length,
                seq_length + past_key_values_length,
                dtype=torch.long,
                device=device,
            )
            position_ids = position_ids.unsqueeze(0)

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)

        # if self._use_flash_attention_2:
        #     # 2d mask is passed through the layers
        #     attention_mask = (
        #         attention_mask
        #         if (attention_mask is not None and 0 in attention_mask)
        #         else None
        #     )
        # elif self._use_sdpa and not output_attentions:
        #     # output_attentions=True can not be supported when using SDPA, and we fall back on
        #     # the manual implementation that requires a 4D causal mask in all cases.
        #     attention_mask = _prepare_4d_causal_attention_mask_for_sdpa(
        #         attention_mask,
        #         (batch_size, seq_length),
        #         inputs_embeds,
        #         past_key_values_length,
        #     )
        # else:
        #     # 4d mask is passed through the layers
        #     attention_mask = _prepare_4d_causal_attention_mask(
        #         attention_mask,
        #         (batch_size, seq_length),
        #         inputs_embeds,
        #         past_key_values_length,
        #     )

        # embed positions
        hidden_states = inputs_embeds

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None
        next_decoder_cache = None

        # ----------------------------------------------------------#
        B = hidden_states.shape[0]
        init_n = hidden_states.shape[1]
        policy = None
        image_masks = []
        output_text_masks_batch_list = []
        instruct_masks_batch_list = []
        image_score_predictor_logits = []
        text_decision = None  # for eval kv cache
        if (
            self.config.sparse_config["use_vision_predictor"]
            and input_embeds_indices is not None
            and hidden_states.shape[0] == len(input_embeds_indices)
        ):
            # all images are same size
            init_image_n = (
                input_embeds_indices[0]["image"][1]
                - input_embeds_indices[0]["image"][0]
            )

            # prev_decision = torch.ones(
            #     B, init_n, 1, dtype=hidden_states.dtype, device=hidden_states.device
            # )
            image_prev_decision = torch.ones(
                B,
                init_image_n,
                1,
                dtype=hidden_states.dtype,
                device=hidden_states.device,
            )

            # if (
            #     self.config.sparse_config["use_text_predictor"]
            #     and input_embeds_indices is not None
            #     and hidden_states.shape[0] == len(input_embeds_indices)
            # ):
            #     # padding answer
            #     if self.config.sparse_config["use_output_text_predictor"]:
            #         init_max_output_text_n = max(
            #             input_embeds_indices[b]["answer"][1]
            #             - input_embeds_indices[b]["answer"][0]
            #             for b in range(B)
            #         )
            #         output_text_prev_decision = torch.ones(
            #             B,
            #             init_max_output_text_n,
            #             1,
            #             dtype=hidden_states.dtype,
            #             device=hidden_states.device,
            #         )
            #         for b in range(B):
            #             output_text_prev_decision[
            #                 b,
            #                 input_embeds_indices[b]["answer"][1]
            #                 - input_embeds_indices[b]["answer"][0] :,
            #                 :,
            #             ] = 0

            # # padding instruct
            # if self.config.sparse_config["use_instruct_predictor"]:
            #     init_max_instruct_n = max(
            #         input_embeds_indices[b]["last_instruct"][1]
            #         - input_embeds_indices[b]["last_instruct"][0]
            #         for b in range(B)
            #     )
            #     instruct_prev_decision = torch.ones(
            #         B,
            #         init_max_instruct_n,
            #         1,
            #         dtype=hidden_states.dtype,
            #         device=hidden_states.device,
            #     )
            #     for b in range(B):
            #         instruct_prev_decision[
            #             b,
            #             input_embeds_indices[b]["last_instruct"][1]
            #             - input_embeds_indices[b]["last_instruct"][0] :,
            #             :,
            #         ] = 0
        # ----------------------------------------------------------#

        for i, decoder_layer in enumerate(self.layers):
            # ----------------------------------------------------------#
            # dynamic attention mask
            if use_cache:
                past_key_values_length = past_key_values.get_usable_length(
                    seq_length, i
                )
            if self._use_flash_attention_2:
                # 2d mask is passed through the layers
                attention_mask = (
                    attention_mask
                    if (attention_mask is not None and 0 in attention_mask)
                    else None
                )
            elif self._use_sdpa and not output_attentions:
                # output_attentions=True can not be supported when using SDPA, and we fall back on
                # the manual implementation that requires a 4D causal mask in all cases.
                attention_mask = _prepare_4d_causal_attention_mask_for_sdpa(
                    attention_mask,
                    (batch_size, seq_length),
                    inputs_embeds,
                    past_key_values_length,
                )
            else:
                # 4d mask is passed through the layers
                attention_mask = _prepare_4d_causal_attention_mask(
                    attention_mask,
                    (batch_size, seq_length),
                    inputs_embeds,
                    past_key_values_length,
                )
            # ----------------------------------------------------------#

            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            # ----------------------------------------------------------#
            # only for image
            if (
                self.config.sparse_config["use_vision_predictor"]
                and i == self.config.sparse_config["sparse_layer"]
                and input_embeds_indices is not None
                and hidden_states.shape[0] == len(input_embeds_indices)
            ):
                image = []
                # instruct = []
                # image_len = (
                #     input_embeds_indices[0]["image"][1] - input_embeds_indices[0]["image"][0]
                # )
                for index in range(hidden_states.shape[0]):
                    cur_input_embeds_indices = input_embeds_indices[index]
                    cur_image = hidden_states[
                        index,
                        cur_input_embeds_indices["image"][0] : cur_input_embeds_indices[
                            "image"
                        ][1],
                        :,
                    ]
                    # cur_instruct = x[
                    #     index,
                    #     cur_input_embeds_indices["instruct"][0] : cur_input_embeds_indices[
                    #         "instruct"
                    #     ][1],
                    #     :,
                    # ]
                    image.append(cur_image)
                    # instruct.append(cur_instruct)

                new_image_input = []
                max_image_len = max(cur_image.shape[0] for cur_image in image)
                for cur_image in image:
                    cur_len = cur_image.shape[0]
                    new_image_input.append(
                        torch.cat(
                            [
                                cur_image,
                                torch.zeros(
                                    (max_image_len - cur_len, cur_image.shape[1]),
                                    dtype=cur_image.dtype,
                                    device=cur_image.device,
                                ),
                            ],
                            dim=0,
                        )
                    )

                new_image_x = torch.stack(new_image_input, dim=0)

                image_score_predictor_logit = self.image_score_predictor(
                    new_image_x, image_prev_decision
                ).reshape(B, -1, 2)
                image_pred_score = F.log_softmax(image_score_predictor_logit, dim=-1)
                if self.training:
                    image_hard_keep_decision = (
                        F.gumbel_softmax(
                            image_pred_score, tau=self.gumbel_tau, hard=True
                        )[:, :, 0:1]
                        * image_prev_decision
                    )
                    image_masks.append(
                        image_hard_keep_decision.reshape(B, init_image_n)
                    )

                    left_policy = torch.ones(
                        B,
                        input_embeds_indices[0]["image"][0],
                        1,
                        dtype=hidden_states.dtype,
                        device=hidden_states.device,
                    )
                    right_policy = torch.ones(
                        B,
                        init_n - input_embeds_indices[0]["image"][1],
                        1,
                        dtype=hidden_states.dtype,
                        device=hidden_states.device,
                    )
                    policy = torch.cat(
                        [left_policy, image_hard_keep_decision, right_policy], dim=1
                    )
                    image_prev_decision = image_hard_keep_decision

                    if self.config.sparse_config["maskclip"] is not None:
                        image_score_predictor_logits.append(image_score_predictor_logit)
                else:
                    image_score = image_pred_score[:, :, 0]
                    num_keep_node = int(
                        init_image_n * self.config.sparse_config["vision_keep_rate"]
                    )
                    keep_index, _ = torch.sort(
                        torch.argsort(image_score, dim=1, descending=True)[
                            :, :num_keep_node
                        ],
                        dim=1,
                        descending=False,
                    )
                    # unkeep_index, _ = torch.sort(
                    #     torch.argsort(image_score, dim=1, descending=False)[
                    #         :, : init_image_n - num_keep_node
                    #     ],
                    #     dim=1,
                    #     descending=False,
                    # )

                    image_hidden_states = hidden_states[
                        :,
                        input_embeds_indices[0]["image"][0] : input_embeds_indices[0][
                            "image"
                        ][1],
                        :,
                    ]
                    left_hidden_states = hidden_states[
                        :,
                        : input_embeds_indices[0]["image"][0],
                        :,
                    ]
                    right_hidden_states = hidden_states[
                        :,
                        input_embeds_indices[0]["image"][1] :,
                        :,
                    ]

                    keep_image_hidden_states = image_hidden_states.gather(
                        dim=1,
                        index=keep_index[..., None].expand(
                            image_hidden_states.shape[0],
                            num_keep_node,
                            image_hidden_states.shape[2],
                        ),
                    )

                    hidden_states = torch.cat(
                        [
                            left_hidden_states,
                            keep_image_hidden_states,
                            right_hidden_states,
                        ],
                        dim=1,
                    )
                    image_prev_decision = image_prev_decision.gather(
                        dim=1,
                        index=keep_index[..., None].expand(
                            image_prev_decision.shape[0],
                            num_keep_node,
                            image_prev_decision.shape[2],
                        ),
                    )
                    policy = None

                    # for position
                    keep_index_for_image_position = (
                        keep_index + input_embeds_indices[0]["image"][0]
                    ).to(dtype=position_ids.dtype, device=position_ids.device)
                    keep_index_for_left_position = (
                        torch.arange(0, input_embeds_indices[0]["image"][0])
                        .repeat(B, 1)
                        .to(dtype=position_ids.dtype, device=position_ids.device)
                    )
                    keep_index_for_right_position = (
                        torch.arange(input_embeds_indices[0]["image"][1], init_n)
                        .repeat(B, 1)
                        .to(dtype=position_ids.dtype, device=position_ids.device)
                    )
                    position_ids = torch.cat(
                        [
                            keep_index_for_left_position,
                            keep_index_for_image_position,
                            keep_index_for_right_position,
                        ],
                        dim=1,
                    )

                    # for input_embeds_indices
                    num_unkeep_node = init_image_n - num_keep_node
                    for input_embeds_indice in input_embeds_indices:
                        input_embeds_indice["image"][1] -= num_unkeep_node
                        input_embeds_indice["instruct"][0] -= num_unkeep_node
                        input_embeds_indice["instruct"][1] -= num_unkeep_node
                        input_embeds_indice["last_instruct"][0] -= num_unkeep_node
                        input_embeds_indice["last_instruct"][1] -= num_unkeep_node
                        input_embeds_indice["answer"][0] -= num_unkeep_node
                        input_embeds_indice["answer"][1] -= num_unkeep_node
            if (
                self.config.sparse_config["use_text_predictor"]
                and i == self.config.sparse_config["sparse_layer"]
            ):
                if (
                    input_embeds_indices is not None
                    and hidden_states.shape[0] == len(input_embeds_indices)
                    and self.training
                ):

                    # for output token
                    if self.config.sparse_config["use_output_text_predictor"]:

                        # padding answer
                        init_max_output_text_n = max(
                            input_embeds_indices[b]["answer"][1]
                            - input_embeds_indices[b]["answer"][0]
                            for b in range(B)
                        )
                        output_text_prev_decision = torch.ones(
                            B,
                            init_max_output_text_n,
                            1,
                            dtype=hidden_states.dtype,
                            device=hidden_states.device,
                        )
                        for b in range(B):
                            output_text_prev_decision[
                                b,
                                input_embeds_indices[b]["answer"][1]
                                - input_embeds_indices[b]["answer"][0] :,
                                :,
                            ] = 0

                        answer_batch_list = [
                            hidden_states[b][
                                input_embeds_indices[b]["answer"][
                                    0
                                ] : input_embeds_indices[b]["answer"][1],
                                :,
                            ]
                            for b in range(B)
                        ]

                        padded_answer = []
                        max_answer_len = max(
                            input_embeds_indices[b]["answer"][1]
                            - input_embeds_indices[b]["answer"][0]
                            for b in range(B)
                        )
                        for cur_answer in answer_batch_list:
                            cur_len = cur_answer.shape[0]
                            padded_answer.append(
                                torch.cat(
                                    [
                                        cur_answer,
                                        torch.zeros(
                                            (
                                                max_answer_len - cur_len,
                                                cur_answer.shape[1],
                                            ),
                                            dtype=cur_answer.dtype,
                                            device=cur_answer.device,
                                        ),
                                    ],
                                    dim=0,
                                )
                            )

                        padded_answer = torch.stack(padded_answer, dim=0)

                        # padded_output_text_score_predictor_logit = (
                        #     self.output_text_score_predictor(padded_answer).reshape(
                        #         B, -1, 2
                        #     )
                        # )
                        padded_output_text_score_predictor_logit = (
                            self.output_text_score_predictor(padded_answer).reshape(
                                B, -1, 2
                            )
                        )
                        padded_output_text_pred_score = F.log_softmax(
                            padded_output_text_score_predictor_logit, dim=-1
                        )

                        output_text_hard_keep_decision = (
                            F.gumbel_softmax(
                                padded_output_text_pred_score,
                                tau=self.gumbel_tau,
                                hard=True,
                            )[:, :, 0:1]
                            * output_text_prev_decision
                        )

                        # for stable training
                        for b in range(B):
                            if (
                                input_embeds_indices[b]["answer"][1]
                                - input_embeds_indices[b]["answer"][0]
                                < self.config.sparse_config[
                                    "output_text_len_for_training"
                                ]
                            ):
                                output_text_hard_keep_decision[b][
                                    : input_embeds_indices[b]["answer"][1]
                                    - input_embeds_indices[b]["answer"][0],
                                    :,
                                ] = torch.ones_like(
                                    output_text_hard_keep_decision[b][
                                        : input_embeds_indices[b]["answer"][1]
                                        - input_embeds_indices[b]["answer"][0],
                                        :,
                                    ],
                                    dtype=output_text_hard_keep_decision.dtype,
                                    device=output_text_hard_keep_decision.device,
                                )

                        output_text_masks_batch_list.append(
                            [
                                output_text_hard_keep_decision[b][
                                    : input_embeds_indices[b]["answer"][1]
                                    - input_embeds_indices[b]["answer"][0],
                                    :,
                                ].reshape(
                                    input_embeds_indices[b]["answer"][1]
                                    - input_embeds_indices[b]["answer"][0]
                                )
                                for b in range(B)
                            ]
                        )

                        if policy is not None:
                            for b in range(B):
                                policy[b][
                                    input_embeds_indices[b]["answer"][
                                        0
                                    ] : input_embeds_indices[b]["answer"][1]
                                ] = output_text_hard_keep_decision[b][
                                    : input_embeds_indices[b]["answer"][1]
                                    - input_embeds_indices[b]["answer"][0],
                                    :,
                                ]
                        else:
                            pass

                    # for instruct token
                    if self.config.sparse_config["use_instruct_predictor"]:
                        # padding instruct
                        init_max_instruct_n = max(
                            input_embeds_indices[b]["last_instruct"][1]
                            - input_embeds_indices[b]["last_instruct"][0]
                            for b in range(B)
                        )
                        instruct_prev_decision = torch.ones(
                            B,
                            init_max_instruct_n,
                            1,
                            dtype=hidden_states.dtype,
                            device=hidden_states.device,
                        )
                        for b in range(B):
                            instruct_prev_decision[
                                b,
                                input_embeds_indices[b]["last_instruct"][1]
                                - input_embeds_indices[b]["last_instruct"][0] :,
                                :,
                            ] = 0

                        instruct_batch_list = [
                            hidden_states[b][
                                input_embeds_indices[b]["last_instruct"][
                                    0
                                ] : input_embeds_indices[b]["last_instruct"][1],
                                :,
                            ]
                            for b in range(B)
                        ]

                        padded_instruct = []
                        max_instruct_len = max(
                            input_embeds_indices[b]["last_instruct"][1]
                            - input_embeds_indices[b]["last_instruct"][0]
                            for b in range(B)
                        )
                        for cur_instruct in instruct_batch_list:
                            cur_len = cur_instruct.shape[0]
                            padded_instruct.append(
                                torch.cat(
                                    [
                                        cur_instruct,
                                        torch.zeros(
                                            (
                                                max_instruct_len - cur_len,
                                                cur_instruct.shape[1],
                                            ),
                                            dtype=cur_instruct.dtype,
                                            device=cur_instruct.device,
                                        ),
                                    ],
                                    dim=0,
                                )
                            )

                        padded_instruct = torch.stack(padded_instruct, dim=0)

                        # padded_instruct_score_predictor_logit = (
                        #     self.instruct_score_predictor(padded_instruct).reshape(
                        #         B, -1, 2
                        #     )
                        # )
                        padded_instruct_score_predictor_logit = (
                            self.instruct_score_predictor(padded_instruct).reshape(
                                B, -1, 2
                            )
                        )
                        padded_instruct_pred_score = F.log_softmax(
                            padded_instruct_score_predictor_logit, dim=-1
                        )

                        instruct_hard_keep_decision = (
                            F.gumbel_softmax(
                                padded_instruct_pred_score,
                                tau=self.gumbel_tau,
                                hard=True,
                            )[:, :, 0:1]
                            * instruct_prev_decision
                        )

                        # for stable training
                        for b in range(B):
                            if (
                                input_embeds_indices[b]["last_instruct"][1]
                                - input_embeds_indices[b]["last_instruct"][0]
                                < self.config.sparse_config["instruct_len_for_training"]
                            ):
                                instruct_hard_keep_decision[b][
                                    : input_embeds_indices[b]["last_instruct"][1]
                                    - input_embeds_indices[b]["last_instruct"][0],
                                    :,
                                ] = torch.ones_like(
                                    instruct_hard_keep_decision[b][
                                        : input_embeds_indices[b]["last_instruct"][1]
                                        - input_embeds_indices[b]["last_instruct"][0],
                                        :,
                                    ],
                                    dtype=instruct_hard_keep_decision.dtype,
                                    device=instruct_hard_keep_decision.device,
                                )

                        instruct_masks_batch_list.append(
                            [
                                instruct_hard_keep_decision[b][
                                    : input_embeds_indices[b]["last_instruct"][1]
                                    - input_embeds_indices[b]["last_instruct"][0],
                                    :,
                                ].reshape(
                                    input_embeds_indices[b]["last_instruct"][1]
                                    - input_embeds_indices[b]["last_instruct"][0]
                                )
                                for b in range(B)
                            ]
                        )

                        if policy is not None:
                            for b in range(B):
                                policy[b][
                                    input_embeds_indices[b]["last_instruct"][
                                        0
                                    ] : input_embeds_indices[b]["last_instruct"][1]
                                ] = instruct_hard_keep_decision[b][
                                    : input_embeds_indices[b]["last_instruct"][1]
                                    - input_embeds_indices[b]["last_instruct"][0],
                                    :,
                                ]
                        else:
                            pass

                elif (
                    not self.training
                    and not past_key_values_length
                    # and self.pre_answer is None
                    and input_embeds_indices is not None
                    and hidden_states.shape[0] == len(input_embeds_indices)
                    and self.config.sparse_config["use_instruct_predictor"]
                ):  # pre fill stage, first instruct
                    assert B == 1, "Using text predictor must keep the batch size to 1"
                    instruct_hidden_states = hidden_states[
                        :,
                        input_embeds_indices[0]["last_instruct"][
                            0
                        ] : input_embeds_indices[0]["last_instruct"][1]
                        - 1,
                        :,
                    ]
                    instruct_score_predictor_logit = self.instruct_score_predictor(
                        instruct_hidden_states
                    ).reshape(B, -1, 2)

                    indices = torch.where(
                        instruct_score_predictor_logit[0, :, 0]
                        > instruct_score_predictor_logit[0, :, 1]
                    )
                    keep_index = indices[0].unsqueeze(0)

                    left_hidden_states = hidden_states[
                        :,
                        : input_embeds_indices[0]["last_instruct"][0],
                        :,
                    ]
                    right_hidden_states = hidden_states[
                        :,
                        input_embeds_indices[0]["last_instruct"][1] - 1 :,
                        :,
                    ]
                    keep_instruct_hidden_states = instruct_hidden_states.gather(
                        dim=1,
                        index=keep_index[..., None].expand(
                            instruct_hidden_states.shape[0],
                            -1,
                            instruct_hidden_states.shape[2],
                        ),
                    )
                    hidden_states = torch.cat(
                        [
                            left_hidden_states,
                            keep_instruct_hidden_states,
                            right_hidden_states,
                        ],
                        dim=1,
                    )

                    # for position
                    if position_ids is not None:
                        keep_index_for_instruct_position = position_ids[
                            :,
                            input_embeds_indices[0]["last_instruct"][
                                0
                            ] : input_embeds_indices[0]["last_instruct"][1]
                            - 1,
                        ].gather(dim=1, index=keep_index)
                        keep_index_for_left_position = position_ids[
                            :, : input_embeds_indices[0]["last_instruct"][0]
                        ]
                        keep_index_for_right_position = position_ids[
                            :, input_embeds_indices[0]["last_instruct"][1] - 1 :
                        ]
                        position_ids = torch.cat(
                            [
                                keep_index_for_left_position,
                                keep_index_for_instruct_position,
                                keep_index_for_right_position,
                            ],
                            dim=1,
                        )

                    else:
                        keep_index_for_instruct_position = (
                            keep_index + input_embeds_indices[0]["last_instruct"][0]
                        ).to(dtype=position_ids.dtype, device=position_ids.device)
                        keep_index_for_left_position = (
                            torch.arange(0, input_embeds_indices[0]["last_instruct"][0])
                            .repeat(B, 1)
                            .to(dtype=position_ids.dtype, device=position_ids.device)
                        )
                        keep_index_for_right_position = (
                            torch.arange(
                                input_embeds_indices[0]["last_instruct"][1] - 1, init_n
                            )
                            .repeat(B, 1)
                            .to(dtype=position_ids.dtype, device=position_ids.device)
                        )
                        position_ids = torch.cat(
                            [
                                keep_index_for_left_position,
                                keep_index_for_instruct_position,
                                keep_index_for_right_position,
                            ],
                            dim=1,
                        )

                    # for input_embeds_indices
                    num_unkeep_node = (
                        input_embeds_indices[0]["last_instruct"][1]
                        - 1
                        - input_embeds_indices[0]["last_instruct"][0]
                        - keep_index.shape[1]
                    )
                    for input_embeds_indice in input_embeds_indices:
                        input_embeds_indice["instruct"][1] -= num_unkeep_node
                        input_embeds_indice["last_instruct"][1] -= num_unkeep_node
                        input_embeds_indice["answer"][0] -= num_unkeep_node
                        input_embeds_indice["answer"][1] -= num_unkeep_node

                elif (
                    not self.training
                    and past_key_values_length
                    # and self.pre_answer is not None
                    and hidden_states.shape[1] == 1
                    and self.config.sparse_config["use_output_text_predictor"]
                ):  # output infer stage, with KV cache
                    # assert B == 1, "Using text predictor must keep the batch size to 1"
                    text_score_predictor_logit = self.output_text_score_predictor(
                        hidden_states
                    ).reshape(B, -1, 2)
                    text_decision = (
                        text_score_predictor_logit[:, :, 0]
                        > text_score_predictor_logit[:, :, 1]
                    )

                elif (
                    not self.training
                    # and past_key_values_length
                    and input_embeds_indices is not None
                    and hidden_states.shape[0] == len(input_embeds_indices)
                    # and hidden_states.shape[1] == 1
                    and self.config.sparse_config["use_output_text_predictor"]
                    and not use_cache
                ):  # output infer stage, without KV cache
                    if self.answer_indice is None:
                        self.answer_indice = input_embeds_indices[0]["instruct"][1]
                    # all output token length is same
                    output_text_hidden_states = hidden_states[
                        :,
                        self.answer_indice : -1,
                        :,
                    ]

                    text_score_predictor_logit = self.output_text_score_predictor(
                        output_text_hidden_states
                    ).reshape(B, -1, 2)
                    text_decision = (
                        text_score_predictor_logit[:, :, 0]
                        > text_score_predictor_logit[:, :, 1]
                    )
                    num_keep_node = text_decision.sum(dim=1).max()
                    text_score = text_score_predictor_logit[:, :, 0]

                    keep_index, _ = torch.sort(
                        torch.argsort(text_score, dim=1, descending=True)[
                            :, :num_keep_node
                        ],
                        dim=1,
                        descending=False,
                    )

                    left_hidden_states = hidden_states[
                        :,
                        : self.answer_indice,
                        :,
                    ]
                    right_hidden_states = hidden_states[
                        :,
                        -1:,
                        :,
                    ]

                    keep_output_text_hidden_states = output_text_hidden_states.gather(
                        dim=1,
                        index=keep_index[..., None].expand(
                            output_text_hidden_states.shape[0],
                            num_keep_node,
                            output_text_hidden_states.shape[2],
                        ),
                    )

                    hidden_states = torch.cat(
                        [
                            left_hidden_states,
                            keep_output_text_hidden_states,
                            right_hidden_states,
                        ],
                        dim=1,
                    )
                    policy = None

                    # for position
                    if position_ids is not None:
                        keep_index_for_output_text_position = position_ids[
                            :,
                            self.answer_indice : -1,
                        ].gather(dim=1, index=keep_index)
                        keep_index_for_left_position = position_ids[
                            :, : self.answer_indice
                        ]
                        keep_index_for_right_position = position_ids[:, -1:]
                        position_ids = torch.cat(
                            [
                                keep_index_for_left_position,
                                keep_index_for_output_text_position,
                                keep_index_for_right_position,
                            ],
                            dim=1,
                        )

                    else:
                        keep_index_for_output_text_position = (
                            keep_index + self.answer_indice
                        ).to(dtype=position_ids.dtype, device=position_ids.device)
                        keep_index_for_left_position = (
                            torch.arange(0, self.answer_indice)
                            .repeat(B, 1)
                            .to(dtype=position_ids.dtype, device=position_ids.device)
                        )
                        keep_index_for_right_position = (
                            torch.arange(init_n - 1, init_n)
                            .repeat(B, 1)
                            .to(dtype=position_ids.dtype, device=position_ids.device)
                        )
                        position_ids = torch.cat(
                            [
                                keep_index_for_left_position,
                                keep_index_for_output_text_position,
                                keep_index_for_right_position,
                            ],
                            dim=1,
                        )

                    # for input_embeds_indices
                    num_unkeep_node = (
                        init_n - 1 - self.answer_indice - keep_index.shape[1]
                    )
                    # for input_embeds_indice in input_embeds_indices:
                    #     input_embeds_indice["answer"][1] -= num_unkeep_node

                elif (
                    not self.training
                    and past_key_values_length
                    # and self.pre_answer is not None
                    and hidden_states.shape[1] > 1
                    and self.config.sparse_config["use_instruct_predictor"]
                ):  # for new instrct
                    # assert B == 1, "Using text predictor must keep the batch size to 1"
                    text_score_predictor_logit = self.instruct_score_predictor(
                        hidden_states
                    ).reshape(B, -1, 2)
                    text_decision = (
                        text_score_predictor_logit[:, :, 0]
                        > text_score_predictor_logit[:, :, 1]
                    )
                    text_decision[:, -1:] = True
                    # # new instruct template, "USER: **** ASSISTANT:"
                    # text_decision[:2] = True
                    # text_decision[-5:] = True

                # else:
                #     assert False, "text predictor does not work"

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    decoder_layer.__call__,
                    hidden_states,
                    attention_mask,
                    position_ids,
                    past_key_values,
                    output_attentions,
                    use_cache,
                    policy,
                    init_n,
                    self.config.sparse_config["sparse_layer"],
                    text_decision,
                )
            else:
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=attention_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_values,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                    policy=policy,
                    kv_seq_len_for_position=init_n,
                    sparse_layer=self.config.sparse_config["sparse_layer"],
                    text_decision=text_decision,
                )

            attention_mask = None
            # ----------------------------------------------------------#

            hidden_states = layer_outputs[0]

            if use_cache:
                next_decoder_cache = layer_outputs[2 if output_attentions else 1]

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        next_cache = None
        if use_cache:
            next_cache = (
                next_decoder_cache.to_legacy_cache()
                if use_legacy_cache
                else next_decoder_cache
            )
            # # ----------------------------------------------------------#
            # next_true_cache_length = next_decoder_cache.true_cache_length
            # # ----------------------------------------------------------#
        if not return_dict:
            return tuple(
                v
                for v in [hidden_states, next_cache, all_hidden_states, all_self_attns]
                if v is not None
            )
        return DynamicModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=next_cache,
            # true_cache_length=next_true_cache_length,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
            image_masks=image_masks,
            output_text_masks_batch_list=output_text_masks_batch_list,
            instruct_masks_batch_list=instruct_masks_batch_list,
            image_score_predictor_logits=image_score_predictor_logits,
        )


class DynamicLlamaForCausalLM(DynamicLlamaPreTrainedModel):
    _tied_weights_keys = ["lm_head.weight"]

    def __init__(self, config):
        super().__init__(config)
        self.model = DynamicLlamaModel(config)
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    def get_output_embeddings(self):
        return self.lm_head

    def set_output_embeddings(self, new_embeddings):
        self.lm_head = new_embeddings

    def set_decoder(self, decoder):
        self.model = decoder

    def get_decoder(self):
        return self.model

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    @replace_return_docstrings(
        output_type=CausalLMOutputWithPast, config_class=_CONFIG_FOR_DOC
    )
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        input_embeds_indices: Optional[List[dict]] = None,
        maskclip_mask: Optional[torch.IntTensor] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        r"""
        Args:
            labels (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
                Labels for computing the masked language modeling loss. Indices should either be in `[0, ...,
                config.vocab_size]` or -100 (see `input_ids` docstring). Tokens with indices set to `-100` are ignored
                (masked), the loss is only computed for the tokens with labels in `[0, ..., config.vocab_size]`.

        Returns:

        Example:

        ```python
        >>> from transformers import AutoTokenizer, LlamaForCausalLM

        >>> model = LlamaForCausalLM.from_pretrained("meta-llama/Llama-2-7b-hf")
        >>> tokenizer = AutoTokenizer.from_pretrained("meta-llama/Llama-2-7b-hf")

        >>> prompt = "Hey, are you conscious? Can you talk to me?"
        >>> inputs = tokenizer(prompt, return_tensors="pt")

        >>> # Generate
        >>> generate_ids = model.generate(inputs.input_ids, max_length=30)
        >>> tokenizer.batch_decode(generate_ids, skip_special_tokens=True, clean_up_tokenization_spaces=False)[0]
        "Hey, are you conscious? Can you talk to me?\nI'm not conscious, but I can talk to you."
        ```"""
        output_attentions = (
            output_attentions
            if output_attentions is not None
            else self.config.output_attentions
        )
        output_hidden_states = (
            output_hidden_states
            if output_hidden_states is not None
            else self.config.output_hidden_states
        )
        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
        outputs = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
            input_embeds_indices=input_embeds_indices,
        )

        hidden_states = outputs[0]
        if self.config.pretraining_tp > 1:
            lm_head_slices = self.lm_head.weight.split(
                self.vocab_size // self.config.pretraining_tp, dim=0
            )
            logits = [
                F.linear(hidden_states, lm_head_slices[i])
                for i in range(self.config.pretraining_tp)
            ]
            logits = torch.cat(logits, dim=-1)
        else:
            logits = self.lm_head(hidden_states)
        logits = logits.float()

        loss = None
        if labels is not None:
            # Shift so that tokens < n predict n
            shift_logits = logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            shift_logits = shift_logits.view(-1, self.config.vocab_size)
            shift_labels = shift_labels.view(-1)
            # Enable model parallelism
            shift_labels = shift_labels.to(shift_logits.device)
            loss = loss_fct(shift_logits, shift_labels)

            # ----------------------------------------------------------#
            if outputs.image_masks is not None and len(outputs.image_masks):
                image_mask_loss = 0.0
                for mask in outputs.image_masks:
                    batch_ratio = mask.mean(dim=1)
                    image_mask_loss = (
                        image_mask_loss
                        + (
                            (
                                self.config.sparse_config["vision_keep_rate"]
                                - batch_ratio
                            )
                            ** 2
                        ).mean()
                    )
                loss = (
                    loss
                    + self.config.sparse_config["mask_loss_weight"] * image_mask_loss
                )

            if outputs.output_text_masks_batch_list is not None and len(
                outputs.output_text_masks_batch_list
            ):
                output_text_mask_loss = 0.0
                for mask_batch_list in outputs.output_text_masks_batch_list:
                    # batch_ratio_batch_list = [
                    #     mask.mean()
                    #     for mask in mask_batch_list
                    #     if mask.shape[0]
                    #     >= self.config.sparse_config["output_text_len_for_training"]
                    # ]
                    # if len(batch_ratio_batch_list):
                    #     batch_ratio = torch.stack(batch_ratio_batch_list)
                    #     output_text_mask_loss = (
                    #         output_text_mask_loss
                    #         + (
                    #             (
                    #                 self.config.sparse_config["output_text_keep_rate"]
                    #                 - batch_ratio
                    #             )
                    #             ** 2
                    #         ).mean()
                    #     )
                    batch_ratio = torch.stack([mask.mean() for mask in mask_batch_list])
                    target_batch_ratio = torch.tensor(
                        [
                            (
                                self.config.sparse_config["output_text_keep_rate"]
                                if mask.shape[0]
                                >= self.config.sparse_config[
                                    "output_text_len_for_training"
                                ]
                                else mask.mean().item()
                            )
                            for mask in mask_batch_list
                        ]
                    ).to(dtype=batch_ratio.dtype, device=batch_ratio.device)
                    output_text_mask_loss = (
                        output_text_mask_loss
                        + ((target_batch_ratio - batch_ratio) ** 2).mean()
                    )
                loss = (
                    loss
                    + self.config.sparse_config["mask_loss_weight"]
                    * output_text_mask_loss
                )

            if outputs.instruct_masks_batch_list is not None and len(
                outputs.instruct_masks_batch_list
            ):
                instruct_mask_loss = 0.0
                for mask_batch_list in outputs.instruct_masks_batch_list:
                    # batch_ratio_batch_list = [
                    #     mask.mean()
                    #     for mask in mask_batch_list
                    #     if mask.shape[0]
                    #     >= self.config.sparse_config["output_text_len_for_training"]
                    # ]
                    # if len(batch_ratio_batch_list):
                    #     batch_ratio = torch.stack(batch_ratio_batch_list)
                    #     output_text_mask_loss = (
                    #         output_text_mask_loss
                    #         + (
                    #             (
                    #                 self.config.sparse_config["output_text_keep_rate"]
                    #                 - batch_ratio
                    #             )
                    #             ** 2
                    #         ).mean()
                    #     )
                    batch_ratio = torch.stack([mask.mean() for mask in mask_batch_list])
                    target_batch_ratio = torch.tensor(
                        [
                            (
                                self.config.sparse_config["instruct_keep_rate"]
                                if mask.shape[0]
                                >= self.config.sparse_config[
                                    "instruct_len_for_training"
                                ]
                                else mask.mean().item()
                            )
                            for mask in mask_batch_list
                        ]
                    ).to(dtype=batch_ratio.dtype, device=batch_ratio.device)
                    instruct_mask_loss = (
                        instruct_mask_loss
                        + ((target_batch_ratio - batch_ratio) ** 2).mean()
                    )
                loss = (
                    loss
                    + self.config.sparse_config["mask_loss_weight"] * instruct_mask_loss
                )
            # ----------------------------------------------------------#

            # # ----------------------------------------------------------#
            # if maskclip_mask is not None and len(maskclip_mask):
            #     maskclip_distill_loss = 0.0
            #     for distill_mask in outputs.distill_masks:
            #         maskclip_distill_loss = maskclip_distill_loss + F.smooth_l1_loss(
            #             distill_mask, maskclip_mask
            #         )
            #     loss = (
            #         loss
            #         + self.config.sparse_config["maskclip_distill_loss_weight"]
            #         * maskclip_distill_loss
            #     )
            # # ----------------------------------------------------------#
            # ----------------------------------------------------------#
            if maskclip_mask is not None and len(maskclip_mask):
                maskclip_distill_loss = 0.0
                for mask in outputs.masks:
                    # # l1loss
                    # maskclip_distill_loss = maskclip_distill_loss + F.l1_loss(
                    #     mask, maskclip_mask
                    # )

                    # # ce loss
                    # maskclip_distill_loss = maskclip_distill_loss + F.cross_entropy(
                    #     image_score_predictor_logit.permute(0, 2, 1),
                    #     (1 - maskclip_mask).long(),
                    # )

                    # # lovasz loss
                    # from .lovasz_loss import lovasz_softmax_1d

                    # maskclip_distill_loss = maskclip_distill_loss + lovasz_softmax_1d(
                    #     image_score_predictor_logit.permute(0, 2, 1),
                    #     (1 - maskclip_mask).long(),
                    #     per_image=True,
                    # )

                    # bce loss
                    maskclip_distill_loss = (
                        maskclip_distill_loss
                        + F.binary_cross_entropy_with_logits(
                            (mask - 0.5) * 10,
                            maskclip_mask.float(),
                        )
                    )
                loss = (
                    loss
                    + self.config.sparse_config["maskclip_distill_loss_weight"]
                    * maskclip_distill_loss
                )
            # ----------------------------------------------------------#

        if not return_dict:
            output = (logits,) + outputs[1:]
            return (loss,) + output if loss is not None else output

        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=outputs.past_key_values,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )

    def prepare_inputs_for_generation(
        self,
        input_ids,
        past_key_values=None,
        attention_mask=None,
        inputs_embeds=None,
        input_embeds_indices=None,
        **kwargs,
    ):
        if past_key_values is not None:
            if isinstance(past_key_values, Cache):
                cache_length = past_key_values.get_seq_length()
                past_length = past_key_values.seen_tokens
                max_cache_length = past_key_values.get_max_length()
            else:
                # ----------------------------------------------------------#
                cache_length = past_length = past_key_values[0][0][0].shape[2]
                # ----------------------------------------------------------#
                max_cache_length = None

            # Keep only the unprocessed tokens:
            # 1 - If the length of the attention_mask exceeds the length of input_ids, then we are in a setting where
            # some of the inputs are exclusively passed as part of the cache (e.g. when passing input_embeds as
            # input)
            if (
                attention_mask is not None
                and attention_mask.shape[1] > input_ids.shape[1]
            ):
                input_ids = input_ids[:, -(attention_mask.shape[1] - past_length) :]
            # 2 - If the past_length is smaller than input_ids', then input_ids holds all input tokens. We can discard
            # input_ids based on the past_length.
            elif past_length < input_ids.shape[1]:
                input_ids = input_ids[:, past_length:]
            # 3 - Otherwise (past_length >= input_ids.shape[1]), let's assume input_ids only has unprocessed tokens.

            # If we are about to go beyond the maximum cache length, we need to crop the input attention mask.
            if (
                max_cache_length is not None
                and attention_mask is not None
                and cache_length + input_ids.shape[1] > max_cache_length
            ):
                attention_mask = attention_mask[:, -max_cache_length:]

        position_ids = kwargs.get("position_ids", None)
        if attention_mask is not None and position_ids is None:
            # create position_ids on the fly for batch generation
            position_ids = attention_mask.long().cumsum(-1) - 1
            position_ids.masked_fill_(attention_mask == 0, 1)
            if past_key_values:
                position_ids = position_ids[:, -input_ids.shape[1] :]

        # if `inputs_embeds` are passed, we only want to use them in the 1st generation step
        if inputs_embeds is not None and past_key_values is None:
            model_inputs = {"inputs_embeds": inputs_embeds}
        else:
            model_inputs = {"input_ids": input_ids}

        model_inputs.update(
            {
                "position_ids": position_ids,
                "past_key_values": past_key_values,
                "use_cache": kwargs.get("use_cache"),
                "attention_mask": attention_mask,
                "input_embeds_indices": input_embeds_indices,
            }
        )
        return model_inputs

    @staticmethod
    def _reorder_cache(past_key_values, beam_idx):
        reordered_past = ()
        # ----------------------------------------------------------#
        for layer_past in past_key_values[0]:
            reordered_past += (
                tuple(
                    past_state.index_select(0, beam_idx.to(past_state.device))
                    for past_state in layer_past
                ),
            )
        # ----------------------------------------------------------#
        return reordered_past


# @add_start_docstrings(
#     """
#     The LLaMa Model transformer with a sequence classification head on top (linear layer).

#     [`LlamaForSequenceClassification`] uses the last token in order to do the classification, as other causal models
#     (e.g. GPT-2) do.

#     Since it does classification on the last token, it requires to know the position of the last token. If a
#     `pad_token_id` is defined in the configuration, it finds the last token that is not a padding token in each row. If
#     no `pad_token_id` is defined, it simply takes the last value in each row of the batch. Since it cannot guess the
#     padding tokens when `inputs_embeds` are passed instead of `input_ids`, it does the same (take the last value in
#     each row of the batch).
#     """,
#     LLAMA_START_DOCSTRING,
# )
# class LlamaForSequenceClassification(LlamaPreTrainedModel):
#     def __init__(self, config):
#         super().__init__(config)
#         self.num_labels = config.num_labels
#         self.model = LlamaModel(config)
#         self.score = nn.Linear(config.hidden_size, self.num_labels, bias=False)

#         # Initialize weights and apply final processing
#         self.post_init()

#     def get_input_embeddings(self):
#         return self.model.embed_tokens

#     def set_input_embeddings(self, value):
#         self.model.embed_tokens = value

#     @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
#     def forward(
#         self,
#         input_ids: torch.LongTensor = None,
#         attention_mask: Optional[torch.Tensor] = None,
#         position_ids: Optional[torch.LongTensor] = None,
#         past_key_values: Optional[List[torch.FloatTensor]] = None,
#         inputs_embeds: Optional[torch.FloatTensor] = None,
#         labels: Optional[torch.LongTensor] = None,
#         use_cache: Optional[bool] = None,
#         output_attentions: Optional[bool] = None,
#         output_hidden_states: Optional[bool] = None,
#         return_dict: Optional[bool] = None,
#     ) -> Union[Tuple, SequenceClassifierOutputWithPast]:
#         r"""
#         labels (`torch.LongTensor` of shape `(batch_size,)`, *optional*):
#             Labels for computing the sequence classification/regression loss. Indices should be in `[0, ...,
#             config.num_labels - 1]`. If `config.num_labels == 1` a regression loss is computed (Mean-Square loss), If
#             `config.num_labels > 1` a classification loss is computed (Cross-Entropy).
#         """
#         return_dict = (
#             return_dict if return_dict is not None else self.config.use_return_dict
#         )

#         transformer_outputs = self.model(
#             input_ids,
#             attention_mask=attention_mask,
#             position_ids=position_ids,
#             past_key_values=past_key_values,
#             inputs_embeds=inputs_embeds,
#             use_cache=use_cache,
#             output_attentions=output_attentions,
#             output_hidden_states=output_hidden_states,
#             return_dict=return_dict,
#         )
#         hidden_states = transformer_outputs[0]
#         logits = self.score(hidden_states)

#         if input_ids is not None:
#             batch_size = input_ids.shape[0]
#         else:
#             batch_size = inputs_embeds.shape[0]

#         if self.config.pad_token_id is None and batch_size != 1:
#             raise ValueError(
#                 "Cannot handle batch sizes > 1 if no padding token is defined."
#             )
#         if self.config.pad_token_id is None:
#             sequence_lengths = -1
#         else:
#             if input_ids is not None:
#                 # if no pad token found, use modulo instead of reverse indexing for ONNX compatibility
#                 sequence_lengths = (
#                     torch.eq(input_ids, self.config.pad_token_id).int().argmax(-1) - 1
#                 )
#                 sequence_lengths = sequence_lengths % input_ids.shape[-1]
#                 sequence_lengths = sequence_lengths.to(logits.device)
#             else:
#                 sequence_lengths = -1

#         pooled_logits = logits[
#             torch.arange(batch_size, device=logits.device), sequence_lengths
#         ]

#         loss = None
#         if labels is not None:
#             labels = labels.to(logits.device)
#             if self.config.problem_type is None:
#                 if self.num_labels == 1:
#                     self.config.problem_type = "regression"
#                 elif self.num_labels > 1 and (
#                     labels.dtype == torch.long or labels.dtype == torch.int
#                 ):
#                     self.config.problem_type = "single_label_classification"
#                 else:
#                     self.config.problem_type = "multi_label_classification"

#             if self.config.problem_type == "regression":
#                 loss_fct = MSELoss()
#                 if self.num_labels == 1:
#                     loss = loss_fct(pooled_logits.squeeze(), labels.squeeze())
#                 else:
#                     loss = loss_fct(pooled_logits, labels)
#             elif self.config.problem_type == "single_label_classification":
#                 loss_fct = CrossEntropyLoss()
#                 loss = loss_fct(
#                     pooled_logits.view(-1, self.num_labels), labels.view(-1)
#                 )
#             elif self.config.problem_type == "multi_label_classification":
#                 loss_fct = BCEWithLogitsLoss()
#                 loss = loss_fct(pooled_logits, labels)
#         if not return_dict:
#             output = (pooled_logits,) + transformer_outputs[1:]
#             return ((loss,) + output) if loss is not None else output

#         return SequenceClassifierOutputWithPast(
#             loss=loss,
#             logits=pooled_logits,
#             past_key_values=transformer_outputs.past_key_values,
#             hidden_states=transformer_outputs.hidden_states,
#             attentions=transformer_outputs.attentions,
#         )
