"""Class names for datasets."""

from typing import List


def voc_classes():
    """Pascal-VOC classes."""
    return [
        "airplane",
        "bicycle",
        "bird",
        "boat",
        "bottle",
        "bus",
        "car",
        "cat",
        "chair",
        "cow",
        "dining table",
        "dog",
        "horse",
        "motorbike",
        "person",
        "potted plant",
        "sheep",
        "sofa",
        "train",
        "tv monitor",
    ]


def context_classes():
    """Pascal-Context classes."""
    return [
        "background",
        "airplane",
        "bag",
        "bed",
        "bedclothes",
        "bench",
        "bicycle",
        "bird",
        "boat",
        "book",
        "bottle",
        "building",
        "bus",
        "cabinet",
        "car",
        "cat",
        "ceiling",
        "chair",
        "cloth",
        "computer",
        "cow",
        "cup",
        "curtain",
        "dog",
        "door",
        "fence",
        "floor",
        "flower",
        "food",
        "grass",
        "ground",
        "horse",
        "keyboard",
        "light",
        "motorbike",
        "mountain",
        "mouse",
        "person",
        "plate",
        "platform",
        "potted plant",
        "road",
        "rock",
        "sheep",
        "shelves",
        "sidewalk",
        "sign",
        "sky",
        "snow",
        "sofa",
        "table",
        "track",
        "train",
        "tree",
        "truck",
        "tv monitor",
        "wall",
        "water",
        "window",
        "wood",
    ]


def stuff_classes():
    """COCO-Stuff classes."""
    return [
        "person",
        "bicycle",
        "car",
        "motorcycle",
        "airplane",
        "bus",
        "train",
        "truck",
        "boat",
        "traffic light",
        "fire hydrant",
        "stop sign",
        "parking meter",
        "bench",
        "bird",
        "cat",
        "dog",
        "horse",
        "sheep",
        "cow",
        "elephant",
        "bear",
        "zebra",
        "giraffe",
        "backpack",
        "umbrella",
        "handbag",
        "tie",
        "suitcase",
        "frisbee",
        "skis",
        "snowboard",
        "sports ball",
        "kite",
        "baseball bat",
        "baseball glove",
        "skateboard",
        "surfboard",
        "tennis racket",
        "bottle",
        "wine glass",
        "cup",
        "fork",
        "knife",
        "spoon",
        "bowl",
        "banana",
        "apple",
        "sandwich",
        "orange",
        "broccoli",
        "carrot",
        "hot dog",
        "pizza",
        "donut",
        "cake",
        "chair",
        "couch",
        "potted plant",
        "bed",
        "dining table",
        "toilet",
        "tv",
        "laptop",
        "mouse",
        "remote",
        "keyboard",
        "cell phone",
        "microwave",
        "oven",
        "toaster",
        "sink",
        "refrigerator",
        "book",
        "clock",
        "vase",
        "scissors",
        "teddy bear",
        "hair drier",
        "toothbrush",
        "banner",
        "blanket",
        "branch",
        "bridge",
        "building",
        "bush",
        "cabinet",
        "cage",
        "cardboard",
        "carpet",
        "ceiling",
        "tile ceiling",
        "cloth",
        "clothes",
        "clouds",
        "counter",
        "cupboard",
        "curtain",
        "desk",
        "dirt",
        "door",
        "fence",
        "marble floor",
        "floor",
        "stone floor",
        "tile floor",
        "wood floor",
        "flower",
        "fog",
        "food",
        "fruit",
        "furniture",
        "grass",
        "gravel",
        "ground",
        "hill",
        "house",
        "leaves",
        "light",
        "mat",
        "metal",
        "mirror",
        "moss",
        "mountain",
        "mud",
        "napkin",
        "net",
        "paper",
        "pavement",
        "pillow",
        "plant",
        "plastic",
        "platform",
        "playingfield",
        "railing",
        "railroad",
        "river",
        "road",
        "rock",
        "roof",
        "rug",
        "salad",
        "sand",
        "sea",
        "shelf",
        "sky",
        "skyscraper",
        "snow",
        "solid",
        "stairs",
        "stone",
        "straw",
        "structural",
        "table",
        "tent",
        "textile",
        "towel",
        "tree",
        "vegetable",
        "brick wall",
        "concrete wall",
        "wall",
        "panel wall",
        "stone wall",
        "tile wall",
        "wood wall",
        "water",
        "waterdrops",
        "blind window",
        "window",
        "wood",
    ]


def all_pascal_context_classes():
    return [
        "accordion",
        "airplane",
        "air conditioner",
        "antenna",
        "artillery",
        "ashtray",
        "atrium",
        "baby carriage",
        "bag",
        "ball",
        "balloon",
        "bamboo weaving",
        "barrel",
        "baseball bat",
        "basket",
        "basketball backboard",
        "bathtub",
        "bed",
        "bedclothes",
        "beer",
        "bell",
        "bench",
        "bicycle",
        "binoculars",
        "bird",
        "bird cage",
        "bird feeder",
        "bird nest",
        "blackboard",
        "board",
        "boat",
        "bone",
        "book",
        "bottle",
        "bottle opener",
        "bowl",
        "box",
        "bracelet",
        "brick",
        "bridge",
        "broom",
        "brush",
        "bucket",
        "building",
        "bus",
        "cabinet",
        "cabinet door",
        "cage",
        "cake",
        "calculator",
        "calendar",
        "camel",
        "camera",
        "camera lens",
        "can",
        "candle",
        "candle holder",
        "cap",
        "car",
        "card",
        "cart",
        "case",
        "casette recorder",
        "cash register",
        "cat",
        "cd",
        "cd player",
        "ceiling",
        "cell phone",
        "cello",
        "chain",
        "chair",
        "chessboard",
        "chicken",
        "chopstick",
        "clip",
        "clippers",
        "clock",
        "closet",
        "cloth",
        "clothes tree",
        "coffee",
        "coffee machine",
        "comb",
        "computer",
        "concrete",
        "cone",
        "container",
        "control booth",
        "controller",
        "cooker",
        "copying machine",
        "coral",
        "cork",
        "corkscrew",
        "counter",
        "court",
        "cow",
        "crabstick",
        "crane",
        "crate",
        "cross",
        "crutch",
        "cup",
        "curtain",
        "cushion",
        "cutting board",
        "dais",
        "disc",
        "disc case",
        "dishwasher",
        "dock",
        "dog",
        "dolphin",
        "door",
        "drainer",
        "dray",
        "drink dispenser",
        "drinking machine",
        "drop",
        "drug",
        "drum",
        "drum kit",
        "duck",
        "dumbbell",
        "earphone",
        "earrings",
        "egg",
        "electric fan",
        "electric iron",
        "electric pot",
        "electric saw",
        "electronic keyboard",
        "engine",
        "envelope",
        "equipment",
        "escalator",
        "exhibition booth",
        "extinguisher",
        "eyeglass",
        "fan",
        "faucet",
        "fax machine",
        "fence",
        "ferris wheel",
        "fire extinguisher",
        "fire hydrant",
        "fire place",
        "fish",
        "fish tank",
        "fishbowl",
        "fishing net",
        "fishing pole",
        "flag",
        "flagstaff",
        "flame",
        "flashlight",
        "floor",
        "flower",
        "fly",
        "foam",
        "food",
        "footbridge",
        "forceps",
        "fork",
        "forklift",
        "fountain",
        "fox",
        "frame",
        "fridge",
        "frog",
        "fruit",
        "funnel",
        "furnace",
        "game controller",
        "game machine",
        "gas cylinder",
        "gas hood",
        "gas stove",
        "gift box",
        "glass",
        "glass marble",
        "globe",
        "glove",
        "goal",
        "grandstand",
        "grass",
        "gravestone",
        "ground",
        "guardrail",
        "guitar",
        "gun",
        "hammer",
        "hand cart",
        "handle",
        "handrail",
        "hanger",
        "hard disk drive",
        "hat",
        "hay",
        "headphone",
        "heater",
        "helicopter",
        "helmet",
        "holder",
        "hook",
        "horse",
        "horse-drawn carriage",
        "hot-air balloon",
        "hydrovalve",
        "ice",
        "inflator pump",
        "ipod",
        "iron",
        "ironing board",
        "jar",
        "kart",
        "kettle",
        "key",
        "keyboard",
        "kitchen range",
        "kite",
        "knife",
        "knife block",
        "ladder",
        "ladder truck",
        "ladle",
        "laptop",
        "leaves",
        "lid",
        "life buoy",
        "light",
        "light bulb",
        "lighter",
        "line",
        "lion",
        "lobster",
        "lock",
        "machine",
        "mailbox",
        "mannequin",
        "map",
        "mask",
        "mat",
        "match book",
        "mattress",
        "menu",
        "metal",
        "meter box",
        "microphone",
        "microwave",
        "mirror",
        "missile",
        "model",
        "money",
        "monkey",
        "mop",
        "motorbike",
        "mountain",
        "mouse",
        "mouse pad",
        "musical instrument",
        "napkin",
        "net",
        "newspaper",
        "oar",
        "ornament",
        "outlet",
        "oven",
        "oxygen bottle",
        "pack",
        "pan",
        "paper",
        "paper box",
        "paper cutter",
        "parachute",
        "parasol",
        "parterre",
        "patio",
        "pelage",
        "pen",
        "pen container",
        "pencil",
        "person",
        "photo",
        "piano",
        "picture",
        "pig",
        "pillar",
        "pillow",
        "pipe",
        "pitcher",
        "plant",
        "plastic",
        "plate",
        "platform",
        "player",
        "playground",
        "pliers",
        "plume",
        "poker",
        "poker chip",
        "pole",
        "pool table",
        "postcard",
        "poster",
        "pot",
        "potted plant",
        "printer",
        "projector",
        "pumpkin",
        "rabbit",
        "racket",
        "radiator",
        "radio",
        "rail",
        "rake",
        "ramp",
        "range hood",
        "receiver",
        "recorder",
        "recreational machines",
        "remote control",
        "road",
        "robot",
        "rock",
        "rocket",
        "rocking horse",
        "rope",
        "rug",
        "ruler",
        "runway",
        "saddle",
        "sand",
        "saw",
        "scale",
        "scanner",
        "scissors",
        "scoop",
        "screen",
        "screwdriver",
        "sculpture",
        "scythe",
        "sewer",
        "sewing machine",
        "shed",
        "sheep",
        "shell",
        "shelves",
        "shoe",
        "shopping cart",
        "shovel",
        "sidecar",
        "sidewalk",
        "sign",
        "signal light",
        "sink",
        "skateboard",
        "ski",
        "sky",
        "sled",
        "slippers",
        "smoke",
        "snail",
        "snake",
        "snow",
        "snowmobiles",
        "sofa",
        "spanner",
        "spatula",
        "speaker",
        "speed bump",
        "spice container",
        "spoon",
        "sprayer",
        "squirrel",
        "stage",
        "stair",
        "stapler",
        "stick",
        "sticky note",
        "stone",
        "stool",
        "stove",
        "straw",
        "stretcher",
        "sun",
        "sunglass",
        "sunshade",
        "surveillance camera",
        "swan",
        "sweeper",
        "swim ring",
        "swimming pool",
        "swing",
        "switch",
        "table",
        "tableware",
        "tank",
        "tap",
        "tape",
        "tarp",
        "telephone",
        "telephone booth",
        "tent",
        "tire",
        "toaster",
        "toilet",
        "tong",
        "tool",
        "toothbrush",
        "towel",
        "toy",
        "toy car",
        "track",
        "train",
        "trampoline",
        "trash bin",
        "tray",
        "tree",
        "tricycle",
        "tripod",
        "trophy",
        "truck",
        "tube",
        "turtle",
        "tv monitor",
        "tweezers",
        "typewriter",
        "umbrella",
        "unknown",
        "vacuum cleaner",
        "vending machine",
        "video camera",
        "video game console",
        "video player",
        "video tape",
        "violin",
        "wakeboard",
        "wall",
        "wallet",
        "wardrobe",
        "washing machine",
        "watch",
        "water",
        "water dispenser",
        "water pipe",
        "water skate board",
        "watermelon",
        "whale",
        "wharf",
        "wheel",
        "wheelchair",
        "window",
        "window blinds",
        "wineglass",
        "wire",
        "wood",
        "wool",
    ]


def bg_classes():
    return ["building", "ground", "grass", "tree", "sky", "background"]


_DATASETS = {
    "voc": ["voc", "pascal_voc", "pascal-voc"],
    "context": ["context", "pascal_context", "pascal-context"],
    "stuff": ["stuff", "coco_stuff", "coco-stuff"],
}


def available_datasets() -> List[str]:
    """Returns the names of available datasets"""
    return list(_DATASETS.keys())


def get_classes(dataset: str):
    """Get class names of a dataset."""
    alias2name = {}
    for name, aliases in _DATASETS.items():
        for alias in aliases:
            alias2name[alias] = name

    if isinstance(dataset, str):
        if dataset in alias2name:
            labels = eval(alias2name[dataset] + "_classes()")
        else:
            raise ValueError(f"unrecognized dataset: {dataset}")
    else:
        raise TypeError(f"dataset must a str, but got {type(dataset)}")
    return labels
