"""Palette for datasets."""

from typing import List


def voc_palette():
    """Pascal-VOC palette."""
    return [
        [0, 0, 0],
        [128, 0, 0],
        [0, 128, 0],
        [128, 128, 0],
        [0, 0, 128],
        [128, 0, 128],
        [0, 128, 128],
        [128, 128, 128],
        [64, 0, 0],
        [192, 0, 0],
        [64, 128, 0],
        [192, 128, 0],
        [64, 0, 128],
        [192, 0, 128],
        [64, 128, 128],
        [192, 128, 128],
        [0, 64, 0],
        [128, 64, 0],
        [0, 192, 0],
        [128, 192, 0],
        [0, 64, 128],
    ]


def context_palette():
    """Pascal-Context palette."""
    return [
        [120, 120, 120],
        [180, 120, 120],
        [6, 230, 230],
        [80, 50, 50],
        [4, 200, 3],
        [120, 120, 80],
        [140, 140, 140],
        [204, 5, 255],
        [230, 230, 230],
        [4, 250, 7],
        [224, 5, 255],
        [235, 255, 7],
        [150, 5, 61],
        [120, 120, 70],
        [8, 255, 51],
        [255, 6, 82],
        [143, 255, 140],
        [204, 255, 4],
        [255, 51, 7],
        [204, 70, 3],
        [0, 102, 200],
        [61, 230, 250],
        [255, 6, 51],
        [11, 102, 255],
        [255, 7, 71],
        [255, 9, 224],
        [9, 7, 230],
        [220, 220, 220],
        [255, 9, 92],
        [112, 9, 255],
        [8, 255, 214],
        [7, 255, 224],
        [255, 184, 6],
        [10, 255, 71],
        [255, 41, 10],
        [7, 255, 255],
        [224, 255, 8],
        [102, 8, 255],
        [255, 61, 6],
        [255, 194, 7],
        [255, 122, 8],
        [0, 255, 20],
        [255, 8, 41],
        [255, 5, 153],
        [6, 51, 255],
        [235, 12, 255],
        [160, 150, 20],
        [0, 163, 255],
        [140, 140, 140],
        [250, 10, 15],
        [20, 255, 0],
        [31, 255, 0],
        [255, 31, 0],
        [255, 224, 0],
        [153, 255, 0],
        [0, 0, 255],
        [255, 71, 0],
        [0, 235, 255],
        [0, 173, 255],
        [31, 0, 255],
    ]


def stuff_palette():
    """COCO-Stuff palette."""
    return [
        [0, 192, 64],
        [0, 192, 64],
        [0, 64, 96],
        [128, 192, 192],
        [0, 64, 64],
        [0, 192, 224],
        [0, 192, 192],
        [128, 192, 64],
        [0, 192, 96],
        [128, 192, 64],
        [128, 32, 192],
        [0, 0, 224],
        [0, 0, 64],
        [0, 160, 192],
        [128, 0, 96],
        [128, 0, 192],
        [0, 32, 192],
        [128, 128, 224],
        [0, 0, 192],
        [128, 160, 192],
        [128, 128, 0],
        [128, 0, 32],
        [128, 32, 0],
        [128, 0, 128],
        [64, 128, 32],
        [0, 160, 0],
        [0, 0, 0],
        [192, 128, 160],
        [0, 32, 0],
        [0, 128, 128],
        [64, 128, 160],
        [128, 160, 0],
        [0, 128, 0],
        [192, 128, 32],
        [128, 96, 128],
        [0, 0, 128],
        [64, 0, 32],
        [0, 224, 128],
        [128, 0, 0],
        [192, 0, 160],
        [0, 96, 128],
        [128, 128, 128],
        [64, 0, 160],
        [128, 224, 128],
        [128, 128, 64],
        [192, 0, 32],
        [128, 96, 0],
        [128, 0, 192],
        [0, 128, 32],
        [64, 224, 0],
        [0, 0, 64],
        [128, 128, 160],
        [64, 96, 0],
        [0, 128, 192],
        [0, 128, 160],
        [192, 224, 0],
        [0, 128, 64],
        [128, 128, 32],
        [192, 32, 128],
        [0, 64, 192],
        [0, 0, 32],
        [64, 160, 128],
        [128, 64, 64],
        [128, 0, 160],
        [64, 32, 128],
        [128, 192, 192],
        [0, 0, 160],
        [192, 160, 128],
        [128, 192, 0],
        [128, 0, 96],
        [192, 32, 0],
        [128, 64, 128],
        [64, 128, 96],
        [64, 160, 0],
        [0, 64, 0],
        [192, 128, 224],
        [64, 32, 0],
        [0, 192, 128],
        [64, 128, 224],
        [192, 160, 0],
        [0, 192, 0],
        [192, 128, 96],
        [192, 96, 128],
        [0, 64, 128],
        [64, 0, 96],
        [64, 224, 128],
        [128, 64, 0],
        [192, 0, 224],
        [64, 96, 128],
        [128, 192, 128],
        [64, 0, 224],
        [192, 224, 128],
        [128, 192, 64],
        [192, 0, 96],
        [192, 96, 0],
        [128, 64, 192],
        [0, 128, 96],
        [0, 224, 0],
        [64, 64, 64],
        [128, 128, 224],
        [0, 96, 0],
        [64, 192, 192],
        [0, 128, 224],
        [128, 224, 0],
        [64, 192, 64],
        [128, 128, 96],
        [128, 32, 128],
        [64, 0, 192],
        [0, 64, 96],
        [0, 160, 128],
        [192, 0, 64],
        [128, 64, 224],
        [0, 32, 128],
        [192, 128, 192],
        [0, 64, 224],
        [128, 160, 128],
        [192, 128, 0],
        [128, 64, 32],
        [128, 32, 64],
        [192, 0, 128],
        [64, 192, 32],
        [0, 160, 64],
        [64, 0, 0],
        [192, 192, 160],
        [0, 32, 64],
        [64, 128, 128],
        [64, 192, 160],
        [128, 160, 64],
        [64, 128, 0],
        [192, 192, 32],
        [128, 96, 192],
        [64, 0, 128],
        [64, 64, 32],
        [0, 224, 192],
        [192, 0, 0],
        [192, 64, 160],
        [0, 96, 192],
        [192, 128, 128],
        [64, 64, 160],
        [128, 224, 192],
        [192, 128, 64],
        [192, 64, 32],
        [128, 96, 64],
        [192, 0, 192],
        [0, 192, 32],
        [64, 224, 64],
        [64, 0, 64],
        [128, 192, 160],
        [64, 96, 64],
        [64, 128, 192],
        [0, 192, 160],
        [192, 224, 64],
        [64, 128, 64],
        [128, 192, 32],
        [192, 32, 192],
        [64, 64, 192],
        [0, 64, 32],
        [64, 160, 192],
        [192, 64, 64],
        [128, 64, 160],
        [64, 32, 192],
        [192, 192, 192],
        [0, 64, 160],
        [192, 160, 192],
        [192, 192, 0],
        [128, 64, 96],
        [192, 32, 64],
        [192, 64, 128],
        [64, 192, 96],
        [64, 160, 64],
        [64, 64, 0],
    ]


_PALETTE = {
    "voc": ["voc", "pascal_voc", "pascal-voc"],
    "context": ["context", "pascal_context", "pascal-context"],
    "stuff": ["stuff", "coco_stuff", "coco-stuff"],
}


def available_palette() -> List[str]:
    """Returns the names of available palette"""
    return list(_PALETTE.keys())


def get_palette(dataset: str) -> List[List[int]]:
    """Get class palette (RGB) of a dataset."""
    alias2name = {}
    for name, aliases in _PALETTE.items():
        for alias in aliases:
            alias2name[alias] = name

    if isinstance(dataset, str):
        if dataset in alias2name:
            colors = eval(alias2name[dataset] + "_palette()")
        else:
            raise ValueError(f"unrecognized dataset: {dataset}")
    else:
        raise TypeError(f"dataset must a str, but got {type(dataset)}")
    return colors
